/*
 * @Copyright 2010, MechSoft 
 * MechSoft, Mechanical and Software Solutions 
 * 
 * Licensed under the Apache License, Version 2.0 (the 
 * "License"); you may not use this file except in compliance 
 * with the License. You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.defne.server.json;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.defne.utility.log.ILogger;
import org.defne.utility.log.LoggerProvider;
import org.defne.utility.reflection.ReflectionUtils;

import com.google.gson.Gson;

/**
 * Defne JSON servlet.
 * @author gurkanerdogdu
 * @version $Rev$ $Date$
 */
public class DefneJsonServlet extends HttpServlet
{
    private static final long serialVersionUID = 1L;
    
    private static final ILogger logger = LoggerProvider.getLogProvider(DefneJsonServlet.class);

    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
    {
        resp.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED, "Get is not supported");
    }

    @Override
    protected void doPost(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException
    {
        //Getting json parameter
        BufferedReader reader = req.getReader();
        StringBuffer buf = new StringBuffer();
        String s = null;
        while((s = reader.readLine()) != null)
        {
            buf.append(s);
        }
        
        String jsonString = buf.toString();
        if(jsonString.equals(""))
        {
            resp.sendError(HttpServletResponse.SC_BAD_REQUEST, "JSON payload is not found in the request");
            return;
        }
                
        
        Gson gson = new Gson();
        try
        {
            Object result = null;
            Map<String, String> arguments = new HashMap<String, String>();
            JsonServiceMessage jsonObj = gson.fromJson(jsonString, JsonServiceMessage.class);   
            String serviceName = jsonObj.getService();
            
            StringBuffer error = new StringBuffer();
            if(serviceName == null || serviceName.equals(""))
            {
                error.append("No service name");
            }
            
            String methodNameName = jsonObj.getMethod();
            
            if(methodNameName == null || methodNameName.equals(""))
            {
                error.append("No method name");
            }
            
            String adaptorClass = jsonObj.getAdaptorClass();
            
            if(adaptorClass == null || adaptorClass.equals(""))
            {
                error.append("No adaptror class name");
            }
            
            if(error.length() > 0)
            {
                arguments.put("ERROR", error.toString());
            }
            else
            {
                //Loading adaptor class
                Class<?> clazz = ReflectionUtils.getClassFromName(adaptorClass);                
                jsonObj = (JsonServiceMessage)gson.fromJson(jsonString, clazz);
                                
                //Execute service
                result = JsonServiceAdaptor.execute(jsonObj);                
            }            
            
            //Send response
            resp.setHeader("Cache-Control", "no-cache");
            resp.setHeader("max-age", "0");
            resp.setHeader("must-revalidate", "true");
            resp.setHeader("Content-Type", "application/json;charset=utf-8");
            
            if(error.length() > 0)
            {
                resp.getWriter().write(gson.toJson(arguments));   
            }  
            else
            {
                resp.getWriter().write(gson.toJson(result));   
            }
        }
        catch(Exception e)
        {
            String error = "Error is occured while parsing JSON";
            logger.error(error,e);
            resp.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, error + ". See log for the details");
        }
    }

    
}
