/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCParameters;
import eu.europa.esig.dss.enumerations.ASiCContainerType;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.MimeType;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ASiCUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ASiCUtils.class);
    public static final String MANIFEST_FILENAME = "Manifest";
    public static final String ASIC_MANIFEST_FILENAME = "ASiCManifest";
    public static final String ASIC_ARCHIVE_MANIFEST_FILENAME = "ASiCArchiveManifest";
    public static final String MIME_TYPE = "mimetype";
    public static final String MIME_TYPE_COMMENT = "mimetype=";
    public static final String META_INF_FOLDER = "META-INF/";
    public static final String PACKAGE_ZIP = "package.zip";
    public static final String SIGNATURE_FILENAME = "signature";
    public static final String TIMESTAMP_FILENAME = "timestamp";
    public static final String CADES_SIGNATURE_EXTENSION = ".p7s";
    public static final String TST_EXTENSION = ".tst";
    public static final String XML_EXTENSION = ".xml";

    private ASiCUtils() {
    }

    public static boolean isSignature(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(SIGNATURE_FILENAME) && !entryName.contains(MANIFEST_FILENAME);
    }

    public static boolean isTimestamp(String entryName) {
        return entryName.startsWith(META_INF_FOLDER) && entryName.contains(TIMESTAMP_FILENAME) && entryName.endsWith(TST_EXTENSION);
    }

    public static String getMimeTypeString(ASiCParameters asicParameters) {
        String asicParameterMimeType = asicParameters.getMimeType();
        String mimeTypeString = Utils.isStringBlank((String)asicParameterMimeType) ? (ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE.getMimeTypeString() : MimeType.ASICS.getMimeTypeString()) : asicParameterMimeType;
        return mimeTypeString;
    }

    public static String getZipComment(ASiCParameters asicParameters) {
        if (asicParameters.isZipComment()) {
            return ASiCUtils.getZipComment(ASiCUtils.getMimeTypeString(asicParameters));
        }
        return "";
    }

    public static String getZipComment(String mimeTypeString) {
        return MIME_TYPE_COMMENT + mimeTypeString;
    }

    public static boolean isASiCMimeType(MimeType mimeType) {
        return MimeType.ASICS.equals((Object)mimeType) || MimeType.ASICE.equals((Object)mimeType);
    }

    public static boolean isOpenDocumentMimeType(MimeType mimeType) {
        return MimeType.ODT.equals((Object)mimeType) || MimeType.ODS.equals((Object)mimeType) || MimeType.ODG.equals((Object)mimeType) || MimeType.ODP.equals((Object)mimeType);
    }

    public static ASiCContainerType getASiCContainerType(MimeType asicMimeType) {
        if (MimeType.ASICS.equals((Object)asicMimeType)) {
            return ASiCContainerType.ASiC_S;
        }
        if (MimeType.ASICE.equals((Object)asicMimeType) || ASiCUtils.isOpenDocumentMimeType(asicMimeType)) {
            return ASiCContainerType.ASiC_E;
        }
        throw new IllegalArgumentException("Not allowed mimetype '" + asicMimeType.getMimeTypeString() + "'");
    }

    public static boolean isASiCE(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_E.equals((Object)asicParameters.getContainerType());
    }

    public static boolean isASiCS(ASiCParameters asicParameters) {
        Objects.requireNonNull(asicParameters.getContainerType(), "ASiCContainerType must be defined!");
        return ASiCContainerType.ASiC_S.equals((Object)asicParameters.getContainerType());
    }

    public static MimeType getMimeType(ASiCParameters asicParameters) {
        return ASiCUtils.isASiCE(asicParameters) ? MimeType.ASICE : MimeType.ASICS;
    }

    public static boolean areFilesContainCorrectSignatureFileWithExtension(List<String> filenames, String extension) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename) || !filename.endsWith(extension)) continue;
            return true;
        }
        return false;
    }

    public static boolean areFilesContainSignatures(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isSignature(filename)) continue;
            return true;
        }
        return false;
    }

    public static boolean isAsicFileContent(List<String> filenames) {
        return ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, XML_EXTENSION) || ASiCUtils.areFilesContainTimestamp(filenames);
    }

    public static boolean areFilesContainTimestamp(List<String> filenames) {
        for (String filename : filenames) {
            if (!ASiCUtils.isTimestamp(filename)) continue;
            return true;
        }
        return false;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean isZip(DSSDocument document) {
        if (document == null) {
            return false;
        }
        byte[] preamble = new byte[2];
        try (InputStream is = document.openStream();){
            int r = is.read(preamble, 0, 2);
            if (r != 2) {
                boolean bl = false;
                return bl;
            }
        }
        catch (IOException e) {
            throw new DSSException("Unable to read the 2 first bytes", (Throwable)e);
        }
        if (preamble[0] != 80) return false;
        if (preamble[1] != 75) return false;
        return true;
    }

    public static boolean isASiCWithCAdES(List<String> filenames) {
        return ASiCUtils.areFilesContainCorrectSignatureFileWithExtension(filenames, CADES_SIGNATURE_EXTENSION) || ASiCUtils.areFilesContainTimestamp(filenames);
    }

    public static boolean isXAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(XML_EXTENSION);
    }

    public static boolean isCAdES(String entryName) {
        return ASiCUtils.isSignature(entryName) && entryName.endsWith(CADES_SIGNATURE_EXTENSION);
    }

    public static boolean isOpenDocument(DSSDocument mimeTypeDoc) {
        MimeType mimeType = ASiCUtils.getMimeType(mimeTypeDoc);
        if (mimeTypeDoc != null) {
            return ASiCUtils.isOpenDocumentMimeType(mimeType);
        }
        return false;
    }

    public static boolean isMimetype(String entryName) {
        return MIME_TYPE.equals(entryName);
    }

    public static MimeType getMimeType(DSSDocument mimeTypeDocument) {
        MimeType mimeType;
        block9: {
            if (mimeTypeDocument == null) {
                return null;
            }
            InputStream is = mimeTypeDocument.openStream();
            try {
                byte[] byteArray = Utils.toByteArray((InputStream)is);
                String mimeTypeString = new String(byteArray, StandardCharsets.UTF_8);
                mimeType = MimeType.fromMimeTypeString((String)mimeTypeString);
                if (is == null) break block9;
            }
            catch (Throwable throwable) {
                try {
                    if (is != null) {
                        try {
                            is.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new DSSException((Throwable)e);
                }
            }
            is.close();
        }
        return mimeType;
    }

    public static ASiCContainerType getContainerType(DSSDocument archive, DSSDocument mimetype, String zipComment, List<DSSDocument> signedDocuments) {
        ASiCContainerType containerType = ASiCUtils.getContainerTypeFromMimeType(archive.getMimeType());
        if (containerType == null && (containerType = ASiCUtils.getContainerTypeFromMimeTypeDocument(mimetype)) == null) {
            containerType = ASiCUtils.getContainerTypeFromZipComment(zipComment);
        }
        if (containerType == null) {
            LOG.warn("Unable to define the ASiC Container type with its properties");
            if (Utils.collectionSize(signedDocuments) == 1) {
                containerType = ASiCContainerType.ASiC_S;
            } else if (Utils.collectionSize(signedDocuments) > 1) {
                containerType = ASiCContainerType.ASiC_E;
            } else {
                throw new DSSException("The provided file does not contain signed documents. The signature validation is not possible");
            }
        }
        return containerType;
    }

    private static ASiCContainerType getContainerTypeFromZipComment(String zipComment) {
        int indexOf;
        if (Utils.isStringNotBlank((String)zipComment) && (indexOf = zipComment.indexOf(MIME_TYPE_COMMENT)) > -1) {
            String asicCommentMimeTypeString = zipComment.substring(MIME_TYPE_COMMENT.length() + indexOf);
            MimeType mimeTypeFromZipComment = MimeType.fromMimeTypeString((String)asicCommentMimeTypeString);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromZipComment);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeTypeDocument(DSSDocument mimetype) {
        if (mimetype != null) {
            MimeType mimeTypeFromEmbeddedFile = ASiCUtils.getMimeType(mimetype);
            return ASiCUtils.getContainerTypeFromMimeType(mimeTypeFromEmbeddedFile);
        }
        return null;
    }

    private static ASiCContainerType getContainerTypeFromMimeType(MimeType mimeType) {
        if (ASiCUtils.isASiCMimeType(mimeType)) {
            return ASiCUtils.getASiCContainerType(mimeType);
        }
        return null;
    }

    public static String getPadNumber(int num) {
        String numStr = String.valueOf(num);
        String zeroPad = "000";
        return zeroPad.substring(numStr.length()) + numStr;
    }

    public static boolean isArchiveManifest(String fileName) {
        return fileName.contains(ASIC_ARCHIVE_MANIFEST_FILENAME) && fileName.endsWith(XML_EXTENSION);
    }

    public static String getNextASiCEManifestName(String expectedManifestName, List<DSSDocument> existingManifests) {
        List manifestNames = DSSUtils.getDocumentNames(existingManifests);
        String manifestName = null;
        for (int i = 0; i < existingManifests.size() + 1; ++i) {
            String suffix = i == 0 ? "" : String.valueOf(i);
            manifestName = META_INF_FOLDER + expectedManifestName + suffix + XML_EXTENSION;
            if (ASiCUtils.isValidName(manifestName, manifestNames)) break;
        }
        return manifestName;
    }

    private static boolean isValidName(String name, List<String> notValidNames) {
        return !notValidNames.contains(name);
    }

    public static boolean isASiCSArchive(DSSDocument document) {
        return Utils.areStringsEqual((String)PACKAGE_ZIP, (String)document.getName());
    }
}

