/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.asic.common;

import eu.europa.esig.dss.asic.common.ASiCUtils;
import eu.europa.esig.dss.asic.common.ContainerEntryDocument;
import eu.europa.esig.dss.asic.common.DSSZipEntry;
import eu.europa.esig.dss.asic.common.DSSZipEntryDocument;
import eu.europa.esig.dss.asic.common.FileArchiveEntry;
import eu.europa.esig.dss.asic.common.ZipContainerHandler;
import eu.europa.esig.dss.enumerations.MimeType;
import eu.europa.esig.dss.exception.IllegalInputException;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.FileDocument;
import eu.europa.esig.dss.model.InMemoryDocument;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.utils.Utils;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SecureContainerHandler
implements ZipContainerHandler {
    private static final Logger LOG = LoggerFactory.getLogger(SecureContainerHandler.class);
    private long threshold = 1000000L;
    private long maxCompressionRatio = 100L;
    private int maxAllowedFilesAmount = 1000;
    private int maxMalformedFiles = 100;
    private boolean extractComments = false;
    private int byteCounter = 0;
    private int malformedFilesCounter = 0;

    public void setThreshold(long threshold) {
        this.threshold = threshold;
    }

    public void setMaxCompressionRatio(long maxCompressionRatio) {
        this.maxCompressionRatio = maxCompressionRatio;
    }

    public void setMaxAllowedFilesAmount(int maxAllowedFilesAmount) {
        this.maxAllowedFilesAmount = maxAllowedFilesAmount;
    }

    public void setMaxMalformedFiles(int maxMalformedFiles) {
        this.maxMalformedFiles = maxMalformedFiles;
    }

    public void setExtractComments(boolean extractComments) {
        this.extractComments = extractComments;
    }

    @Override
    public List<DSSDocument> extractContainerContent(DSSDocument zipArchive) {
        this.resetCounters();
        ArrayList<DSSDocument> result = new ArrayList<DSSDocument>();
        if (this.isInFileProcessingSupported(zipArchive)) {
            FileDocument zipFileDocument = (FileDocument)zipArchive;
            List<ZipEntry> zipEntries = this.extractZipEntries((DSSDocument)zipFileDocument);
            if (!this.malformedEntriesDetected()) {
                for (ZipEntry zipEntry : zipEntries) {
                    result.add(new FileArchiveEntry(zipFileDocument, zipEntry));
                }
                return result;
            }
            LOG.warn("The archive with name '{}' contains malformed entries. Unable to parse with ZipFile. Continue with ZipInputStream...", (Object)zipArchive.getName());
        }
        long containerSize = DSSUtils.getFileByteSize((DSSDocument)zipArchive);
        try (InputStream is = zipArchive.openStream();
             ZipInputStream zis = new ZipInputStream(is);){
            DSSDocument document;
            while ((document = this.getNextDocument(zis, containerSize)) != null) {
                result.add(document);
                this.assertCollectionSizeValid(result);
            }
        }
        catch (IOException e) {
            throw new IllegalInputException("Unable to extract content from zip archive", (Throwable)e);
        }
        return result;
    }

    private boolean isInFileProcessingSupported(DSSDocument zipArchive) {
        if (zipArchive instanceof FileDocument) {
            boolean bl;
            ZipFile zipFile = new ZipFile(((FileDocument)zipArchive).getFile());
            try {
                bl = true;
            }
            catch (Throwable throwable) {
                try {
                    try {
                        zipFile.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    LOG.warn("Unable to process archive with name '{}' using in-file processing. Continue validation using in-memory processing. Reason : {}", new Object[]{zipArchive.getName(), e.getMessage(), e});
                }
            }
            zipFile.close();
            return bl;
        }
        return false;
    }

    private boolean malformedEntriesDetected() {
        return this.malformedFilesCounter > 0;
    }

    private DSSDocument getNextDocument(ZipInputStream zis, long containerSize) {
        ZipEntry entry = this.getNextValidEntry(zis);
        if (entry != null) {
            return this.getCurrentEntryDocument(zis, entry, containerSize);
        }
        return null;
    }

    @Override
    public List<String> extractEntryNames(DSSDocument zipArchive) {
        List<ZipEntry> zipEntries = this.extractZipEntries(zipArchive);
        if (Utils.isCollectionNotEmpty(zipEntries)) {
            return zipEntries.stream().map(ZipEntry::getName).collect(Collectors.toList());
        }
        return Collections.emptyList();
    }

    private List<ZipEntry> extractZipEntries(DSSDocument zipArchive) {
        this.resetCounters();
        long containerSize = DSSUtils.getFileByteSize((DSSDocument)zipArchive);
        long allowedSize = containerSize * this.maxCompressionRatio;
        ArrayList<ZipEntry> result = new ArrayList<ZipEntry>();
        try (InputStream is = zipArchive.openStream();
             ZipInputStream zis = new ZipInputStream(is);){
            ZipEntry entry;
            while ((entry = this.getNextValidEntry(zis)) != null) {
                result.add(entry);
                this.assertCollectionSizeValid(result);
                this.secureRead(zis, allowedSize);
            }
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Unable to extract entries from zip archive", e);
        }
        this.extractComments(zipArchive, result);
        return result;
    }

    private void extractComments(DSSDocument zipArchive, List<ZipEntry> zipEntries) {
        if (this.extractComments && zipArchive instanceof FileDocument) {
            FileDocument fileDocument = (FileDocument)zipArchive;
            try (ZipFile zipFile = new ZipFile(fileDocument.getFile());){
                for (ZipEntry zipEntry : zipEntries) {
                    ZipEntry zipFileEntry = zipFile.getEntry(zipEntry.getName());
                    zipEntry.setComment(zipFileEntry.getComment());
                }
            }
            catch (IOException e) {
                LOG.warn("Unable to read comments from zip archive", (Throwable)e);
            }
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    @Override
    public DSSDocument createZipArchive(List<DSSDocument> containerEntries, Date creationTime, String zipComment) {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            InMemoryDocument inMemoryDocument;
            try (ZipOutputStream zos = new ZipOutputStream(baos);){
                for (DSSDocument entry : containerEntries) {
                    ZipEntry zipEntry = this.getZipEntry(entry, creationTime);
                    zos.putNextEntry(zipEntry);
                    InputStream entryIS = entry.openStream();
                    try {
                        this.secureCopy(entryIS, zos, -1L);
                    }
                    finally {
                        if (entryIS == null) continue;
                        entryIS.close();
                    }
                }
                if (Utils.isStringNotEmpty((String)zipComment)) {
                    zos.setComment(zipComment);
                }
                zos.finish();
                inMemoryDocument = new InMemoryDocument(baos.toByteArray());
            }
            return inMemoryDocument;
        }
        catch (IOException e) {
            throw new DSSException(String.format("Unable to create an ASiC container. Reason : %s", e.getMessage()), (Throwable)e);
        }
    }

    private ZipEntry getZipEntry(DSSDocument entry, Date creationTime) {
        DSSZipEntry zipEntryWrapper;
        if (entry instanceof DSSZipEntryDocument) {
            DSSZipEntryDocument dssZipEntry = (DSSZipEntryDocument)entry;
            zipEntryWrapper = dssZipEntry.getZipEntry();
        } else {
            zipEntryWrapper = new DSSZipEntry(entry.getName());
        }
        ZipEntry zipEntry = zipEntryWrapper.createZipEntry();
        this.ensureCompressionMethod(zipEntry, entry);
        this.ensureTime(zipEntry, creationTime);
        return zipEntry;
    }

    private void ensureCompressionMethod(ZipEntry zipEntry, DSSDocument content) {
        if (ASiCUtils.isMimetype(zipEntry.getName())) {
            if (zipEntry.getMethod() != -1 && 0 != zipEntry.getMethod()) {
                LOG.warn("'mimetype' shall not be compressed! Compression method in its ZIP header will be set to zero.");
            }
            zipEntry.setMethod(0);
        }
        if (0 == zipEntry.getMethod()) {
            byte[] byteArray = DSSUtils.toByteArray((DSSDocument)content);
            zipEntry.setSize(byteArray.length);
            zipEntry.setCompressedSize(byteArray.length);
            CRC32 crc = new CRC32();
            crc.update(byteArray, 0, byteArray.length);
            zipEntry.setCrc(crc.getValue());
        }
    }

    private void ensureTime(ZipEntry zipEntry, Date creationTime) {
        if (creationTime != null) {
            zipEntry.setTime(creationTime.getTime());
        }
    }

    private void resetCounters() {
        this.byteCounter = 0;
        this.malformedFilesCounter = 0;
    }

    private ZipEntry getNextValidEntry(ZipInputStream zis) {
        while (this.malformedFilesCounter < this.maxMalformedFiles) {
            try {
                return zis.getNextEntry();
            }
            catch (Exception e) {
                LOG.warn("ZIP container contains a malformed, corrupted or not accessible entry! The entry is skipped. Reason: [{}]", (Object)e.getMessage());
                ++this.malformedFilesCounter;
                this.closeEntry(zis);
            }
        }
        throw new DSSException("Unable to retrieve a valid ZipEntry (" + this.maxMalformedFiles + " tries)");
    }

    private void closeEntry(ZipInputStream zis) {
        try {
            zis.closeEntry();
        }
        catch (IOException e) {
            throw new DSSException("Unable to close entry", (Throwable)e);
        }
    }

    private DSSDocument getCurrentEntryDocument(ZipInputStream zis, ZipEntry entry, long containerSize) {
        ContainerEntryDocument containerEntryDocument;
        long allowedSize = containerSize * this.maxCompressionRatio;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            this.secureCopy(zis, baos, allowedSize);
            baos.flush();
            InMemoryDocument currentDocument = new InMemoryDocument(baos.toByteArray());
            String fileName = entry.getName();
            currentDocument.setName(entry.getName());
            currentDocument.setMimeType(MimeType.fromFileName((String)fileName));
            containerEntryDocument = new ContainerEntryDocument((DSSDocument)currentDocument, new DSSZipEntry(entry));
        }
        catch (Throwable throwable) {
            try {
                try {
                    baos.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (IOException e) {
                this.closeEntry(zis);
                throw new DSSException(String.format("Unable to read an entry binaries. Reason : %s", e.getMessage()), (Throwable)e);
            }
        }
        baos.close();
        return containerEntryDocument;
    }

    private void secureCopy(InputStream is, OutputStream os, long allowedSize) throws IOException {
        int nRead;
        byte[] data = new byte[2048];
        while ((nRead = is.read(data)) != -1) {
            this.byteCounter += nRead;
            this.assertExtractEntryLengthValid(allowedSize);
            os.write(data, 0, nRead);
        }
    }

    private void secureRead(InputStream is, long allowedSize) throws IOException {
        int nRead;
        byte[] data = new byte[2048];
        while ((nRead = is.read(data)) != -1) {
            this.byteCounter += nRead;
            this.assertExtractEntryLengthValid(allowedSize);
        }
    }

    private void assertExtractEntryLengthValid(long allowedSize) {
        if (allowedSize != -1L && (long)this.byteCounter > this.threshold && (long)this.byteCounter > allowedSize) {
            throw new IllegalInputException("Zip Bomb detected in the ZIP container. Validation is interrupted.");
        }
    }

    private void assertCollectionSizeValid(Collection<?> collection) {
        if (collection.size() > this.maxAllowedFilesAmount) {
            throw new IllegalInputException("Too many files detected. Cannot extract ASiC content from the file.");
        }
    }
}

