/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.pdf.visible;

import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.MimeType;
import eu.europa.esig.dss.pades.SignatureFieldParameters;
import eu.europa.esig.dss.pades.SignatureImageParameters;
import eu.europa.esig.dss.pdf.AnnotationBox;
import eu.europa.esig.dss.pdf.visible.ImageAndResolution;
import eu.europa.esig.dss.utils.Utils;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.image.Raster;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Iterator;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageTypeSpecifier;
import javax.imageio.metadata.IIOMetadata;
import javax.imageio.stream.ImageInputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

public class ImageUtils {
    private static final Logger LOG = LoggerFactory.getLogger(ImageUtils.class);
    private static final int[] IMAGE_TRANSPARENT_TYPES;
    private static final int DEFAULT_DPI = 96;
    public static final int DEFAULT_FIRST_PAGE = 1;

    private ImageUtils() {
    }

    public static ImageAndResolution secureReadMetadata(SignatureImageParameters imageParameters) throws IOException {
        ImageAndResolution imageAndResolution;
        try {
            imageAndResolution = ImageUtils.readDisplayMetadata(imageParameters.getImage());
        }
        catch (Exception e) {
            LOG.warn("Cannot access the image metadata : {}. Returns default info.", (Object)e.getMessage());
            imageAndResolution = new ImageAndResolution(imageParameters.getImage(), imageParameters.getDpi(), imageParameters.getDpi());
        }
        return imageAndResolution;
    }

    public static ImageAndResolution readDisplayMetadata(DSSDocument image) throws IOException {
        if (ImageUtils.isImageWithContentType(image, MimeType.JPEG)) {
            return ImageUtils.readAndDisplayMetadataJPEG(image);
        }
        if (ImageUtils.isImageWithContentType(image, MimeType.PNG)) {
            return ImageUtils.readAndDisplayMetadataPNG(image);
        }
        throw new DSSException("Unsupported image type");
    }

    private static boolean isImageWithContentType(DSSDocument image, MimeType expectedContentType) {
        if (image.getMimeType() != null) {
            return image.getMimeType().equals((Object)expectedContentType);
        }
        if (image.getName() != null) {
            String contentType = null;
            try {
                contentType = Files.probeContentType(Paths.get(image.getName(), new String[0]));
            }
            catch (IOException e) {
                LOG.warn("Unable to retrieve the content-type : {}", (Object)e.getMessage());
            }
            catch (Exception e) {
                throw new DSSException("An error occurred during an attempt to read the image's content type", (Throwable)e);
            }
            return Utils.areStringsEqual((String)expectedContentType.getMimeTypeString(), (String)contentType);
        }
        throw new DSSException("Cannot read image metadata. MimeType or image name must be specified!");
    }

    private static ImageAndResolution readAndDisplayMetadataJPEG(DSSDocument image) throws IOException {
        try (InputStream is = image.openStream();){
            ImageAndResolution imageAndResolution;
            block14: {
                ImageInputStream iis = ImageIO.createImageInputStream(is);
                try {
                    ImageReader reader = ImageUtils.getImageReader("jpeg");
                    reader.setInput(iis, true);
                    int hdpi = 96;
                    int vdpi = 96;
                    if (ImageUtils.isSupportedColorSpace(reader)) {
                        IIOMetadata metadata = reader.getImageMetadata(0);
                        Element root = (Element)metadata.getAsTree("javax_imageio_jpeg_image_1.0");
                        NodeList elements = root.getElementsByTagName("app0JFIF");
                        Element e = (Element)elements.item(0);
                        hdpi = Integer.parseInt(e.getAttribute("Xdensity"));
                        vdpi = Integer.parseInt(e.getAttribute("Ydensity"));
                    } else {
                        LOG.warn("Cannot read metadata of the image with name [{}]. The color space is not supported. Using the default dpi with value [{}]", (Object)image.getName(), (Object)96);
                    }
                    imageAndResolution = new ImageAndResolution(image, hdpi, vdpi);
                    if (iis == null) break block14;
                }
                catch (Throwable throwable) {
                    if (iis != null) {
                        try {
                            iis.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                iis.close();
            }
            return imageAndResolution;
        }
    }

    private static ImageAndResolution readAndDisplayMetadataPNG(DSSDocument image) throws IOException {
        try (InputStream is = image.openStream();){
            ImageAndResolution imageAndResolution;
            block18: {
                ImageInputStream iis = ImageIO.createImageInputStream(is);
                try {
                    ImageReader reader = ImageUtils.getImageReader("png");
                    reader.setInput(iis, true);
                    int hdpi = 96;
                    int vdpi = 96;
                    if (ImageUtils.isSupportedColorSpace(reader)) {
                        IIOMetadata metadata = reader.getImageMetadata(0);
                        double mm2inch = 25.4;
                        Element node = (Element)metadata.getAsTree("javax_imageio_1.0");
                        NodeList lst = node.getElementsByTagName("HorizontalPixelSize");
                        if (lst != null && lst.getLength() == 1) {
                            hdpi = (int)(mm2inch / (double)Float.parseFloat(((Element)lst.item(0)).getAttribute("value")));
                        } else {
                            LOG.debug("Cannot get HorizontalPixelSize value. Using the default dpi [{}]", (Object)96);
                        }
                        lst = node.getElementsByTagName("VerticalPixelSize");
                        if (lst != null && lst.getLength() == 1) {
                            vdpi = (int)(mm2inch / (double)Float.parseFloat(((Element)lst.item(0)).getAttribute("value")));
                        } else {
                            LOG.debug("Cannot get HorizontalPixelSize value. Using the default dpi [{}]", (Object)96);
                        }
                    } else {
                        LOG.warn("Cannot read metadata of the image with name [{}]. The color space is not supported. Using the default dpi with value [{}]", (Object)image.getName(), (Object)96);
                    }
                    imageAndResolution = new ImageAndResolution(image, hdpi, vdpi);
                    if (iis == null) break block18;
                }
                catch (Throwable throwable) {
                    if (iis != null) {
                        try {
                            iis.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                iis.close();
            }
            return imageAndResolution;
        }
    }

    private static boolean isSupportedColorSpace(ImageReader reader) throws IOException {
        Iterator<ImageTypeSpecifier> imageTypes = reader.getImageTypes(0);
        return imageTypes.hasNext();
    }

    public static AnnotationBox getImageBoundaryBox(SignatureImageParameters imageParameters) {
        float scaleFactor;
        float height;
        float width;
        block9: {
            SignatureFieldParameters fieldParameters = imageParameters.getFieldParameters();
            width = fieldParameters.getWidth();
            height = fieldParameters.getHeight();
            scaleFactor = ImageUtils.getScaleFactor(imageParameters.getZoom());
            if (width == 0.0f && height == 0.0f) {
                try {
                    DSSDocument docImage = imageParameters.getImage();
                    if (docImage == null) break block9;
                    try (InputStream is = docImage.openStream();){
                        BufferedImage bufferedImage = ImageUtils.read(is);
                        width = bufferedImage.getWidth();
                        height = bufferedImage.getHeight();
                    }
                }
                catch (IOException e) {
                    LOG.error("Cannot read the given image", (Throwable)e);
                }
            }
        }
        return new AnnotationBox(0.0f, 0.0f, width * scaleFactor, height * scaleFactor);
    }

    public static float getScaleFactor(int zoom) {
        return (float)zoom / 100.0f;
    }

    public static BufferedImage read(InputStream is) throws IOException {
        try (ImageInputStream iis = ImageIO.createImageInputStream(is);){
            ImageReader imageReader = ImageUtils.getImageReader(iis);
            imageReader.setInput(iis, true, true);
            if (ImageUtils.isSupportedColorSpace(imageReader)) {
                BufferedImage bufferedImage = imageReader.read(0, imageReader.getDefaultReadParam());
                return bufferedImage;
            }
            LOG.warn("The image format is not supported by the current ImageReader!");
            Raster raster = ImageUtils.getRaster(imageReader);
            if (ImageUtils.isCMYKType(raster)) {
                LOG.info("Converting from CMYK to RGB...");
                BufferedImage bufferedImage = ImageUtils.convertCMYKToRGB(raster);
                return bufferedImage;
            }
            throw new DSSException("The color space of image is not supported!");
        }
    }

    private static Raster getRaster(ImageReader imageReader) throws IOException {
        return imageReader.readRaster(0, imageReader.getDefaultReadParam());
    }

    private static boolean isCMYKType(Raster raster) {
        return raster.getNumBands() == 4;
    }

    private static BufferedImage convertCMYKToRGB(Raster raster) throws IOException {
        int width = raster.getWidth();
        int height = raster.getHeight();
        BufferedImage rgbImage = new BufferedImage(width, height, 1);
        int[] cmyk = new int[4];
        for (int y = 0; y < height; ++y) {
            for (int x = 0; x < width; ++x) {
                cmyk = raster.getPixel(x, y, cmyk);
                int r = cmyk[0] * cmyk[3] / 255;
                int g = cmyk[1] * cmyk[3] / 255;
                int b = cmyk[2] * cmyk[3] / 255;
                int p = r << 16 | g << 8 | b;
                rgbImage.setRGB(x, y, p);
            }
        }
        return rgbImage;
    }

    public static BufferedImage rotate(BufferedImage image, double angle) {
        double sin = Math.abs(Math.sin(Math.toRadians(angle)));
        double cos = Math.abs(Math.cos(Math.toRadians(angle)));
        int w = image.getWidth();
        int h = image.getHeight();
        double neww = Math.floor((double)w * cos + (double)h * sin);
        double newh = Math.floor((double)h * cos + (double)w * sin);
        BufferedImage result = new BufferedImage((int)neww, (int)newh, image.getType());
        Graphics2D g = result.createGraphics();
        g.translate((neww - (double)w) / 2.0, (newh - (double)h) / 2.0);
        g.rotate(Math.toRadians(angle), (double)w / 2.0, (double)h / 2.0);
        g.drawRenderedImage(image, null);
        g.dispose();
        return result;
    }

    private static ImageReader getImageReader(String type) {
        Iterator<ImageReader> readers = ImageIO.getImageReadersByFormatName(type);
        ImageReader reader = ImageUtils.getRasterReader(readers);
        if (reader == null) {
            throw new DSSException("No reader for '" + type + "' found");
        }
        return reader;
    }

    private static ImageReader getImageReader(ImageInputStream iis) {
        Iterator<ImageReader> readers = ImageIO.getImageReaders(iis);
        ImageReader reader = ImageUtils.getRasterReader(readers);
        if (reader == null) {
            throw new DSSException("No reader for the image found");
        }
        return reader;
    }

    private static ImageReader getRasterReader(Iterator<ImageReader> readers) {
        ImageReader reader = null;
        while (readers.hasNext() && !(reader = readers.next()).canReadRaster()) {
        }
        return reader;
    }

    public static boolean isTransparent(BufferedImage bufferedImage) {
        int type = bufferedImage.getType();
        return Arrays.binarySearch(IMAGE_TRANSPARENT_TYPES, type) > -1;
    }

    public static int getImageType(BufferedImage image) {
        int imageType = 1;
        if (ImageUtils.isTransparent(image)) {
            LOG.warn("Transparency detected and enabled (Be aware: not valid with PDF/A !)");
            imageType = 2;
        }
        return imageType;
    }

    public static boolean imagesEqual(BufferedImage img1, BufferedImage img2) {
        if (ImageUtils.imageDimensionsEqual(img1, img2)) {
            int diffAmount = ImageUtils.drawSubtractionImage(img1, img2, null);
            return diffAmount == 0;
        }
        return false;
    }

    public static boolean imageDimensionsEqual(BufferedImage img1, BufferedImage img2) {
        if (img1.getWidth() != img2.getWidth() || img1.getHeight() != img2.getHeight()) {
            LOG.warn("Screenshot comparison error! Images dimensions mismatch.");
            return false;
        }
        return true;
    }

    public static int drawSubtractionImage(BufferedImage img1, BufferedImage img2, BufferedImage outImg) {
        int diffAmount = 0;
        for (int i = 0; i < img1.getHeight() && i < img2.getHeight(); ++i) {
            for (int j = 0; j < img1.getWidth() && j < img2.getWidth(); ++j) {
                int rgb1 = img1.getRGB(j, i);
                int rgb2 = img2.getRGB(j, i);
                int r1 = rgb1 >> 16 & 0xFF;
                int g1 = rgb1 >> 8 & 0xFF;
                int b1 = rgb1 & 0xFF;
                int r2 = rgb2 >> 16 & 0xFF;
                int g2 = rgb2 >> 8 & 0xFF;
                int b2 = rgb2 & 0xFF;
                int diff = Math.abs(r1 - r2);
                diff += Math.abs(g1 - g2);
                if ((diff += Math.abs(b1 - b2)) > 0) {
                    ++diffAmount;
                }
                if (outImg == null) continue;
                int result = (diff /= 3) << 16 | diff << 8 | diff;
                outImg.setRGB(j, i, result);
            }
        }
        return diffAmount;
    }

    static {
        int[] imageAlphaTypes = new int[]{6, 7, 2, 3};
        Arrays.sort(imageAlphaTypes);
        IMAGE_TRANSPARENT_TYPES = imageAlphaTypes;
    }
}

