/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss.service.http.commons;

import eu.europa.esig.dss.enumerations.DigestAlgorithm;
import eu.europa.esig.dss.model.DSSDocument;
import eu.europa.esig.dss.model.DSSException;
import eu.europa.esig.dss.model.FileDocument;
import eu.europa.esig.dss.service.http.commons.ResourceLoader;
import eu.europa.esig.dss.spi.DSSUtils;
import eu.europa.esig.dss.spi.client.http.DSSFileLoader;
import eu.europa.esig.dss.spi.client.http.DataLoader;
import eu.europa.esig.dss.spi.client.http.Protocol;
import eu.europa.esig.dss.spi.exception.DSSDataLoaderMultipleException;
import eu.europa.esig.dss.utils.Utils;
import java.io.File;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Objects;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileCacheDataLoader
implements DataLoader,
DSSFileLoader {
    private static final long serialVersionUID = 1028849693098211169L;
    private static final Logger LOG = LoggerFactory.getLogger(FileCacheDataLoader.class);
    private static final String DATA_LOADER_NOT_CONFIGURED = "The DataLoader is not configured";
    private File fileCacheDirectory = new File(System.getProperty("java.io.tmpdir"));
    private ResourceLoader resourceLoader = new ResourceLoader();
    private List<String> toBeLoaded;
    private List<String> toIgnored;
    private Long cacheExpirationTime;
    private DataLoader dataLoader;

    public FileCacheDataLoader() {
    }

    public FileCacheDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public DataLoader getDataLoader() {
        return this.dataLoader;
    }

    public void setDataLoader(DataLoader dataLoader) {
        this.dataLoader = dataLoader;
    }

    public void setFileCacheDirectory(File fileCacheDirectory) {
        this.fileCacheDirectory = fileCacheDirectory;
        this.fileCacheDirectory.mkdirs();
    }

    public void setCacheExpirationTime(long cacheExpirationTimeInMilliseconds) {
        this.cacheExpirationTime = cacheExpirationTimeInMilliseconds;
    }

    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    public void addToBeLoaded(String url) {
        if (this.toBeLoaded == null) {
            this.toBeLoaded = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank((String)url)) {
            this.toBeLoaded.add(url);
        }
    }

    public void addToBeIgnored(String urlString) {
        if (this.toIgnored == null) {
            this.toIgnored = new ArrayList<String>();
        }
        if (Utils.isStringNotBlank((String)urlString)) {
            String normalizedFileName = DSSUtils.getNormalizedString((String)urlString);
            this.toIgnored.add(normalizedFileName);
        }
    }

    public byte[] get(String url, boolean refresh) throws DSSException {
        DSSDocument document = this.getDocument(url, refresh);
        return DSSUtils.toByteArray((DSSDocument)document);
    }

    public byte[] get(String url) throws DSSException {
        return this.get(url, false);
    }

    private DSSDocument getDocument(String url, boolean refresh) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        if (this.toBeLoaded != null && !this.toBeLoaded.contains(url)) {
            throw new DSSException(String.format("The toBeLoaded list does not contain URL [%s]!", url));
        }
        String fileName = DSSUtils.getNormalizedString((String)url);
        File file = this.getCacheFile(fileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !refresh && !isCacheExpired) {
            LOG.debug("Cached file was used");
            return new FileDocument(file);
        }
        if (!fileExists) {
            LOG.debug("There is no cached file!");
        } else {
            LOG.debug("The refresh is forced!");
        }
        byte[] bytes = null;
        bytes = !this.isNetworkProtocol(url) ? this.getLocalFileContent(url) : this.dataLoader.get(url);
        if (Utils.isArrayNotEmpty((byte[])bytes)) {
            File out = this.createFile(fileName, bytes);
            return new FileDocument(out);
        }
        throw new DSSException(String.format("Cannot retrieve data from url [%s]. Empty content is obtained!", url));
    }

    public DSSDocument getDocument(String url) {
        return this.getDocument(url, false);
    }

    public boolean remove(String url) {
        String fileName = DSSUtils.getNormalizedString((String)url);
        File file = this.getCacheFile(fileName);
        return file.delete();
    }

    protected boolean isNetworkProtocol(String urlString) {
        String normalizedUrl = Utils.trim((String)urlString).toLowerCase();
        return Protocol.isHttpUrl((String)normalizedUrl) || Protocol.isLdapUrl((String)normalizedUrl) || Protocol.isFtpUrl((String)normalizedUrl);
    }

    private byte[] getLocalFileContent(String urlString) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        String resourcePath = this.resourceLoader.getAbsoluteResourceFolder(Utils.trim((String)urlString));
        if (resourcePath != null) {
            File fileResource = new File(resourcePath);
            return DSSUtils.toByteArray((File)fileResource);
        }
        return this.dataLoader.get(urlString);
    }

    private File getCacheFile(String fileName) {
        String trimmedFileName = Utils.trim((String)fileName);
        if (this.toIgnored != null && this.toIgnored.contains(trimmedFileName)) {
            throw new DSSException("Part of urls to ignore.");
        }
        LOG.debug("Cached file: {}/{}", (Object)this.fileCacheDirectory, (Object)trimmedFileName);
        File file = new File(this.fileCacheDirectory, trimmedFileName);
        return file;
    }

    public File createFile(String urlString, byte[] bytes) {
        String fileName = DSSUtils.getNormalizedString((String)urlString);
        File file = this.getCacheFile(fileName);
        DSSUtils.saveToFile((byte[])bytes, (File)file);
        return file;
    }

    public byte[] loadFileFromCache(String urlString) throws DSSException {
        String fileName = DSSUtils.getNormalizedString((String)urlString);
        File file = this.getCacheFile(fileName);
        if (file.exists()) {
            byte[] bytes = DSSUtils.toByteArray((File)file);
            return bytes;
        }
        throw new DSSException(String.format("The file with URL [%s] does not exist in the cache!", urlString));
    }

    public byte[] post(String urlString, byte[] content) throws DSSException {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        String fileName = DSSUtils.getNormalizedString((String)urlString);
        byte[] digest = DSSUtils.digest((DigestAlgorithm)DigestAlgorithm.MD5, (byte[])content);
        String digestHexEncoded = DSSUtils.toHex((byte[])digest);
        String cacheFileName = fileName + "." + digestHexEncoded;
        File file = this.getCacheFile(cacheFileName);
        boolean fileExists = file.exists();
        boolean isCacheExpired = this.isCacheExpired(file);
        if (fileExists && !isCacheExpired) {
            LOG.debug("Cached file was used");
            byte[] byteArray = DSSUtils.toByteArray((File)file);
            return byteArray;
        }
        LOG.debug("There is no cached file!");
        byte[] returnedBytes = null;
        if (this.isNetworkProtocol(urlString)) {
            returnedBytes = this.dataLoader.post(urlString, content);
        }
        if (Utils.isArrayNotEmpty(returnedBytes)) {
            File cacheFile = this.getCacheFile(cacheFileName);
            DSSUtils.saveToFile((byte[])returnedBytes, (File)cacheFile);
            return returnedBytes;
        }
        throw new DSSException(String.format("Cannot retrieve data from URL [%s]", urlString));
    }

    private boolean isCacheExpired(File file) {
        if (this.cacheExpirationTime == null) {
            return false;
        }
        if (!file.exists()) {
            return true;
        }
        long currentTime = new Date().getTime();
        if (currentTime - file.lastModified() > this.cacheExpirationTime) {
            LOG.debug("Cache is expired");
            return true;
        }
        return false;
    }

    public DataLoader.DataAndUrl get(List<String> urlStrings) {
        if (Utils.isCollectionEmpty(urlStrings)) {
            throw new DSSException("Cannot process the GET call. List of URLs is empty!");
        }
        HashMap<String, Exception> exceptions = new HashMap<String, Exception>();
        for (String urlString : urlStrings) {
            LOG.debug("Processing a GET call to URL [{}]...", (Object)urlString);
            try {
                byte[] bytes = this.get(urlString);
                if (Utils.isArrayEmpty((byte[])bytes)) {
                    LOG.debug("The retrieved content from URL [{}] is empty. Continue with other URLs...", (Object)urlString);
                    continue;
                }
                return new DataLoader.DataAndUrl(bytes, urlString);
            }
            catch (Exception e) {
                LOG.warn("Cannot obtain data using '{}' : {}", (Object)urlString, (Object)e.getMessage());
                exceptions.put(urlString, e);
            }
        }
        throw new DSSDataLoaderMultipleException(exceptions);
    }

    public void setContentType(String contentType) {
        Objects.requireNonNull(this.dataLoader, DATA_LOADER_NOT_CONFIGURED);
        this.dataLoader.setContentType(contentType);
    }
}

