/*
 * Decompiled with CFR 0.152.
 */
package eu.europa.esig.dss;

import eu.europa.esig.dss.DSSException;
import eu.europa.esig.dss.DSSUtils;
import eu.europa.esig.dss.DigestAlgorithm;
import eu.europa.esig.dss.tsl.ServiceInfo;
import eu.europa.esig.dss.tsl.ServiceInfoStatus;
import eu.europa.esig.dss.utils.Utils;
import eu.europa.esig.dss.x509.CertificateToken;
import java.io.Closeable;
import java.io.IOException;
import java.math.BigInteger;
import java.security.Provider;
import java.security.Security;
import java.security.cert.CertificateException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import javax.naming.ldap.Rdn;
import javax.security.auth.x500.X500Principal;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.ASN1GeneralizedTime;
import org.bouncycastle.asn1.ASN1InputStream;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.asn1.ASN1OctetString;
import org.bouncycastle.asn1.ASN1Primitive;
import org.bouncycastle.asn1.ASN1Sequence;
import org.bouncycastle.asn1.ASN1String;
import org.bouncycastle.asn1.DERBitString;
import org.bouncycastle.asn1.DERNull;
import org.bouncycastle.asn1.DEROctetString;
import org.bouncycastle.asn1.DERSequence;
import org.bouncycastle.asn1.DERTaggedObject;
import org.bouncycastle.asn1.DLSequence;
import org.bouncycastle.asn1.DLSet;
import org.bouncycastle.asn1.ocsp.BasicOCSPResponse;
import org.bouncycastle.asn1.ocsp.OCSPObjectIdentifiers;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x509.AccessDescription;
import org.bouncycastle.asn1.x509.AlgorithmIdentifier;
import org.bouncycastle.asn1.x509.AuthorityInformationAccess;
import org.bouncycastle.asn1.x509.CRLDistPoint;
import org.bouncycastle.asn1.x509.DistributionPoint;
import org.bouncycastle.asn1.x509.DistributionPointName;
import org.bouncycastle.asn1.x509.Extension;
import org.bouncycastle.asn1.x509.GeneralName;
import org.bouncycastle.asn1.x509.GeneralNames;
import org.bouncycastle.asn1.x509.IssuerSerial;
import org.bouncycastle.asn1.x509.KeyPurposeId;
import org.bouncycastle.asn1.x509.PolicyInformation;
import org.bouncycastle.asn1.x509.SubjectKeyIdentifier;
import org.bouncycastle.asn1.x509.Time;
import org.bouncycastle.asn1.x509.X509ObjectIdentifiers;
import org.bouncycastle.asn1.x509.qualified.QCStatement;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cert.ocsp.BasicOCSPResp;
import org.bouncycastle.cms.CMSSignedData;
import org.bouncycastle.cms.SignerInformation;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.tsp.TimeStampToken;
import org.bouncycastle.x509.extension.X509ExtensionUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class DSSASN1Utils {
    private static final Logger LOG = LoggerFactory.getLogger(DSSASN1Utils.class);
    private static final String QC_TYPE_STATEMENT_OID = "0.4.0.1862.1.6";
    private static final BouncyCastleProvider securityProvider = new BouncyCastleProvider();

    private DSSASN1Utils() {
    }

    public static <T extends ASN1Primitive> T toASN1Primitive(byte[] bytes) throws DSSException {
        try {
            ASN1Primitive asn1Primitive = ASN1Primitive.fromByteArray((byte[])bytes);
            return (T)asn1Primitive;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    private static boolean isDEROctetStringNull(DEROctetString derOctetString) {
        byte[] derOctetStringBytes = derOctetString.getOctets();
        Object asn1Null = DSSASN1Utils.toASN1Primitive(derOctetStringBytes);
        return DERNull.INSTANCE.equals(asn1Null);
    }

    public static byte[] getDEREncoded(ASN1Encodable asn1Encodable) {
        try {
            return asn1Encodable.toASN1Primitive().getEncoded("DER");
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static byte[] getEncoded(BasicOCSPResp basicOCSPResp) {
        try {
            BasicOCSPResponse basicOCSPResponse = BasicOCSPResponse.getInstance((Object)basicOCSPResp.getEncoded());
            return DSSASN1Utils.getDEREncoded((ASN1Encodable)basicOCSPResponse);
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static Date toDate(ASN1GeneralizedTime asn1Date) throws DSSException {
        try {
            return asn1Date.getDate();
        }
        catch (ParseException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static String toString(ASN1OctetString value) {
        return new String(value.getOctets());
    }

    public static byte[] getEncoded(TimeStampToken timeStampToken) throws DSSException {
        try {
            byte[] encoded = timeStampToken.getEncoded();
            return encoded;
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static ASN1Sequence getAsn1SequenceFromDerOctetString(byte[] bytes) throws DSSException {
        ASN1Sequence aSN1Sequence;
        ASN1InputStream input = null;
        try {
            ASN1Sequence seq;
            input = new ASN1InputStream(bytes);
            DEROctetString s = (DEROctetString)input.readObject();
            byte[] content = s.getOctets();
            input.close();
            input = new ASN1InputStream(content);
            aSN1Sequence = seq = (ASN1Sequence)input.readObject();
        }
        catch (IOException e) {
            try {
                throw new DSSException("Error when computing certificate's extensions.", (Throwable)e);
            }
            catch (Throwable throwable) {
                Utils.closeQuietly(input);
                throw throwable;
            }
        }
        Utils.closeQuietly((Closeable)input);
        return aSN1Sequence;
    }

    public static byte[] getAsn1SignaturePolicyDigest(DigestAlgorithm digestAlgorithm, byte[] policyBytes) {
        ASN1Sequence asn1Seq = (ASN1Sequence)DSSASN1Utils.toASN1Primitive(policyBytes);
        ASN1Sequence signPolicyHashAlgObject = (ASN1Sequence)asn1Seq.getObjectAt(0);
        AlgorithmIdentifier signPolicyHashAlgIdentifier = AlgorithmIdentifier.getInstance((Object)signPolicyHashAlgObject);
        ASN1Sequence signPolicyInfo = (ASN1Sequence)asn1Seq.getObjectAt(1);
        byte[] hashAlgorithmDEREncoded = DSSASN1Utils.getDEREncoded((ASN1Encodable)signPolicyHashAlgIdentifier);
        byte[] signPolicyInfoDEREncoded = DSSASN1Utils.getDEREncoded((ASN1Encodable)signPolicyInfo);
        return DSSUtils.digest(digestAlgorithm, hashAlgorithmDEREncoded, signPolicyInfoDEREncoded);
    }

    public static String getCanonicalizedName(GeneralNames generalNames) {
        GeneralName[] names = generalNames.getNames();
        TreeMap<String, String> treeMap = new TreeMap<String, String>();
        for (GeneralName name : names) {
            String ldapString = String.valueOf(name.getName());
            LOG.debug("ldapString to canonicalize: {} ", (Object)ldapString);
            try {
                LdapName ldapName = new LdapName(ldapString);
                List<Rdn> rdns = ldapName.getRdns();
                for (Rdn rdn : rdns) {
                    treeMap.put(rdn.getType().toLowerCase(), String.valueOf(rdn.getValue()).toLowerCase());
                }
            }
            catch (InvalidNameException e) {
                throw new DSSException((Throwable)e);
            }
        }
        StringBuilder stringBuilder = new StringBuilder();
        for (Map.Entry entry : treeMap.entrySet()) {
            stringBuilder.append((String)entry.getKey()).append('=').append((String)entry.getValue()).append('|');
        }
        String canonicalizedName = stringBuilder.toString();
        LOG.debug("canonicalizedName: {} ", (Object)canonicalizedName);
        return canonicalizedName;
    }

    public static AlgorithmIdentifier getAlgorithmIdentifier(DigestAlgorithm digestAlgorithm) {
        ASN1ObjectIdentifier asn1ObjectIdentifier = new ASN1ObjectIdentifier(digestAlgorithm.getOid());
        AlgorithmIdentifier algorithmIdentifier = new AlgorithmIdentifier(asn1ObjectIdentifier, (ASN1Encodable)DERNull.INSTANCE);
        return algorithmIdentifier;
    }

    public static boolean hasIdPkixOcspNoCheckExtension(CertificateToken token) {
        byte[] extensionValue = token.getCertificate().getExtensionValue(OCSPObjectIdentifiers.id_pkix_ocsp_nocheck.getId());
        if (extensionValue != null) {
            try {
                Object derObject = DSSASN1Utils.toASN1Primitive(extensionValue);
                if (derObject instanceof DEROctetString) {
                    return DSSASN1Utils.isDEROctetStringNull((DEROctetString)derObject);
                }
            }
            catch (Exception e) {
                LOG.debug("Exception when processing 'id_pkix_ocsp_no_check'", (Throwable)e);
            }
        }
        return false;
    }

    public static List<String> getPolicyIdentifiers(CertificateToken certToken) {
        ArrayList<String> policyIdentifiers = new ArrayList<String>();
        byte[] certificatePolicies = certToken.getCertificate().getExtensionValue(Extension.certificatePolicies.getId());
        if (certificatePolicies != null) {
            ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(certificatePolicies);
            for (int ii = 0; ii < seq.size(); ++ii) {
                PolicyInformation policyInfo = PolicyInformation.getInstance((Object)seq.getObjectAt(ii));
                policyIdentifiers.add(policyInfo.getPolicyIdentifier().getId());
            }
        }
        return policyIdentifiers;
    }

    public static List<String> getQCStatementsIdList(CertificateToken certToken) {
        ArrayList<String> extensionIdList = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (qcStatement != null) {
            ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
            for (int ii = 0; ii < seq.size(); ++ii) {
                QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(ii));
                extensionIdList.add(statement.getStatementId().getId());
            }
        }
        return extensionIdList;
    }

    public static List<String> getQCTypesIdList(CertificateToken certToken) {
        ArrayList<String> qcTypesIdList = new ArrayList<String>();
        byte[] qcStatement = certToken.getCertificate().getExtensionValue(Extension.qCStatements.getId());
        if (qcStatement != null) {
            ASN1Sequence seq = DSSASN1Utils.getAsn1SequenceFromDerOctetString(qcStatement);
            for (int ii = 0; ii < seq.size(); ++ii) {
                QCStatement statement = QCStatement.getInstance((Object)seq.getObjectAt(ii));
                if (!QC_TYPE_STATEMENT_OID.equals(statement.getStatementId().getId())) continue;
                ASN1Encodable qcTypeInfo1 = statement.getStatementInfo();
                if (qcTypeInfo1 instanceof ASN1Sequence) {
                    ASN1Sequence qcTypeInfo = (ASN1Sequence)qcTypeInfo1;
                    for (int jj = 0; jj < qcTypeInfo.size(); ++jj) {
                        ASN1Encodable e1 = qcTypeInfo.getObjectAt(jj);
                        if (!(e1 instanceof ASN1ObjectIdentifier)) {
                            throw new IllegalStateException("ASN1Sequence in QcTypes does not contain ASN1ObjectIdentifer, but " + e1.getClass().getName());
                        }
                        ASN1ObjectIdentifier oid = (ASN1ObjectIdentifier)e1;
                        qcTypesIdList.add(oid.getId());
                    }
                    continue;
                }
                throw new IllegalStateException("QcTypes not an ASN1Sequence, but " + qcTypeInfo1.getClass().getName());
            }
        }
        return qcTypesIdList;
    }

    public static byte[] getSki(CertificateToken certificateToken) throws DSSException {
        return DSSASN1Utils.getSki(certificateToken, false);
    }

    public static byte[] getSki(CertificateToken certificateToken, boolean computeIfMissing) throws DSSException {
        try {
            byte[] sKI = certificateToken.getCertificate().getExtensionValue(Extension.subjectKeyIdentifier.getId());
            if (Utils.isArrayNotEmpty((byte[])sKI)) {
                ASN1Primitive extension = X509ExtensionUtil.fromExtensionValue((byte[])sKI);
                SubjectKeyIdentifier skiBC = SubjectKeyIdentifier.getInstance((Object)extension);
                return skiBC.getKeyIdentifier();
            }
            if (computeIfMissing) {
                DLSequence seq = (DLSequence)DERSequence.fromByteArray((byte[])certificateToken.getPublicKey().getEncoded());
                DERBitString item = (DERBitString)seq.getObjectAt(1);
                return DSSUtils.digest(DigestAlgorithm.SHA1, item.getOctets());
            }
            return null;
        }
        catch (Exception e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static List<String> getCAAccessLocations(CertificateToken certificate) {
        return DSSASN1Utils.getAccessLocations(certificate, X509ObjectIdentifiers.id_ad_caIssuers);
    }

    public static List<String> getOCSPAccessLocations(CertificateToken certificate) {
        return DSSASN1Utils.getOCSPAccessLocations(certificate, true);
    }

    public static List<String> getOCSPAccessLocations(CertificateToken certificate, boolean checkInTrustAnchors) {
        List<String> ocspUrls = DSSASN1Utils.getAccessLocations(certificate, X509ObjectIdentifiers.id_ad_ocsp);
        if (Utils.isCollectionEmpty(ocspUrls) && checkInTrustAnchors) {
            return DSSASN1Utils.getServiceSupplyPoints(certificate, "ocsp");
        }
        return ocspUrls;
    }

    private static List<String> getAccessLocations(CertificateToken certificate, ASN1ObjectIdentifier aiaType) {
        ArrayList<String> locationsUrls = new ArrayList<String>();
        byte[] authInfoAccessExtensionValue = certificate.getCertificate().getExtensionValue(Extension.authorityInfoAccess.getId());
        if (null == authInfoAccessExtensionValue) {
            return locationsUrls;
        }
        try {
            AccessDescription[] accessDescriptions;
            ASN1Sequence asn1Sequence = DSSASN1Utils.getAsn1SequenceFromDerOctetString(authInfoAccessExtensionValue);
            AuthorityInformationAccess authorityInformationAccess = AuthorityInformationAccess.getInstance((Object)asn1Sequence);
            for (AccessDescription accessDescription : accessDescriptions = authorityInformationAccess.getAccessDescriptions()) {
                GeneralName gn;
                String location;
                if (!aiaType.equals((Object)accessDescription.getAccessMethod()) || (location = DSSASN1Utils.parseGn(gn = accessDescription.getAccessLocation())) == null) continue;
                locationsUrls.add(location);
            }
        }
        catch (Exception e) {
            LOG.error("Unable to parse authorityInfoAccess", (Throwable)e);
        }
        return locationsUrls;
    }

    public static List<String> getCrlUrls(CertificateToken certificateToken) {
        return DSSASN1Utils.getCrlUrls(certificateToken, true);
    }

    public static List<String> getCrlUrls(CertificateToken certificateToken, boolean checkInTrustAnchors) {
        ArrayList<String> urls = new ArrayList<String>();
        byte[] crlDistributionPointsBytes = certificateToken.getCertificate().getExtensionValue(Extension.cRLDistributionPoints.getId());
        if (crlDistributionPointsBytes != null) {
            try {
                DistributionPoint[] distributionPoints;
                ASN1Sequence asn1Sequence = DSSASN1Utils.getAsn1SequenceFromDerOctetString(crlDistributionPointsBytes);
                CRLDistPoint distPoint = CRLDistPoint.getInstance((Object)asn1Sequence);
                for (DistributionPoint distributionPoint : distributionPoints = distPoint.getDistributionPoints()) {
                    GeneralName[] names;
                    DistributionPointName distributionPointName = distributionPoint.getDistributionPoint();
                    if (0 != distributionPointName.getType()) continue;
                    GeneralNames generalNames = (GeneralNames)distributionPointName.getName();
                    for (GeneralName name : names = generalNames.getNames()) {
                        String location = DSSASN1Utils.parseGn(name);
                        if (location == null) continue;
                        urls.add(location);
                    }
                }
            }
            catch (Exception e) {
                LOG.error("Unable to parse cRLDistributionPoints", (Throwable)e);
            }
        }
        if (Utils.isCollectionEmpty(urls) && checkInTrustAnchors) {
            return DSSASN1Utils.getServiceSupplyPoints(certificateToken, "crl", "certificateRevocationList");
        }
        return urls;
    }

    private static List<String> getServiceSupplyPoints(CertificateToken certificateToken, String ... keywords) {
        ArrayList<String> urls = new ArrayList<String>();
        for (CertificateToken issuerToken = certificateToken.getIssuerToken(); issuerToken != null; issuerToken = issuerToken.getIssuerToken()) {
            if (!issuerToken.isTrusted() || !Utils.isCollectionNotEmpty((Collection)issuerToken.getAssociatedTSPS())) continue;
            Set services = issuerToken.getAssociatedTSPS();
            for (ServiceInfo serviceInfo : services) {
                for (ServiceInfoStatus serviceInfoStatus : serviceInfo.getStatus()) {
                    List serviceSupplyPoints = serviceInfoStatus.getServiceSupplyPoints();
                    if (!Utils.isCollectionNotEmpty((Collection)serviceSupplyPoints)) continue;
                    for (String serviceSupplyPoint : serviceSupplyPoints) {
                        for (String keyword : keywords) {
                            if (!serviceSupplyPoint.contains(keyword)) continue;
                            LOG.debug("ServiceSupplyPoints (TL) found for keyword '{}'", (Object)keyword);
                            urls.add(serviceSupplyPoint);
                        }
                    }
                }
            }
        }
        return urls;
    }

    private static String parseGn(GeneralName gn) {
        try {
            if (6 == gn.getTagNo()) {
                ASN1String str = (ASN1String)((DERTaggedObject)gn.toASN1Primitive()).getObject();
                return str.getString();
            }
        }
        catch (Exception e) {
            LOG.warn("Unable to parse GN " + gn, (Throwable)e);
        }
        return null;
    }

    public static boolean isOCSPSigning(CertificateToken certToken) {
        return DSSASN1Utils.isExtendedKeyUsagePresent(certToken, KeyPurposeId.id_kp_OCSPSigning.toOID());
    }

    public static boolean isExtendedKeyUsagePresent(CertificateToken certToken, ASN1ObjectIdentifier oid) {
        try {
            List<String> keyPurposes = certToken.getCertificate().getExtendedKeyUsage();
            if (keyPurposes != null && keyPurposes.contains(oid.getId())) {
                return true;
            }
        }
        catch (CertificateParsingException e) {
            LOG.error(e.getMessage(), (Throwable)e);
        }
        return false;
    }

    public static X509CertificateHolder getX509CertificateHolder(CertificateToken certToken) {
        try {
            return new X509CertificateHolder(certToken.getEncoded());
        }
        catch (IOException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static CertificateToken getCertificate(X509CertificateHolder x509CertificateHolder) {
        try {
            JcaX509CertificateConverter converter = new JcaX509CertificateConverter().setProvider("BC");
            X509Certificate x509Certificate = converter.getCertificate(x509CertificateHolder);
            return new CertificateToken(x509Certificate);
        }
        catch (CertificateException e) {
            throw new DSSException((Throwable)e);
        }
    }

    public static IssuerSerial getIssuerSerial(CertificateToken certToken) {
        X500Name issuerX500Name = DSSASN1Utils.getX509CertificateHolder(certToken).getIssuer();
        GeneralName generalName = new GeneralName(issuerX500Name);
        GeneralNames generalNames = new GeneralNames(generalName);
        BigInteger serialNumber = certToken.getCertificate().getSerialNumber();
        IssuerSerial issuerSerial = new IssuerSerial(generalNames, serialNumber);
        return issuerSerial;
    }

    public static Map<String, String> get(X500Principal x500Principal) {
        ASN1Encodable[] asn1Encodables;
        HashMap<String, String> treeMap = new HashMap<String, String>();
        byte[] encoded = x500Principal.getEncoded();
        ASN1Sequence asn1Sequence = ASN1Sequence.getInstance((Object)encoded);
        for (ASN1Encodable asn1Encodable : asn1Encodables = asn1Sequence.toArray()) {
            DLSet dlSet = (DLSet)asn1Encodable;
            for (int ii = 0; ii < dlSet.size(); ++ii) {
                DLSequence dlSequence = (DLSequence)dlSet.getObjectAt(ii);
                if (dlSequence.size() != 2) {
                    throw new DSSException("The DLSequence must contains exactly 2 elements.");
                }
                ASN1Encodable asn1EncodableAttributeType = dlSequence.getObjectAt(0);
                String stringAttributeType = DSSASN1Utils.getString(asn1EncodableAttributeType);
                ASN1Encodable asn1EncodableAttributeValue = dlSequence.getObjectAt(1);
                String stringAttributeValue = DSSASN1Utils.getString(asn1EncodableAttributeValue);
                treeMap.put(stringAttributeType, stringAttributeValue);
            }
        }
        return treeMap;
    }

    public static String getUtf8String(X500Principal x500Principal) {
        byte[] encoded = x500Principal.getEncoded();
        ASN1Sequence asn1Sequence = ASN1Sequence.getInstance((Object)encoded);
        ASN1Encodable[] asn1Encodables = asn1Sequence.toArray();
        StringBuilder stringBuilder = new StringBuilder();
        for (int ii = asn1Encodables.length - 1; ii >= 0; --ii) {
            ASN1Encodable asn1Encodable = asn1Encodables[ii];
            DLSet dlSet = (DLSet)asn1Encodable;
            for (int jj = 0; jj < dlSet.size(); ++jj) {
                DLSequence dlSequence = (DLSequence)dlSet.getObjectAt(jj);
                if (dlSequence.size() != 2) {
                    throw new DSSException("The DLSequence must contains exactly 2 elements.");
                }
                ASN1Encodable attributeType = dlSequence.getObjectAt(0);
                ASN1Encodable attributeValue = dlSequence.getObjectAt(1);
                String string = DSSASN1Utils.getString(attributeValue);
                string = Rdn.escapeValue(string);
                if (stringBuilder.length() != 0) {
                    stringBuilder.append(',');
                }
                stringBuilder.append(attributeType).append('=').append(string);
            }
        }
        return stringBuilder.toString();
    }

    private static String getString(ASN1Encodable attributeValue) {
        String string;
        if (attributeValue instanceof ASN1String) {
            string = ((ASN1String)attributeValue).getString();
        } else if (attributeValue instanceof ASN1ObjectIdentifier) {
            string = ((ASN1ObjectIdentifier)attributeValue).getId();
        } else {
            LOG.error("!!!*******!!! This encoding is unknown: " + attributeValue.getClass().getSimpleName());
            string = attributeValue.toString();
            LOG.error("!!!*******!!! value: " + string);
        }
        return string;
    }

    public static String extractAttributeFromX500Principal(ASN1ObjectIdentifier identifier, X500Principal x500PrincipalName) {
        RDN[] rdns;
        X500Name x500Name = X500Name.getInstance((Object)x500PrincipalName.getEncoded());
        for (RDN rdn : rdns = x500Name.getRDNs(identifier)) {
            if (rdn.isMultiValued()) {
                AttributeTypeAndValue[] typesAndValues;
                for (AttributeTypeAndValue typeAndValue : typesAndValues = rdn.getTypesAndValues()) {
                    if (!identifier.equals((Object)typeAndValue.getType())) continue;
                    return typeAndValue.getValue().toString();
                }
                continue;
            }
            AttributeTypeAndValue typeAndValue = rdn.getFirst();
            if (!identifier.equals((Object)typeAndValue.getType())) continue;
            return typeAndValue.getValue().toString();
        }
        return null;
    }

    public static String getSubjectCommonName(CertificateToken cert) {
        return DSSASN1Utils.extractAttributeFromX500Principal(BCStyle.CN, cert.getSubjectX500Principal());
    }

    public static String getHumanReadableName(CertificateToken cert) {
        return DSSASN1Utils.firstNotNull(cert, BCStyle.CN, BCStyle.GIVENNAME, BCStyle.SURNAME, BCStyle.NAME, BCStyle.PSEUDONYM, BCStyle.O, BCStyle.OU);
    }

    private static String firstNotNull(CertificateToken cert, ASN1ObjectIdentifier ... oids) {
        for (ASN1ObjectIdentifier oid : oids) {
            String value = DSSASN1Utils.extractAttributeFromX500Principal(oid, cert.getSubjectX500Principal());
            if (value == null) continue;
            return value;
        }
        return null;
    }

    public static SignerInformation getFirstSignerInformation(CMSSignedData cms) {
        Collection signers = cms.getSignerInfos().getSigners();
        if (signers.size() > 1) {
            LOG.warn("!!! The framework handles only one signer (SignerInformation) !!!");
        }
        return (SignerInformation)signers.iterator().next();
    }

    public static boolean isASN1SequenceTag(byte tagByte) {
        return 48 == tagByte;
    }

    public static Date getDate(ASN1Encodable encodable) {
        try {
            return Time.getInstance((Object)encodable).getDate();
        }
        catch (Exception e) {
            LOG.warn("Unable to retrieve the date : " + encodable, (Throwable)e);
            return null;
        }
    }

    static {
        Security.addProvider((Provider)securityProvider);
    }
}

