/**
 *  Copyright 2021 Peter Nerg
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.dmonix.prometheus

import io.prometheus.client.Gauge

import scala.concurrent.{ExecutionContext, Future}

/**
 * Enriches the ''Gauge'' class with new functions.
 * @since 1.0
 */
trait GaugeImplicits {
  /**
   * Class with extensions for the `Gauge` class
   * @param underlying The instance to enrich with functions
   * @since 1.0
   */
  implicit class GaugeDecorator(underlying:Gauge) {
    /**
     * Automatically increases/decreases the gauge around the provided function block.
     * {{{
     *   val gauge:Gauge = ...
     *   gauge.measure{
     *     //code goes here
     *   }
     * }}}
     * @param block The block of code to execute/measure
     * @tparam T
     * @return The result created from the provided function block
     */
    def measure[T](block: => T): T = Gauges.measure(underlying)(block)

    /**
     * Automatically increases/decreases the gauge around the future generated by the function block.
     * Increases before the future is created and decreases as a side effect of completing the future.
     * {{{
     *   val gauge:Gauge = ...
     *   val fut = gauge.measureAsync{
     *     Future {
     *       //code goes here
     *     }
     *   }
     * }}}
     * @param block The function returning a Future which to measure
     * @param ec
     * @tparam T
     * @return The future created from the provided function block
     */
    def measureAsync[T](block: => Future[T])(implicit ec:ExecutionContext): Future[T] = Gauges.measureAsync(underlying)(block)(ec)

    /**
     * Resets this gauge to '0'
     * @return itself
     */
    def reset():Gauge = Gauges.reset(underlying)
  }

  /**
   * Class with extensions for the `Gauge.Child` class
   *
   * @param underlying The instance to enrich with functions
   * @since 1.0
   */
  implicit class GaugeChildDecorator(underlying: Gauge.Child) {
    /**
     * Automatically increases/decreases the gauge around the provided function block.
     * {{{
     *   val gauge:Gauge.Child = ...
     *   gauge.measure{
     *     //code goes here
     *   }
     * }}}
     *
     * @param block The block of code to execute/measure
     * @tparam T
     * @return The result created from the provided function block
     */
    def measure[T](block: => T): T = Gauges.measure(underlying)(block)

    /**
     * Automatically increases/decreases the gauge around the future generated by the function block.
     * Increases before the future is created and decreases as a side effect of completing the future.
     * {{{
     *   val gauge:Gauge.Child = ...
     *   val fut = gauge.measureAsync{
     *     Future {
     *       //code goes here
     *     }
     *   }
     * }}}
     *
     * @param block The function returning a Future which to measure
     * @param ec
     * @tparam T
     * @return The future created from the provided function block
     */
    def measureAsync[T](block: => Future[T])(implicit ec: ExecutionContext): Future[T] = Gauges.measureAsync(underlying)(block)(ec)

    /**
     * Resets this gauge to '0'
     *
     * @return itself
     */
    def reset(): Gauge.Child = Gauges.reset(underlying)
  }
}