/**
 *  Copyright 2021 Peter Nerg
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.dmonix.prometheus

import io.prometheus.client.Gauge

import scala.concurrent.{ExecutionContext, Future}

/**
 * Holds functions related to 'Gauges'
 */
object Gauges {

  /**
   * Automatically increases/decreases the gauge around the provided function block.
   * {{{
   *   val gauge:Gauge = ...
   *   Gauges.measure(gauge){
   *     //code goes here
   *   }
   * }}}
   * @param gauge The gauge to increase/decrease
   * @param block The block of code to execute/measure
   * @tparam T
   * @return The result created from the provided function block
   */
  def measure[T](gauge:Gauge)(block: => T): T = Measurable.prePostFunc(gauge.inc(), gauge.dec())(block)

  /**
   * Automatically increases/decreases the gauge around the provided function block.
   * {{{
   *   val gauge:Gauge.Child = ...
   *   Gauges.measure(gauge){
   *     //code goes here
   *   }
   * }}}
   * @param gauge The gauge to increase/decrease
   * @param block The block of code to execute/measure
   * @tparam T
   * @return The result created from the provided function block
   */
  def measure[T](gauge:Gauge.Child)(block: => T): T = Measurable.prePostFunc(gauge.inc(), gauge.dec())(block)

  /**
   * Automatically increases/decreases the gauge around the future generated by the function block.
   * Increases before the future is created and decreases as a side effect of completing the future.
   * {{{
   *   val gauge:Gauge = ...
   *   val fut = Gauges.measureAsync(gauge){
   *     Future {
   *       //code goes here
   *     }
   *   }
   * }}}
   * @param gauge The gauge to increase/decrease
   * @param block The function returning a Future which to measure
   * @param ec
   * @tparam T
   * @return The future created from the provided function block
   */
  def measureAsync[T](gauge:Gauge)(block: => Future[T])(implicit ec:ExecutionContext): Future[T] = Measurable.prePostFuncAsync(gauge.inc, gauge.dec)(block)(ec)

  /**
   * Automatically increases/decreases the gauge around the future generated by the function block.
   * Increases before the future is created and decreases as a side effect of completing the future.
   * {{{
   *   val gauge:Gauge.Child = ...
   *   val fut = Gauges.measureAsync(gauge){
   *     Future {
   *       //code goes here
   *     }
   *   }
   * }}}
   * @param gauge The gauge to increase/decrease
   * @param block The function returning a Future which to measure
   * @param ec
   * @tparam T
   * @return The future created from the provided function block
   */
  def measureAsync[T](gauge:Gauge.Child)(block: => Future[T])(implicit ec:ExecutionContext): Future[T] = Measurable.prePostFuncAsync(gauge.inc, gauge.dec)(block)(ec)

  /**
   * Resets the gauge to '0'
   * @param gauge The gauge to reset
   * @return The gauge
   */
  def reset(gauge:Gauge):Gauge = {
    gauge.set(0)
    gauge
  }

  /**
   * Resets the gauge to '0'
   * @param gauge The gauge to reset
   * @return The gauge
   */
  def reset(gauge:Gauge.Child):Gauge.Child = {
    gauge.set(0)
    gauge
  }

}
