/*
 * Decompiled with CFR 0.152.
 */
package org.drasyl.handler.connection;

import io.netty.buffer.ByteBuf;
import io.netty.channel.ChannelDuplexHandler;
import io.netty.channel.ChannelFuture;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelPromise;
import io.netty.handler.codec.UnsupportedMessageTypeException;
import io.netty.util.ReferenceCountUtil;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.ScheduledFuture;
import java.nio.channels.ClosedChannelException;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.LongSupplier;
import java.util.function.Supplier;
import org.drasyl.handler.connection.ConnectionHandshakeClosing;
import org.drasyl.handler.connection.ConnectionHandshakeCompleted;
import org.drasyl.handler.connection.ConnectionHandshakeException;
import org.drasyl.handler.connection.ConnectionHandshakeIssued;
import org.drasyl.handler.connection.ConnectionHandshakeSegment;
import org.drasyl.handler.connection.State;
import org.drasyl.util.Preconditions;
import org.drasyl.util.RandomUtil;
import org.drasyl.util.SerialNumberArithmetic;
import org.drasyl.util.logging.Logger;
import org.drasyl.util.logging.LoggerFactory;

public class ConnectionHandshakeHandler
extends ChannelDuplexHandler {
    private static final Logger LOG = LoggerFactory.getLogger(ConnectionHandshakeHandler.class);
    private static final ConnectionHandshakeIssued HANDSHAKE_ISSUED_EVENT = new ConnectionHandshakeIssued();
    private static final ConnectionHandshakeClosing HANDSHAKE_CLOSING_EVENT = new ConnectionHandshakeClosing();
    private static final ConnectionHandshakeException CONNECTION_CLOSING_ERROR = new ConnectionHandshakeException("Connection closing");
    private static final ConnectionHandshakeException CONNECTION_RESET_EXCEPTION = new ConnectionHandshakeException("Connection reset");
    private static final int SEQ_NO_SPACE = 32;
    private final Duration userTimeout;
    private final LongSupplier issProvider;
    private final boolean activeOpen;
    protected ScheduledFuture<?> userTimeoutFuture;
    private ChannelPromise userCallFuture;
    State state;
    long sndUna;
    long sndNxt;
    long iss;
    long rcvNxt;
    long irs;

    ConnectionHandshakeHandler(Duration userTimeout, LongSupplier issProvider, boolean activeOpen, State state, int sndUna, int sndNxt, int rcvNxt) {
        this.userTimeout = Preconditions.requireNonNegative((Duration)userTimeout);
        this.issProvider = Objects.requireNonNull(issProvider);
        this.activeOpen = activeOpen;
        this.state = Objects.requireNonNull(state);
        this.sndUna = sndUna;
        this.sndNxt = sndNxt;
        this.rcvNxt = rcvNxt;
    }

    public ConnectionHandshakeHandler(Duration userTimeout, boolean activeOpen) {
        this(userTimeout, () -> RandomUtil.randomInt((int)0x7FFFFFFE), activeOpen, State.CLOSED, 0, 0, 0);
    }

    public void close(ChannelHandlerContext ctx, ChannelPromise promise) throws Exception {
        this.userCallClose(ctx, promise);
    }

    public void write(ChannelHandlerContext ctx, Object msg, ChannelPromise promise) {
        if (!(msg instanceof ByteBuf)) {
            UnsupportedMessageTypeException exception = new UnsupportedMessageTypeException(msg, new Class[]{ByteBuf.class});
            ReferenceCountUtil.release((Object)msg);
            promise.setFailure((Throwable)exception);
        } else {
            this.userCallSend(ctx, (ByteBuf)msg, promise);
        }
    }

    public void channelActive(ChannelHandlerContext ctx) {
        if (this.state == State.CLOSED) {
            if (this.activeOpen) {
                LOG.trace("{}[{}] Handler is configured to perform active OPEN process.", (Object)ctx.channel(), (Object)this.state);
                this.userCallOpen(ctx, ctx.newPromise());
            } else {
                LOG.trace("{}[{}] Handler is configured to perform passive OPEN process. Wait for remote peer to initiate OPEN process.", (Object)ctx.channel(), (Object)this.state);
                this.switchToNewState(ctx, State.LISTEN);
            }
        }
        ctx.fireChannelActive();
    }

    private void switchToNewState(ChannelHandlerContext ctx, State newState) {
        LOG.trace("{}[{} -> {}] Switched to new state.", new Object[]{ctx.channel(), this.state, newState});
        this.state = newState;
    }

    public void channelInactive(ChannelHandlerContext ctx) {
        this.cancelTimeoutGuards();
        ctx.fireChannelInactive();
    }

    public void channelRead(ChannelHandlerContext ctx, Object msg) {
        if (msg instanceof ConnectionHandshakeSegment) {
            this.segmentArrives(ctx, (ConnectionHandshakeSegment)((Object)msg));
        } else {
            ctx.fireChannelRead(msg);
        }
    }

    private void userCallOpen(ChannelHandlerContext ctx, ChannelPromise promise) {
        LOG.trace("{}[{}] OPEN call received.", (Object)ctx.channel(), (Object)this.state);
        this.userCallFuture = promise;
        promise.addListener((GenericFutureListener)((ChannelFutureListener)future -> {
            if (!future.isSuccess()) {
                ctx.fireExceptionCaught(future.cause());
            }
        }));
        this.performActiveOpen(ctx);
    }

    private void userCallSend(ChannelHandlerContext ctx, ByteBuf data, ChannelPromise promise) {
        switch (this.state) {
            case CLOSED: {
                data.release();
                promise.setFailure((Throwable)new ConnectionHandshakeException("Connection does not exist"));
                break;
            }
            case LISTEN: {
                LOG.trace("{}[{}] Write was performed while we're in passive OPEN mode. Switch to active OPEN mode, enqueue write operation, and initiate OPEN process.", (Object)ctx.channel(), (Object)this.state);
                this.userCallFuture = ctx.newPromise();
                this.userCallFuture.addListener((GenericFutureListener)((ChannelFutureListener)future -> {
                    if (future.isSuccess()) {
                        this.userCallSend(ctx, data, promise);
                    } else {
                        promise.setFailure(future.cause());
                    }
                }));
                this.performActiveOpen(ctx);
                break;
            }
            case SYN_SENT: 
            case SYN_RECEIVED: {
                data.release();
                promise.setFailure((Throwable)new ConnectionHandshakeException("Handshake in progress"));
                break;
            }
            case ESTABLISHED: {
                ConnectionHandshakeSegment seg = ConnectionHandshakeSegment.pshAck(this.sndNxt, this.rcvNxt, data);
                LOG.trace("{}[{}] As connection is established, we can pass the message `{}` to the network.", new Object[]{ctx.channel(), this.state, seg});
                ctx.write((Object)seg, promise);
                break;
            }
            default: {
                LOG.trace("{}[{}] Channel is in process of being closed. Drop write `{}`.", new Object[]{ctx.channel(), this.state, data});
                data.release();
                promise.setFailure((Throwable)CONNECTION_CLOSING_ERROR);
            }
        }
    }

    private void userCallClose(ChannelHandlerContext ctx, ChannelPromise promise) {
        LOG.trace("{}[{}] CLOSE call received.", (Object)ctx.channel(), (Object)this.state);
        switch (this.state) {
            case CLOSED: {
                LOG.trace("{}[{}] Channel is already closed. Pass close call further through the pipeline.", (Object)ctx.channel(), (Object)this.state);
                ctx.close(promise);
                break;
            }
            case LISTEN: 
            case SYN_SENT: {
                if (this.userCallFuture != null) {
                    this.userCallFuture.setFailure((Throwable)CONNECTION_CLOSING_ERROR);
                }
                this.switchToNewState(ctx, State.CLOSED);
                ctx.close(promise);
                break;
            }
            case SYN_RECEIVED: {
                if (this.userCallFuture != null) {
                    this.userCallFuture.setFailure((Throwable)CONNECTION_CLOSING_ERROR);
                }
            }
            case ESTABLISHED: {
                this.userCallFuture = promise;
                ctx.fireUserEventTriggered((Object)HANDSHAKE_CLOSING_EVENT);
                ConnectionHandshakeSegment seg = ConnectionHandshakeSegment.finAck(this.sndNxt, this.rcvNxt);
                ++this.sndNxt;
                LOG.trace("{}[{}] Initiate CLOSE sequence by sending `{}`.", new Object[]{ctx.channel(), this.state, seg});
                ctx.writeAndFlush((Object)seg).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, seg));
                this.switchToNewState(ctx, State.FIN_WAIT_1);
                this.applyUserTimeout(ctx, "CLOSE", promise);
                break;
            }
            default: {
                this.userCallFuture.addListener((GenericFutureListener)((ChannelFutureListener)future -> {
                    if (future.isSuccess()) {
                        promise.setSuccess();
                    } else {
                        promise.setFailure(future.cause());
                    }
                }));
            }
        }
    }

    private void applyUserTimeout(ChannelHandlerContext ctx, String userCall, ChannelPromise promise) {
        if (this.userTimeout.toMillis() > 0L) {
            if (this.userTimeoutFuture != null) {
                this.userTimeoutFuture.cancel(false);
            }
            this.userTimeoutFuture = ctx.executor().schedule(() -> {
                LOG.trace("{}[{}] User timeout for {} user call expired after {}ms. Close channel.", new Object[]{ctx.channel(), this.state, userCall, this.userTimeout});
                this.switchToNewState(ctx, State.CLOSED);
                promise.tryFailure((Throwable)new ConnectionHandshakeException("User timeout for " + userCall + " user call after " + this.userTimeout + "ms. Close channel."));
                ctx.channel().close();
            }, this.userTimeout.toMillis(), TimeUnit.MILLISECONDS);
        }
    }

    private void cancelTimeoutGuards() {
        if (this.userTimeoutFuture != null) {
            this.userTimeoutFuture.cancel(false);
        }
    }

    private void performActiveOpen(ChannelHandlerContext ctx) {
        this.sndUna = this.iss = this.issProvider.getAsLong();
        this.sndNxt = SerialNumberArithmetic.add((long)this.iss, (long)1L, (int)32);
        ConnectionHandshakeSegment seg = ConnectionHandshakeSegment.syn(this.iss);
        LOG.trace("{}[{}] Initiate OPEN process by sending `{}`.", new Object[]{ctx.channel(), this.state, seg});
        ctx.writeAndFlush((Object)seg).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, seg));
        this.switchToNewState(ctx, State.SYN_SENT);
        this.applyUserTimeout(ctx, "OPEN", this.userCallFuture);
        ctx.fireUserEventTriggered((Object)HANDSHAKE_ISSUED_EVENT);
    }

    private void segmentArrives(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
        LOG.trace("{}[{}] Read `{}`.", new Object[]{ctx.channel(), this.state, seg});
        switch (this.state) {
            case CLOSED: {
                this.segmentArrivesOnClosedState(ctx, seg);
                break;
            }
            case LISTEN: {
                this.segmentArrivesOnListenState(ctx, seg);
                break;
            }
            case SYN_SENT: {
                this.segmentArrivesOnSynSentState(ctx, seg);
                break;
            }
            default: {
                this.segmentArrivesOnOtherStates(ctx, seg);
            }
        }
    }

    private void segmentArrivesOnClosedState(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
        if (seg.isRst()) {
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        ConnectionHandshakeSegment response = seg.isAck() ? ConnectionHandshakeSegment.rst(seg.ack()) : ConnectionHandshakeSegment.rstAck(0L, seg.seq());
        LOG.trace("{}[{}] As we're already on CLOSED state, this channel is going to be removed soon. Reset remote peer `{}`.", new Object[]{ctx.channel(), this.state, response});
        ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, response));
        ReferenceCountUtil.release((Object)((Object)seg));
    }

    private void segmentArrivesOnListenState(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
        if (seg.isRst()) {
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        if (seg.isAck()) {
            ConnectionHandshakeSegment response = ConnectionHandshakeSegment.rst(seg.ack());
            LOG.trace("{}[{}] We are on a state were we have never sent ansythink that must be ACKnowledged. Send RST `{}`.", new Object[]{ctx.channel(), this.state, response});
            ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, response));
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        if (seg.isSyn()) {
            LOG.trace("{}[{}] Remote peer initiates handshake by sending a SYN `{}` to us.", new Object[]{ctx.channel(), this.state, seg});
            if (this.userTimeout.toMillis() > 0L) {
                ctx.executor().schedule(() -> {
                    if (this.state != State.ESTABLISHED && this.state != State.CLOSED) {
                        LOG.trace("{}[{}] Handshake initiated by remote port has not been completed within {}ms. Abort handshake, close channel.", new Object[]{ctx.channel(), this.state, this.userTimeout});
                        this.switchToNewState(ctx, State.CLOSED);
                        ctx.channel().close();
                    }
                }, this.userTimeout.toMillis(), TimeUnit.MILLISECONDS);
            }
            this.switchToNewState(ctx, State.SYN_RECEIVED);
            this.rcvNxt = SerialNumberArithmetic.add((long)seg.seq(), (long)1L, (int)32);
            this.irs = seg.seq();
            this.sndUna = this.iss = this.issProvider.getAsLong();
            this.sndNxt = SerialNumberArithmetic.add((long)this.iss, (long)1L, (int)32);
            ConnectionHandshakeSegment response = ConnectionHandshakeSegment.synAck(this.iss, this.rcvNxt);
            LOG.trace("{}[{}] ACKnowlede the received segment and send our SYN `{}`.", new Object[]{ctx.channel(), this.state, response});
            ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, response));
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        this.unexpectedSegment(ctx, seg);
    }

    private void segmentArrivesOnSynSentState(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
        if (seg.isAck() && (SerialNumberArithmetic.lessThanOrEqualTo((long)seg.ack(), (long)this.iss, (int)32) || SerialNumberArithmetic.greaterThan((long)seg.ack(), (long)this.sndNxt, (int)32))) {
            LOG.trace("{}[{}] Get got an ACKnowledgement `{}` for an Segment we never sent. Seems like remote peer is synchronized to another connection.", new Object[]{ctx.channel(), this.state, seg});
            if (seg.isRst()) {
                LOG.trace("{}[{}] As the RST bit is set. It doesn't matter as we will reset or connection now.", (Object)ctx.channel(), (Object)this.state);
            } else {
                ConnectionHandshakeSegment response = ConnectionHandshakeSegment.rst(seg.ack());
                LOG.trace("{}[{}] Inform remote peer about the desynchronization state by sending an `{}` and dropping the inbound Segment.", new Object[]{ctx.channel(), this.state, response});
                ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, response));
            }
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        boolean acceptableAck = this.isAcceptableAck(seg);
        if (seg.isRst()) {
            if (acceptableAck) {
                LOG.trace("{}[{}] Segment `{}` is an acceptable ACKnowledgement. Inform user, drop segment, enter CLOSED state.", new Object[]{ctx.channel(), this.state, seg});
                ReferenceCountUtil.release((Object)((Object)seg));
                this.switchToNewState(ctx, State.CLOSED);
                ctx.fireExceptionCaught((Throwable)CONNECTION_RESET_EXCEPTION);
                ctx.channel().close();
            } else {
                LOG.trace("{}[{}] Segment `{}` is not an acceptable ACKnowledgement. Drop it.", new Object[]{ctx.channel(), this.state, seg});
                ReferenceCountUtil.release((Object)((Object)seg));
            }
            return;
        }
        if (seg.isSyn()) {
            boolean ourSynHasBeenAcked;
            this.rcvNxt = SerialNumberArithmetic.add((long)seg.seq(), (long)1L, (int)32);
            this.irs = seg.seq();
            if (seg.isAck()) {
                this.sndUna = seg.ack();
            }
            if (ourSynHasBeenAcked = SerialNumberArithmetic.greaterThan((long)this.sndUna, (long)this.iss, (int)32)) {
                LOG.trace("{}[{}] Remote peer has ACKed our SYN package and sent us his SYN `{}`. Handshake on our side is completed.", new Object[]{ctx.channel(), this.state, seg});
                this.cancelTimeoutGuards();
                this.switchToNewState(ctx, State.ESTABLISHED);
                this.userCallFuture.setSuccess();
                this.userCallFuture = null;
                ConnectionHandshakeSegment response = ConnectionHandshakeSegment.ack(this.sndNxt, this.rcvNxt);
                LOG.trace("{}[{}] ACKnowlede the received segment with a `{}` so the remote peer can complete the handshake as well.", new Object[]{ctx.channel(), this.state, response});
                ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
                ReferenceCountUtil.release((Object)((Object)seg));
                ctx.fireUserEventTriggered((Object)new ConnectionHandshakeCompleted(this.sndNxt, this.rcvNxt));
            } else {
                this.switchToNewState(ctx, State.SYN_RECEIVED);
                ConnectionHandshakeSegment response = ConnectionHandshakeSegment.synAck(this.iss, this.rcvNxt);
                LOG.trace("{}[{}] Write `{}`.", new Object[]{ctx.channel(), this.state, response});
                ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, response));
                ReferenceCountUtil.release((Object)((Object)seg));
            }
            return;
        }
        ReferenceCountUtil.release((Object)((Object)seg));
    }

    private void segmentArrivesOnOtherStates(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
        ConnectionHandshakeSegment response;
        boolean validSeg = seg.seq() == this.rcvNxt;
        boolean acceptableAck = this.isAcceptableAck(seg);
        if (!validSeg && !acceptableAck) {
            if (!seg.isRst()) {
                ConnectionHandshakeSegment response2 = ConnectionHandshakeSegment.ack(this.sndNxt, this.rcvNxt);
                LOG.trace("{}[{}] We got an unexpected Segment `{}`. Send an ACKnowledgement `{}` for the Segment we actually expect.", new Object[]{ctx.channel(), this.state, seg, response2});
                ctx.writeAndFlush((Object)response2).addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
            }
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        if (seg.isRst()) {
            switch (this.state) {
                case SYN_RECEIVED: {
                    if (this.activeOpen) {
                        LOG.trace("{}[{}] We got `{}`. Connection has been refused by remote peer.", new Object[]{ctx.channel(), this.state, seg});
                        this.switchToNewState(ctx, State.CLOSED);
                        ReferenceCountUtil.release((Object)((Object)seg));
                        ctx.fireExceptionCaught((Throwable)new ConnectionHandshakeException("Connection refused"));
                        ctx.channel().close();
                        return;
                    }
                    LOG.trace("{}[{}] We got `{}`. Remote peer is not longer interested in a connection. We're going back to the LISTEN state.", new Object[]{ctx.channel(), this.state, seg});
                    this.switchToNewState(ctx, State.LISTEN);
                    ReferenceCountUtil.release((Object)((Object)seg));
                    return;
                }
                case ESTABLISHED: 
                case FIN_WAIT_1: 
                case FIN_WAIT_2: {
                    LOG.trace("{}[{}] We got `{}`. Remote peer is not longer interested in a connection. Close channel.", new Object[]{ctx.channel(), this.state, seg});
                    this.switchToNewState(ctx, State.CLOSED);
                    ReferenceCountUtil.release((Object)((Object)seg));
                    ctx.fireExceptionCaught((Throwable)CONNECTION_RESET_EXCEPTION);
                    ctx.channel().close();
                    return;
                }
            }
            LOG.trace("{}[{}] We got `{}`. Close channel.", new Object[]{ctx.channel(), this.state, seg});
            this.switchToNewState(ctx, State.CLOSED);
            ctx.channel().close();
            ReferenceCountUtil.release((Object)((Object)seg));
            return;
        }
        if (seg.isAck()) {
            switch (this.state) {
                case SYN_RECEIVED: {
                    if (!SerialNumberArithmetic.lessThanOrEqualTo((long)this.sndUna, (long)seg.ack(), (int)32) || !SerialNumberArithmetic.lessThanOrEqualTo((long)seg.ack(), (long)this.sndNxt, (int)32)) break;
                    LOG.trace("{}[{}] Remote peer ACKnowledge `{}` receivable of our SYN. As we've already received his SYN the handshake is now completed on both sides.", new Object[]{ctx.channel(), this.state, seg});
                    this.cancelTimeoutGuards();
                    this.switchToNewState(ctx, State.ESTABLISHED);
                    ctx.fireUserEventTriggered((Object)new ConnectionHandshakeCompleted(this.sndNxt, this.rcvNxt));
                    if (!acceptableAck) {
                        ConnectionHandshakeSegment response3 = ConnectionHandshakeSegment.rst(seg.ack());
                        LOG.trace("{}[{}] Segment `{}` is not an acceptable ACKnowledgement. Send RST `{}` and drop received Segment.", new Object[]{ctx.channel(), this.state, seg, response3});
                        ReferenceCountUtil.release((Object)((Object)seg));
                        ctx.writeAndFlush((Object)response3).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, response3));
                        return;
                    }
                    this.sndUna = seg.ack();
                    break;
                }
                case ESTABLISHED: {
                    break;
                }
                case FIN_WAIT_1: {
                    if (this.establishedProcessing(ctx, seg, acceptableAck)) {
                        return;
                    }
                    if (!acceptableAck) break;
                    this.switchToNewState(ctx, State.FIN_WAIT_2);
                    break;
                }
                case FIN_WAIT_2: {
                    if (!this.establishedProcessing(ctx, seg, acceptableAck)) break;
                    return;
                }
                case CLOSING: {
                    if (this.establishedProcessing(ctx, seg, acceptableAck)) {
                        return;
                    }
                    if (acceptableAck) {
                        LOG.trace("{}[{}] Our sent FIN has been ACKnowledged by `{}`. Close sequence done.", new Object[]{ctx.channel(), this.state, seg});
                        this.switchToNewState(ctx, State.CLOSED);
                        ctx.close(this.userCallFuture);
                        break;
                    }
                    LOG.trace("{}[{}] The received ACKnowledged `{}` does not match our sent FIN. Ignore it.", new Object[]{ctx.channel(), this.state, seg});
                    ReferenceCountUtil.release((Object)((Object)seg));
                    return;
                }
                case LAST_ACK: {
                    LOG.trace("{}[{}] Our sent FIN has been ACKnowledged by `{}`. Close sequence done.", new Object[]{ctx.channel(), this.state, seg});
                    this.switchToNewState(ctx, State.CLOSED);
                    if (this.userCallFuture != null) {
                        ctx.close(this.userCallFuture);
                    } else {
                        ctx.channel().close();
                    }
                    ReferenceCountUtil.release((Object)((Object)seg));
                    return;
                }
                default: {
                    response = ConnectionHandshakeSegment.ack(this.sndNxt, this.rcvNxt);
                    LOG.trace("{}[{}] Write `{}`.", new Object[]{ctx.channel(), this.state, response});
                    ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
                }
            }
        }
        if (seg.content().isReadable()) {
            ctx.fireChannelRead((Object)seg.content());
        } else if (!seg.isFin()) {
            seg.release();
        }
        if (seg.isFin()) {
            if (this.state == State.CLOSED || this.state == State.LISTEN || this.state == State.SYN_SENT) {
                ReferenceCountUtil.release((Object)((Object)seg));
                return;
            }
            this.rcvNxt = SerialNumberArithmetic.add((long)seg.seq(), (long)1L, (int)32);
            response = ConnectionHandshakeSegment.ack(this.sndNxt, this.rcvNxt);
            LOG.trace("{}[{}] Got CLOSE request `{}` from remote peer. ACKnowledge receival with `{}`.", new Object[]{ctx.channel(), this.state, seg, response});
            ChannelFuture ackFuture = ctx.writeAndFlush((Object)response);
            ackFuture.addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
            switch (this.state) {
                case SYN_RECEIVED: 
                case ESTABLISHED: {
                    ctx.fireUserEventTriggered((Object)HANDSHAKE_CLOSING_EVENT);
                    LOG.trace("{}[{}] This channel is going to close now. Trigger channel close.", (Object)ctx.channel(), (Object)this.state);
                    ConnectionHandshakeSegment seg2 = ConnectionHandshakeSegment.finAck(this.sndNxt, this.rcvNxt);
                    ++this.sndNxt;
                    LOG.trace("{}[{}] As we're already waiting for this. We're sending our last Segment `{}` and start waiting for the final ACKnowledgment.", new Object[]{ctx.channel(), this.state, seg2});
                    ctx.writeAndFlush((Object)seg2).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(ctx, seg2));
                    this.switchToNewState(ctx, State.LAST_ACK);
                    ReferenceCountUtil.release((Object)((Object)seg));
                    break;
                }
                case FIN_WAIT_1: {
                    if (acceptableAck) {
                        LOG.trace("{}[{}] Our FIN has been ACKnowledged. Close channel.", new Object[]{ctx.channel(), this.state, seg});
                        this.switchToNewState(ctx, State.CLOSED);
                    } else {
                        this.switchToNewState(ctx, State.CLOSING);
                    }
                    ReferenceCountUtil.release((Object)((Object)seg));
                    break;
                }
                case FIN_WAIT_2: {
                    LOG.trace("{}[{}] Wait for our ACKnowledgment `{}` to be written to the network. Then close the channel.", new Object[]{ctx.channel(), this.state, response});
                    this.switchToNewState(ctx, State.CLOSED);
                    ackFuture.addListener((GenericFutureListener)((ChannelFutureListener)future -> {
                        if (future.isSuccess()) {
                            LOG.trace("{}[{}] Our ACKnowledgment `{}` was written to the network. Close channel!", new Object[]{ctx.channel(), this.state, response});
                            this.userCallFuture.setSuccess();
                        } else {
                            LOG.trace("{}[{}] Failed to write our ACKnowledgment `{}` to the network: {}", new Object[]{ctx.channel(), this.state, response, future.cause()});
                            this.userCallFuture.setFailure(future.cause());
                        }
                        this.userCallFuture = null;
                        future.channel().close();
                    }));
                    ReferenceCountUtil.release((Object)((Object)seg));
                    break;
                }
            }
        }
    }

    private boolean establishedProcessing(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg, boolean acceptableAck) {
        if (acceptableAck) {
            this.sndUna = seg.ack();
        }
        if (SerialNumberArithmetic.lessThan((long)seg.ack(), (long)this.sndUna, (int)32)) {
            return true;
        }
        if (SerialNumberArithmetic.greaterThan((long)seg.ack(), (long)this.sndUna, (int)32)) {
            ConnectionHandshakeSegment response = ConnectionHandshakeSegment.ack(this.sndNxt, this.rcvNxt);
            LOG.trace("{}[{}] Write `{}`.", new Object[]{ctx.channel(), this.state, response});
            ctx.writeAndFlush((Object)response).addListener((GenericFutureListener)ChannelFutureListener.CLOSE_ON_FAILURE);
            return true;
        }
        return false;
    }

    private boolean isAcceptableAck(ConnectionHandshakeSegment seg) {
        return seg.isAck() && SerialNumberArithmetic.lessThan((long)this.sndUna, (long)seg.ack(), (int)32) && SerialNumberArithmetic.lessThanOrEqualTo((long)seg.ack(), (long)this.sndNxt, (int)32);
    }

    private void unexpectedSegment(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
        ReferenceCountUtil.release((Object)((Object)seg));
        LOG.error("{}[{}] Got unexpected segment `{}`.", new Object[]{ctx.channel(), this.state, seg});
    }

    private class RetransmissionTimeoutApplier
    implements ChannelFutureListener {
        private static final long LOWER_BOUND = 100L;
        private static final long UPPER_BOUND = 60000L;
        private static final int RTT = 20;
        private static final float ALPHA = 0.9f;
        private static final float BETA = 1.7f;
        private final ChannelHandlerContext ctx;
        private final ConnectionHandshakeSegment seg;
        private final long srtt;

        RetransmissionTimeoutApplier(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg, long srtt) {
            this.ctx = Objects.requireNonNull(ctx);
            this.seg = Objects.requireNonNull(seg);
            this.srtt = Preconditions.requirePositive((long)srtt);
        }

        RetransmissionTimeoutApplier(ChannelHandlerContext ctx, ConnectionHandshakeSegment seg) {
            this(ctx, seg, 20L);
        }

        public void operationComplete(ChannelFuture future) {
            if (future.isSuccess()) {
                if (!this.seg.isOnlyAck() && !this.seg.isRst()) {
                    long newSrtt = (long)(0.9f * (float)this.srtt + 2.0000005f);
                    long rto = Math.min(60000L, Math.max(100L, (long)(1.7f * (float)newSrtt)));
                    this.ctx.executor().schedule(() -> {
                        if (future.channel().isOpen() && ConnectionHandshakeHandler.this.state != State.CLOSED && SerialNumberArithmetic.lessThanOrEqualTo((long)ConnectionHandshakeHandler.this.sndUna, (long)this.seg.seq(), (int)32)) {
                            LOG.trace("{}[{}] Segment `{}` has not been acknowledged within {}ms. Send again.", new Object[]{future.channel(), ConnectionHandshakeHandler.this.state, this.seg, rto});
                            this.ctx.writeAndFlush((Object)this.seg).addListener((GenericFutureListener)new RetransmissionTimeoutApplier(this.ctx, this.seg, rto));
                        }
                    }, rto, TimeUnit.MILLISECONDS);
                }
            } else if (!(future.cause() instanceof ClosedChannelException)) {
                Supplier[] supplierArray = new Supplier[4];
                supplierArray[0] = () -> ((ChannelHandlerContext)this.ctx).channel();
                supplierArray[1] = () -> ConnectionHandshakeHandler.this.state;
                supplierArray[2] = () -> this.seg;
                supplierArray[3] = () -> ((ChannelFuture)future).cause();
                LOG.trace("{}[{}] Unable to send `{}`:", supplierArray);
                future.channel().close();
            }
        }
    }
}

