/*
 * Decompiled with CFR 0.152.
 */
package jcuda;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;

public final class LibUtils {
    private static final Logger logger = Logger.getLogger(LibUtils.class.getName());
    private static final Level level;
    private static final String LIBRARY_PATH_IN_JAR = "/lib";

    private LibUtils() {
    }

    public static void loadLibrary(String libraryName, String ... dependentLibraryNames) {
        logger.log(level, "Loading library: " + libraryName);
        try {
            logger.log(level, "Loading library as a file");
            System.loadLibrary(libraryName);
            logger.log(level, "Loading library as a file DONE");
            return;
        }
        catch (Throwable t) {
            logger.log(level, "Loading library as a file FAILED");
            Throwable throwableFromFile = t;
            try {
                logger.log(level, "Loading library as a resource");
                LibUtils.loadLibraryResource(LIBRARY_PATH_IN_JAR, libraryName, "", dependentLibraryNames);
                logger.log(level, "Loading library as a resource DONE");
                return;
            }
            catch (Throwable throwableFromResource) {
                logger.log(level, "Loading library as a resource FAILED", throwableFromResource);
                StringWriter sw = new StringWriter();
                PrintWriter pw = new PrintWriter(sw);
                pw.println("Error while loading native library \"" + libraryName + "\"");
                pw.println("Operating system name: " + System.getProperty("os.name"));
                pw.println("Architecture         : " + System.getProperty("os.arch"));
                pw.println("Architecture bit size: " + System.getProperty("sun.arch.data.model"));
                pw.println("---(start of nested stack traces)---");
                pw.println("Stack trace from the attempt to load the library as a file:");
                throwableFromFile.printStackTrace(pw);
                pw.println("Stack trace from the attempt to load the library as a resource:");
                throwableFromResource.printStackTrace(pw);
                pw.println("---(end of nested stack traces)---");
                pw.close();
                throw new UnsatisfiedLinkError(sw.toString());
            }
        }
    }

    private static void loadLibraryResource(String resourceSubdirectoryName, String libraryName, String tempSubdirectoryName, String ... dependentLibraryNames) throws Throwable {
        for (String dependentLibraryName : dependentLibraryNames) {
            logger.log(level, "Library " + libraryName + " depends on " + dependentLibraryName);
            String dependentResourceSubdirectoryName = resourceSubdirectoryName + "/" + LibUtils.osString() + "/" + LibUtils.archString();
            String dependentLibraryTempSubDirectoryName = libraryName + "_dependents" + File.separator + LibUtils.osString() + File.separator + LibUtils.archString() + File.separator;
            LibUtils.loadLibraryResource(dependentResourceSubdirectoryName, dependentLibraryName, dependentLibraryTempSubDirectoryName, new String[0]);
        }
        String libraryFileName = LibUtils.createLibraryFileName(libraryName);
        File libraryTempFile = LibUtils.createTempFile(tempSubdirectoryName, libraryFileName);
        if (!libraryTempFile.exists()) {
            String libraryResourceName = resourceSubdirectoryName + "/" + libraryFileName;
            logger.log(level, "Writing resource  " + libraryResourceName);
            logger.log(level, "to temporary file " + libraryTempFile);
            LibUtils.writeResourceToFile(libraryResourceName, libraryTempFile);
        }
        logger.log(level, "Loading library " + libraryTempFile);
        System.load(libraryTempFile.toString());
        logger.log(level, "Loading library " + libraryTempFile + " DONE");
    }

    private static File createTempFile(String tempSubdirectoryName, String name) throws IOException {
        boolean createdDirectory;
        String tempDirName = System.getProperty("java.io.tmpdir");
        File tempSubDirectory = new File(tempDirName + File.separator + tempSubdirectoryName);
        if (!tempSubDirectory.exists() && !(createdDirectory = tempSubDirectory.mkdirs())) {
            throw new IOException("Could not create directory for temporary file: " + tempSubDirectory);
        }
        String tempFileName = tempSubDirectory + File.separator + name;
        File tempFile = new File(tempFileName);
        return tempFile;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void writeResourceToFile(String resourceName, File file) throws IOException {
        if (file == null) {
            throw new NullPointerException("Target file may not be null");
        }
        if (file.exists()) {
            throw new IllegalArgumentException("Target file already exists: " + file);
        }
        InputStream inputStream = LibUtils.class.getResourceAsStream(resourceName);
        if (inputStream == null) {
            throw new IOException("No resource found with name '" + resourceName + "'");
        }
        FileOutputStream outputStream = null;
        try {
            int read;
            outputStream = new FileOutputStream(file);
            byte[] buffer = new byte[32768];
            while ((read = inputStream.read(buffer)) >= 0) {
                ((OutputStream)outputStream).write(buffer, 0, read);
            }
            outputStream.flush();
        }
        finally {
            if (outputStream != null) {
                try {
                    ((OutputStream)outputStream).close();
                }
                catch (IOException e) {
                    logger.log(Level.SEVERE, e.getMessage(), e);
                }
            }
            try {
                inputStream.close();
            }
            catch (IOException e) {
                logger.log(Level.SEVERE, e.getMessage(), e);
            }
        }
    }

    public static String createLibraryFileName(String libraryName) {
        String libPrefix = LibUtils.createLibraryPrefix();
        String libExtension = LibUtils.createLibraryExtension();
        String fullName = libPrefix + libraryName + "." + libExtension;
        return fullName;
    }

    private static String createLibraryExtension() {
        OSType osType = LibUtils.calculateOS();
        switch (osType) {
            case APPLE: {
                return "dylib";
            }
            case ANDROID: 
            case LINUX: 
            case SUN: {
                return "so";
            }
            case WINDOWS: {
                return "dll";
            }
        }
        return "";
    }

    private static String createLibraryPrefix() {
        OSType osType = LibUtils.calculateOS();
        switch (osType) {
            case APPLE: 
            case ANDROID: 
            case LINUX: 
            case SUN: {
                return "lib";
            }
            case WINDOWS: {
                return "";
            }
        }
        return "";
    }

    public static String createPlatformLibraryName(String baseName) {
        return baseName + "-" + LibUtils.osString() + "-" + LibUtils.archString();
    }

    private static String osString() {
        OSType osType = LibUtils.calculateOS();
        return osType.toString().toLowerCase(Locale.ENGLISH);
    }

    private static String archString() {
        ArchType archType = LibUtils.calculateArch();
        return archType.toString().toLowerCase(Locale.ENGLISH);
    }

    public static OSType calculateOS() {
        String vendor = System.getProperty("java.vendor");
        if ("The Android Project".equals(vendor)) {
            return OSType.ANDROID;
        }
        String osName = System.getProperty("os.name");
        if ((osName = osName.toLowerCase(Locale.ENGLISH)).startsWith("mac os")) {
            return OSType.APPLE;
        }
        if (osName.startsWith("windows")) {
            return OSType.WINDOWS;
        }
        if (osName.startsWith("linux")) {
            return OSType.LINUX;
        }
        if (osName.startsWith("sun")) {
            return OSType.SUN;
        }
        return OSType.UNKNOWN;
    }

    public static ArchType calculateArch() {
        String osArch = System.getProperty("os.arch");
        if ("i386".equals(osArch = osArch.toLowerCase(Locale.ENGLISH)) || "x86".equals(osArch) || "i686".equals(osArch)) {
            return ArchType.X86;
        }
        if (osArch.startsWith("amd64") || osArch.startsWith("x86_64")) {
            return ArchType.X86_64;
        }
        if (osArch.startsWith("arm64")) {
            return ArchType.ARM64;
        }
        if (osArch.startsWith("arm")) {
            return ArchType.ARM;
        }
        if ("ppc".equals(osArch) || "powerpc".equals(osArch)) {
            return ArchType.PPC;
        }
        if (osArch.startsWith("ppc")) {
            return ArchType.PPC_64;
        }
        if (osArch.startsWith("sparc")) {
            return ArchType.SPARC;
        }
        if (osArch.startsWith("mips64")) {
            return ArchType.MIPS64;
        }
        if (osArch.startsWith("mips")) {
            return ArchType.MIPS;
        }
        if (osArch.contains("risc")) {
            return ArchType.RISC;
        }
        if (osArch.startsWith("aarch64")) {
            return ArchType.AARCH64;
        }
        return ArchType.UNKNOWN;
    }

    static {
        String levelPropertyName = "jcuda.LibUtils.level";
        String levelValue = System.getProperty(levelPropertyName);
        Level parsedLevel = Level.FINE;
        if (levelValue != null) {
            try {
                parsedLevel = Level.parse(levelValue);
            }
            catch (IllegalArgumentException e) {
                logger.warning("Invalid value for " + levelPropertyName + ": " + levelValue + " - defaulting to Level.FINE");
            }
        }
        level = parsedLevel;
    }

    public static enum ArchType {
        PPC,
        PPC_64,
        SPARC,
        X86,
        X86_64,
        ARM,
        ARM64,
        AARCH64,
        MIPS,
        MIPS64,
        RISC,
        UNKNOWN;

    }

    public static enum OSType {
        ANDROID,
        APPLE,
        LINUX,
        SUN,
        WINDOWS,
        UNKNOWN;

    }
}

