/**
 * $Id: UserAuthorizationProvider.java 3497 2009-02-25 17:39:08Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/providers/UserAuthorizationProvider.java $
 * UserAuthorizationService.java - DSpace2 - Oct 14, 2008 10:53:14 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.providers;

import java.util.List;

/**
 * This allows customization of the authorization system for DSpace by allowing
 * an external system to indicate the permissions that are allowed for a user <br/>
 * This uses user EIDs instead of the internal user IDs since the external developer
 * should always be able to work in the ids they are used to,
 * translation happens internally <br/>
 * NOTE about external identifiers: <br/>
 * The external id (eid) is the identifier that is used in the source for your user data
 * and is supplied by you when the user is first authenticated, once this id is known it will
 * be used for lookups of data in this provider,
 * if you are not providing the user eid then it will be the username (login name) of the user
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface UserAuthorizationProvider {

    /**
     * Indicate if a user has super admin access in the system,
     * super admins are allowed to basically do anything with the data in the system
     * 
     * @param userEid the user Eid (typically username, not the internal user id)
     * @return true if the user has admin access OR false otherwise
     * @throws IllegalArgumentException if the inputs are null
     * @throws AuthzDoNotCareException if this provider does not have an opinion or more information
     * about the given user, permission, and/or reference
     */
    public boolean isUserAdmin(String userEid);

    /**
     * Return a list of all user Eids that have the given permission on the given resources <br/>
     * <b>Note</b>: this should be external/enterprise IDs (external user id from your provider source) 
     * 
     * @param references complete references to the objects we are checking permissions on (e.g. /collection/123)
     * @param permission a permission string (e.g. collection.create)
     * @return a Set of Strings which represent the user EIDs of all users with that permission
     * OR null if you have no information about this user and reference
     * @throws IllegalArgumentException if the inputs are null
     * @throws AuthzDoNotCareException if this provider does not have an opinion or more information
     * about the given user, permission, and/or reference
     */
    public List<String> getUsersForPermsRefs(String[] references, String permission);

    /**
     * Return a list of all references that the given user has this permission on
     * 
     * @param userEid the user EID (typically username, not the internal user id)
     * @param permission a permission string (e.g. collection.create)
     * @return a List of reference strings which this user has the given permission on
     * OR null if you have no information about this user and reference
     * @throws IllegalArgumentException if the inputs are null
     * @throws AuthzDoNotCareException if this provider does not have an opinion or more information
     * about the given user, permission, and/or reference
     */
    public List<String> getRefsForUserPerm(String userEid, String permission);

    /**
     * Return a list of all permissions that a user has on a specific reference
     * 
     * @param userEid the user EID (typically username, not the internal user id)
     * @param reference a complete reference to an object we are checking permissions on (e.g. /collection/123)
     * @return the list of all permissions that this user has on this reference 
     * OR null if you have no information about this user and reference
     * @throws IllegalArgumentException if the inputs are null
     * @throws AuthzDoNotCareException if this provider does not have an opinion or more information
     * about the given user, permission, and/or reference
     */
    public List<String> getPermsForUserAndRef(String userEid, String reference);

    /**
     * Indicate if a user has a specified permission/role for a given reference
     * 
     * @param userEid the user EID (typically username, not the internal user id)
     * @param permission a permission string (e.g. collection.create)
     * @param reference a complete reference to an object we are checking permissions on (e.g. /collection/123)
     * @return true if allowed, false otherwise
     * @throws IllegalArgumentException if the inputs are null
     * @throws AuthzDoNotCareException if this provider does not have an opinion or more information
     * about the given user, permission, and/or reference
     */
    public boolean isUserAllowedOnRef(String userEid, String permission, String reference);

}
