/**
 * $Id: MappingService.java 3540 2009-03-09 12:37:46Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/MappingService.java $
 * IdEidMappingService.java - DSpace2 - Nov 17, 2008 3:57:37 PM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services;

import java.util.List;
import java.util.Map;

import org.dspace.services.model.IdEidMap;


/**
 * This is a service which is used to manage the mapping of eids (external ids)
 * to internal ids (whatever form they might take), this is here to reduce code duplication
 * and allow for quicker writing of services which use providers and identifiers,
 * it can be used for mapping other things as well but the language in the methods
 * uses id and eid as the 2 things which are mapped together
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface MappingService {

    /**
     * @param type the type key (e.g. "user")
     * @param eid the external id value
     * @return the internal id value OR null if none found
     * @throws IllegalArgumentException if any arguments are null
     */
    public String getIdFromEid(String type, String eid);

    /**
     * @param type the type key (e.g. "user")
     * @param id the internal id value
     * @return the external id value OR null if none found
     * @throws IllegalArgumentException if any arguments are null
     */
    public String getEidFromId(String type, String id);

    /**
     * @param type the type key (e.g. "user")
     * @return a list of all mappings for the given type key OR empty list of none found
     * @throws IllegalArgumentException if any arguments are null
     */
    public List<IdEidMap> getByType(String type);

    /**
     * Allows the creation of mappings,
     * does not get upset if the mapping already existed
     * 
     * @param type the type key (e.g. "user")
     * @param id the internal id value
     * @param eid the external id value
     * @return the mapping that was saved if it is new OR null if the mapping already existed
     * @throws IllegalArgumentException if any arguments are null
     */
    public IdEidMap saveMapping(String type, String id, String eid);

    /**
     * Get a mapping when the id and eid are known (or thought to be known)
     * @param type the type key (e.g. "user")
     * @param id the internal id value
     * @param eid the external id value
     * @return the mapping which matches this OR null if none can be found
     * @throws IllegalArgumentException if any arguments are null
     */
    public IdEidMap getMapping(String type, String id, String eid);

    /**
     * Allows for an efficient check for a large number of mappings by either
     * the ids or eids (identifiers)
     * 
     * @param type the type key (e.g. "user")
     * @param identifiers an array of the mapped ids or eids (depending on matchIds) to check for this type
     * @param matchIds if true then the identifiers are assumed to be ids, if false then they are assumed to be eids
     * @return the map of identifier -> mapping for all mappings which can be found for the given type and identifiers, 
     * empty map if none found, only the existing mappings are returned
     * @throws IllegalArgumentException if any arguments are null
     */
    public Map<String, IdEidMap> findMappings(String type, String[] identifiers, boolean matchIds);

    /**
     * @param mappingId the id of the mapping itself
     * @return the mapping which was removed OR null if no mapping was found
     */
    public IdEidMap deleteById(int mappingId);

}
