/*
 * $URL: https://dspace.svn.sf.net/svnroot/dspace/dspace2/trunk/api/src/main/java/org/dspace/services/exceptions/AuthorizeException.java $
 * 
 * $Revision: 3173 $
 * 
 * $Date: 2008-10-09 07:49:25 -0700 (Thu, 09 Oct 2008) $
 *
 * Copyright (c) 2008, The DSpace Foundation.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 *
 * - Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 *
 * - Neither the name of the DSpace Foundation nor the names of its 
 * contributors may be used to endorse or promote products derived from 
 * this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDERS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 */
package org.dspace.services;

import org.dspace.services.model.MetadataField;

/**
 * DSpace object that represents a metadata field, which is defined by a
 * combination of schema, element, and qualifier. Every metadata element belongs
 * in a field.
 * 
 * @author Martin Hald
 * @author mdiggory
 * 
 * @version $Revision: 3036 $
 */
public interface MetadataFieldService {


    public static final String FIELD_KEY = "field";

    public static final String FIELD_REF_PREFIX = "/"+FIELD_KEY+"/";
    
	/**
	 * Creates a new metadata field.
	 * @param field to create
	 */
	public MetadataField create(MetadataField field);

	/**
	 * Delete the metadata field.
	 * @param field to delete
	 */
	public MetadataField delete(MetadataField field);

	/**
	 * Find the field corresponding to the given numeric ID. The ID is a
	 * database key internal to DSpace.
	 * @param id the metadata field ID
	 * @return the metadata field object
	 */
	public MetadataField findById(String id);

	/**
	 * Retrieve all Dublin Core types from the registry
	 * @return an array of all the Dublin Core types
	 */
	public MetadataField[] findAll();

	/**
	 * Return all metadata fields that are found in a given schema.
	 * @param prefix schema by db prefix
	 * @return array of metadata fields
	 */
	public MetadataField[] findAllInSchema(String prefix);

	/**
	 * Retrieves the metadata field from the database.
	 * 
	 * @param prefix schema prefix
	 * @param name element name
	 * @return field
	 */
	public MetadataField findByName(String prefix, String name);

	/**
	 * Return true if and only if the schema has a field with the given element
	 * and qualifier pair.
	 * 
     * @param prefix schema prefix
     * @param name element name
	 * @return true if the field exists
	 */
	public boolean hasName(String prefix, String name);

	/**
	 * A sanity check that ensures a given element and qualifier are unique
	 * within a given schema. The check happens in code as we cannot use a
	 * database constraint.
     * @param prefix schema prefix
     * @param name element name
	 * @return true if unique
	 * @deprecated
	 */
	public boolean unique(String prefix, String name);

	/**
	 * Update the metadata field in the database.
	 * @param field to update
	 */
	public MetadataField update(MetadataField field);

}
