/**
 * $Id: StorageVersionable.java 3476 2009-02-12 16:37:41Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/mixins/StorageVersionable.java $
 * StorageMovable.java - DS2 - Feb 6, 2009 10:54:41 AM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.mixins;

import java.util.List;

import org.dspace.services.StorageService;
import org.dspace.services.exceptions.StorageInvalidVersionException;
import org.dspace.services.model.StorageEntity;

/**
 * Allows storage entities to be retrieved according to their versions and allows
 * versioning information to be retrieved about entities <br/>
 * Version names are generated internally by the system depending on the internal strategy of the storage system,
 * most likely a new version name is generated for all updates <br/>
 * One of the mixins for the storage service
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public interface StorageVersionable extends StorageService {

    /**
     * If this is used as the versionName it will always return the current version,
     * even if the item is not versioned or cannot be versioned
     */
    public static String CURRENT_VERSION = "";

    /**
     * Gets a list of all version keys of this storage entity,
     * note that if there is only a single version of this entity then this returns an empty list
     * 
     * @param reference the unique reference (path + name) of the given entity (e.g. /trees/apple/golden_delicious)
     * @return the list of all versions for this storage entity OR empty list if this only has one version
     * @throws IllegalArgumentException if any of the required arguments are null or invalid
     * @throws StorageInvalidReferenceException if the given reference does not exist
     * @throws StorageException if there is an internal storage system failure
     */
    public List<StorageVersion> getVersions(String reference);

    /**
     * Get a specific version of a storage entity
     * 
     * @param reference the unique reference (path + name) of the given entity (e.g. /trees/apple/golden_delicious)
     * @param versionName the version key (normally obtained from the list in {@link #getVersions(String)}),
     * using {@link #CURRENT_VERSION} will always retrieve the current version of the entity even if there
     * are no versions stored yet (this is the same as simply getting the entity)
     * @return the version of the storage entity OR null if the version is not found
     * @throws IllegalArgumentException if any of the required arguments are null or invalid
     * @throws StorageInvalidReferenceException if the given reference does not exist
     * @throws StorageInvalidVersionException if the given version does not exist
     * @throws StorageException if there is an internal storage system failure
     */
    public StorageEntity getVersion(String reference, String versionName);

}
