/**
 * $Id: UserSearch.java 3305 2008-11-19 16:01:19Z azeckoski $
 * $URL: https://scm.dspace.org/svn/repo/dspace2/core/tags/OR09/api/src/main/java/org/dspace/services/model/UserSearch.java $
 * UserSearch.java - DSpace2 - Oct 13, 2008 12:24:15 PM - azeckoski
 **************************************************************************
 * Copyright (c) 2008 Aaron Zeckoski
 * Licensed under the Apache License, Version 2.0
 * 
 * A copy of the Apache License has been included in this 
 * distribution and is available at: http://www.apache.org/licenses/LICENSE-2.0.txt
 *
 * Aaron Zeckoski (azeckoski @ gmail.com) (aaronz @ vt.edu) (aaron @ caret.cam.ac.uk)
 */

package org.dspace.services.model;

import java.util.EnumMap;
import java.util.Map;

import org.dspace.search.Order;
import org.dspace.search.Restriction;
import org.dspace.search.Search;


/**
 * Allows for searching for users and restricting the returned data,
 * Includes extensions which define the searchable user fields {@link SearchFields}
 * and some methods to make user searching easier: 
 * {@link #setUserSearch(String)} and {@link #addUserSearch(SearchFields, String, int)},
 * user {@link #addUserOrder(SearchFields, boolean)} to set the order of the search
 * 
 * @author Aaron Zeckoski (azeckoski @ gmail.com)
 */
public class UserSearch extends Search {

    /**
     * Defines the searchable fields for the {@link User} object
     */
    public static enum SearchFields {USERNAME, EMAIL, FULLNAME};
    /**
     * Only properties in this map are searchable for users, others will be ignored,
     * the map goes from the search field enum -> the User object property name
     */
    public final static Map<SearchFields, String> searchPropertyMap;
    static {
        searchPropertyMap = new EnumMap<SearchFields, String>(SearchFields.class);
        searchPropertyMap.put(SearchFields.USERNAME, "eid");
        searchPropertyMap.put(SearchFields.EMAIL, "email");
        searchPropertyMap.put(SearchFields.FULLNAME, "displayName");
    }

    /**
     * Search all the user fields for a partial match on the given text,
     * will attempt to match email, name and username (like gmail),
     * will clear out any search restrictions which currently exist in this search
     * 
     * @param text the text to search for
     */
    public void setUserSearch(String text) {
        this.setConjunction(false);
		text = "%"+text+"%"; // for the like match
        this.setRestrictions( new Restriction[] {
                new Restriction(searchPropertyMap.get(SearchFields.USERNAME), text, Restriction.LIKE),
                new Restriction(searchPropertyMap.get(SearchFields.EMAIL), text, Restriction.LIKE),
                new Restriction(searchPropertyMap.get(SearchFields.FULLNAME), text, Restriction.LIKE)
            }
        );
    }

    /**
     * Adds a single user search term
     * 
     * @param field the user field to search
     * @param searchValue the value to search for
     * @param comparison the comparison to make between the property and the value,
     * use the defined constants: e.g. EQUALS, LIKE, etc... from {@link Restriction}
     */
    public void addUserSearch(SearchFields field, String searchValue, int comparison) {
        this.addRestriction( new Restriction(searchPropertyMap.get(field), searchValue, comparison) );
    }

    /**
     * Adds an ordering for the user search
     * 
     * @param field the user field to order by
     * @param ascending if true then the order will be ascending (natural order), otherwise descending
     */
    public void addUserOrder(SearchFields field, boolean ascending) {
        this.addOrder( new Order(searchPropertyMap.get(field), ascending) );
    }

}
