/*
 * Decompiled with CFR 0.152.
 */
package org.dspace.util;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.TimeZone;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.impl.HttpSolrServer;
import org.apache.solr.client.solrj.request.AbstractUpdateRequest;
import org.apache.solr.client.solrj.request.ContentStreamUpdateRequest;
import org.apache.solr.client.solrj.request.CoreAdminRequest;
import org.apache.solr.client.solrj.request.LukeRequest;
import org.apache.solr.client.solrj.response.CoreAdminResponse;
import org.apache.solr.client.solrj.response.FieldStatsInfo;
import org.apache.solr.client.solrj.response.LukeResponse;
import org.apache.solr.client.solrj.response.RangeFacet;
import org.apache.solr.common.luke.FieldFlag;
import org.apache.solr.common.params.CoreAdminParams;
import org.apache.solr.common.params.SolrParams;
import org.dspace.core.ConfigurationManager;
import org.dspace.util.SolrImportExportException;

public class SolrImportExport {
    private static final ThreadLocal<DateFormat> SOLR_DATE_FORMAT = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS'Z'");
            simpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));
            return simpleDateFormat;
        }
    };
    private static final ThreadLocal<DateFormat> SOLR_DATE_FORMAT_NO_MS = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            return new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        }
    };
    private static final ThreadLocal<DateFormat> EXPORT_DATE_FORMAT = new ThreadLocal<DateFormat>(){

        @Override
        protected DateFormat initialValue() {
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM");
            simpleDateFormat.setTimeZone(TimeZone.getDefault());
            return simpleDateFormat;
        }
    };
    private static final String ACTION_OPTION = "a";
    private static final String CLEAR_OPTION = "c";
    private static final String DIRECTORY_OPTION = "d";
    private static final String HELP_OPTION = "h";
    private static final String INDEX_NAME_OPTION = "i";
    private static final String KEEP_OPTION = "k";
    private static final String LAST_OPTION = "l";
    public static final int ROWS_PER_FILE = 10000;
    private static final Logger log = Logger.getLogger(SolrImportExport.class);

    public static void main(String[] args) throws org.apache.commons.cli.ParseException {
        PosixParser parser = new PosixParser();
        Options options = SolrImportExport.makeOptions();
        try {
            CommandLine line = parser.parse(options, args);
            if (line.hasOption(HELP_OPTION)) {
                SolrImportExport.printHelpAndExit(options, 0);
            }
            if (!line.hasOption(INDEX_NAME_OPTION)) {
                System.err.println("This command requires the index-name option but none was present.");
                SolrImportExport.printHelpAndExit(options, 1);
            }
            String[] indexNames = line.getOptionValues(INDEX_NAME_OPTION);
            String directoryName = SolrImportExport.makeDirectoryName(line.getOptionValue(DIRECTORY_OPTION));
            String action = line.getOptionValue(ACTION_OPTION, "export");
            if ("import".equals(action)) {
                for (String indexName : indexNames) {
                    File importDir = new File(directoryName);
                    if (!importDir.exists() || !importDir.canRead()) {
                        System.err.println("Import directory " + directoryName + " doesn't exist or is not readable by the current user. Not importing index " + indexName);
                        continue;
                    }
                    try {
                        String solrUrl = SolrImportExport.makeSolrUrl(indexName);
                        boolean clear = line.hasOption(CLEAR_OPTION);
                        SolrImportExport.importIndex(indexName, importDir, solrUrl, clear, clear);
                    }
                    catch (IOException | SolrServerException | SolrImportExportException e) {
                        System.err.println("Problem encountered while trying to import index " + indexName + ".");
                        e.printStackTrace(System.err);
                    }
                }
            } else if ("export".equals(action)) {
                for (String indexName : indexNames) {
                    String lastValue = line.getOptionValue(LAST_OPTION);
                    File exportDir = new File(directoryName);
                    if (exportDir.exists() && !exportDir.canWrite()) {
                        System.err.println("Export directory " + directoryName + " is not writable by the current user. Not exporting index " + indexName);
                        continue;
                    }
                    if (!exportDir.exists()) {
                        boolean created = exportDir.mkdirs();
                        if (created) continue;
                        System.err.println("Export directory " + directoryName + " could not be created. Not exporting index " + indexName);
                        continue;
                    }
                    try {
                        String solrUrl = SolrImportExport.makeSolrUrl(indexName);
                        String timeField = SolrImportExport.makeTimeField(indexName);
                        SolrImportExport.exportIndex(indexName, exportDir, solrUrl, timeField, lastValue);
                    }
                    catch (IOException | SolrServerException | SolrImportExportException e) {
                        System.err.println("Problem encountered while trying to export index " + indexName + ".");
                        e.printStackTrace(System.err);
                    }
                }
            } else if ("reindex".equals(action)) {
                for (String indexName : indexNames) {
                    try {
                        boolean keepExport = line.hasOption(KEEP_OPTION);
                        SolrImportExport.reindex(indexName, directoryName, keepExport);
                    }
                    catch (IOException | SolrServerException | SolrImportExportException e) {
                        e.printStackTrace();
                    }
                }
            } else {
                System.err.println("Unknown action " + action + "; must be import, export or reindex.");
                SolrImportExport.printHelpAndExit(options, 1);
            }
        }
        catch (org.apache.commons.cli.ParseException e) {
            System.err.println("Cannot read command options");
            SolrImportExport.printHelpAndExit(options, 1);
        }
    }

    private static Options makeOptions() {
        Options options = new Options();
        options.addOption(ACTION_OPTION, "action", true, "The action to perform: import, export or reindex. Default: export.");
        options.addOption(CLEAR_OPTION, "clear", false, "When importing, also clear the index first. Ignored when action is export or reindex.");
        options.addOption(DIRECTORY_OPTION, "directory", true, "The absolute path for the directory to use for import or export. If omitted, [dspace]/solr-export is used.");
        options.addOption(HELP_OPTION, "help", false, "Get help on options for this command.");
        options.addOption(INDEX_NAME_OPTION, "index-name", true, "The names of the indexes to process. At least one is required. Available indexes are: authority, statistics.");
        options.addOption(KEEP_OPTION, "keep", false, "When reindexing, keep the contents of the data export directory. By default, the contents of this directory will be deleted once the reindex has finished. Ignored when action is export or import.");
        options.addOption(LAST_OPTION, "last", true, "When exporting, export records from the last [timeperiod] only. This can be one of: 'd' (beginning of yesterday through to now); 'm' (beginning of the previous month through to end of the previous month); a number, in which case the last [number] of days are exported, through to now (use 0 for today's data). Date calculation is done in UTC. If omitted, all documents are exported.");
        return options;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void reindex(String indexName, String exportDirName, boolean keepExport) throws IOException, SolrServerException, SolrImportExportException {
        String tempIndexName = indexName + "-temp";
        String origSolrUrl = SolrImportExport.makeSolrUrl(indexName);
        String baseSolrUrl = StringUtils.substringBeforeLast((String)origSolrUrl, (String)"/");
        String tempSolrUrl = baseSolrUrl + "/" + tempIndexName;
        String solrInstanceDir = ConfigurationManager.getProperty("dspace.dir") + File.separator + "solr" + File.separator + indexName;
        File solrInstance = new File(solrInstanceDir);
        if (!(solrInstance.exists() && solrInstance.canRead() && solrInstance.isDirectory())) {
            throw new SolrImportExportException("Directory " + solrInstanceDir + "/conf/ doesn't exist or isn't readable. The reindexing process requires the Solr configuration directory for this index to be present on the local machine even if Solr is running on a different host. Not reindexing index " + indexName);
        }
        String timeField = SolrImportExport.makeTimeField(indexName);
        File exportDir = new File(exportDirName);
        boolean createdExportDir = exportDir.mkdirs();
        if (!createdExportDir && !exportDir.exists()) {
            throw new SolrImportExportException("Could not create export directory " + exportDirName);
        }
        if (!exportDir.canWrite()) {
            throw new SolrImportExportException("Can't write to export directory " + exportDirName);
        }
        try {
            HttpSolrServer adminSolr = new HttpSolrServer(baseSolrUrl);
            CoreAdminResponse status = CoreAdminRequest.getStatus((String)indexName, (SolrServer)adminSolr);
            Object coreSizeObj = status.getCoreStatus(indexName).get("sizeInBytes");
            long coreSize = coreSizeObj != null ? Long.valueOf(coreSizeObj.toString()) : -1L;
            long usableExportSpace = exportDir.getUsableSpace();
            if (coreSize >= 0L && usableExportSpace < coreSize) {
                System.err.println("Not enough space in export directory " + exportDirName + "; need at least as much space as the index (" + FileUtils.byteCountToDisplaySize((long)coreSize) + ") but usable space in export directory is only " + FileUtils.byteCountToDisplaySize((long)usableExportSpace) + ". Not continuing with reindex, please use the " + DIRECTORY_OPTION + " option to specify an alternative export directy with sufficient space.");
                return;
            }
            File tempDataDir = new File(ConfigurationManager.getProperty("dspace.dir") + File.separator + "temp" + File.separator + "solr-data");
            boolean createdTempDataDir = tempDataDir.mkdirs();
            if (!createdTempDataDir && !tempDataDir.exists()) {
                throw new SolrImportExportException("Could not create temporary data directory " + tempDataDir.getCanonicalPath());
            }
            if (!tempDataDir.canWrite()) {
                throw new SolrImportExportException("Can't write to temporary data directory " + tempDataDir.getCanonicalPath());
            }
            try {
                CoreAdminRequest.Create createRequest = new CoreAdminRequest.Create();
                createRequest.setInstanceDir(solrInstanceDir);
                createRequest.setDataDir(tempDataDir.getCanonicalPath());
                createRequest.setCoreName(tempIndexName);
                createRequest.process((SolrServer)adminSolr).getStatus();
            }
            catch (SolrServerException e) {
                System.err.println("Caught exception when trying to create temporary core: " + e.getMessage() + "; trying to recover.");
                e.printStackTrace(System.err);
            }
            CoreAdminRequest swapRequest = new CoreAdminRequest();
            swapRequest.setCoreName(indexName);
            swapRequest.setOtherCoreName(tempIndexName);
            swapRequest.setAction(CoreAdminParams.CoreAdminAction.SWAP);
            swapRequest.process((SolrServer)adminSolr);
            try {
                SolrImportExport.exportIndex(indexName, exportDir, tempSolrUrl, timeField);
                SolrImportExport.importIndex(indexName, exportDir, tempSolrUrl, true, true);
            }
            catch (Exception e) {
                System.err.println("Encountered problem during reindex: " + e.getMessage() + ", will attempt to restore Solr cores");
                e.printStackTrace(System.err);
            }
            HttpSolrServer origSolr = new HttpSolrServer(origSolrUrl);
            origSolr.commit();
            swapRequest = new CoreAdminRequest();
            swapRequest.setCoreName(tempIndexName);
            swapRequest.setOtherCoreName(indexName);
            swapRequest.setAction(CoreAdminParams.CoreAdminAction.SWAP);
            swapRequest.process((SolrServer)adminSolr);
            SolrImportExport.exportIndex(tempIndexName, exportDir, tempSolrUrl, timeField);
            SolrImportExport.importIndex(tempIndexName, exportDir, origSolrUrl, false, true);
            origSolr.commit();
            CoreAdminRequest.unloadCore((String)tempIndexName, (boolean)false, (boolean)false, (SolrServer)adminSolr);
            if (createdTempDataDir && tempDataDir.exists()) {
                FileUtils.deleteDirectory((File)tempDataDir);
            }
        }
        finally {
            if (!keepExport && createdExportDir && exportDir.exists()) {
                FileUtils.deleteDirectory((File)exportDir);
            }
        }
    }

    public static void exportIndex(String indexName, File toDir, String solrUrl, String timeField) throws SolrServerException, SolrImportExportException, IOException {
        SolrImportExport.exportIndex(indexName, toDir, solrUrl, timeField, null);
    }

    public static void importIndex(final String indexName, File fromDir, String solrUrl, boolean clear, boolean overwrite) throws IOException, SolrServerException, SolrImportExportException {
        Object[] files;
        if (StringUtils.isBlank((String)solrUrl)) {
            throw new SolrImportExportException("Could not construct solr URL for index" + indexName + ", aborting export.");
        }
        if (!fromDir.exists() || !fromDir.canRead()) {
            throw new SolrImportExportException("Source directory " + fromDir + " doesn't exist or isn't readable, aborting export of index " + indexName);
        }
        HttpSolrServer solr = new HttpSolrServer(solrUrl);
        List<String> multivaluedFields = SolrImportExport.getMultiValuedFields(solr);
        if (clear) {
            SolrImportExport.clearIndex(solrUrl);
        }
        if ((files = fromDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.startsWith(indexName) && name.endsWith(".csv");
            }
        })) == null || files.length == 0) {
            log.warn((Object)("No export files found in directory " + fromDir.getCanonicalPath() + " for index " + indexName));
            return;
        }
        Arrays.sort(files);
        for (Object file : files) {
            log.info((Object)("Importing file " + ((File)file).getCanonicalPath()));
            ContentStreamUpdateRequest contentStreamUpdateRequest = new ContentStreamUpdateRequest("/update/csv");
            if (overwrite) {
                contentStreamUpdateRequest.setParam("skip", "_version_");
            }
            for (String mvField : multivaluedFields) {
                contentStreamUpdateRequest.setParam("f." + mvField + ".split", "true");
                contentStreamUpdateRequest.setParam("f." + mvField + ".escape", "\\");
            }
            contentStreamUpdateRequest.setParam("stream.contentType", "text/csv;charset=utf-8");
            contentStreamUpdateRequest.setAction(AbstractUpdateRequest.ACTION.COMMIT, true, true);
            contentStreamUpdateRequest.addFile((File)file, "text/csv;charset=utf-8");
            solr.request((SolrRequest)contentStreamUpdateRequest);
        }
        solr.commit(true, true);
    }

    private static List<String> getMultiValuedFields(HttpSolrServer solr) {
        ArrayList<String> result = new ArrayList<String>();
        try {
            LukeRequest request = new LukeRequest();
            LukeResponse response = request.process((SolrServer)solr);
            Map fields = response.getFieldInfo();
            for (LukeResponse.FieldInfo info : fields.values()) {
                if (!info.getSchema().contains(FieldFlag.MULTI_VALUED.getAbbreviation() + "")) continue;
                result.add(info.getName());
            }
        }
        catch (IOException | SolrServerException e) {
            log.fatal((Object)("Cannot determine which fields are multi valued: " + e.getMessage()), e);
        }
        return result;
    }

    public static void clearIndex(String solrUrl) throws IOException, SolrServerException {
        HttpSolrServer solr = new HttpSolrServer(solrUrl);
        solr.deleteByQuery("*:*");
        solr.commit();
        solr.optimize();
    }

    public static void exportIndex(String indexName, File toDir, String solrUrl, String timeField, String fromWhen) throws SolrServerException, IOException, SolrImportExportException {
        String lastValueFilter;
        if (StringUtils.isBlank((String)solrUrl)) {
            throw new SolrImportExportException("Could not construct solr URL for index" + indexName + ", aborting export.");
        }
        if (!toDir.exists() || !toDir.canWrite()) {
            throw new SolrImportExportException("Target directory " + toDir + " doesn't exist or is not writable, aborting export of index " + indexName);
        }
        HttpSolrServer solr = new HttpSolrServer(solrUrl);
        SolrQuery query = new SolrQuery("*:*");
        if (StringUtils.isNotBlank((String)fromWhen) && StringUtils.isNotBlank((String)(lastValueFilter = SolrImportExport.makeFilterQuery(timeField, fromWhen)))) {
            query.addFilterQuery(new String[]{lastValueFilter});
        }
        query.setRows(Integer.valueOf(0));
        query.setGetFieldStatistics(timeField);
        Map fieldInfo = solr.query((SolrParams)query).getFieldStatsInfo();
        if (fieldInfo == null || !fieldInfo.containsKey(timeField)) {
            log.warn((Object)("Cannot get earliest date, not exporting index " + indexName + ", time field " + timeField + ", from " + fromWhen));
            return;
        }
        FieldStatsInfo timeFieldInfo = (FieldStatsInfo)fieldInfo.get(timeField);
        if (timeFieldInfo == null || timeFieldInfo.getMin() == null) {
            log.warn((Object)("Cannot get earliest date, not exporting index " + indexName + ", time field " + timeField + ", from " + fromWhen));
            return;
        }
        Date earliestTimestamp = (Date)timeFieldInfo.getMin();
        query.setGetFieldStatistics(false);
        query.clearSorts();
        query.setRows(Integer.valueOf(0));
        query.setFacet(true);
        query.add("facet.range", new String[]{timeField});
        query.add("facet.range.start", new String[]{SOLR_DATE_FORMAT.get().format(earliestTimestamp) + "/MONTH"});
        query.add("facet.range.end", new String[]{"NOW/MONTH+1MONTH"});
        query.add("facet.range.gap", new String[]{"+1MONTH"});
        query.setFacetMinCount(1);
        List monthFacets = ((RangeFacet)solr.query((SolrParams)query).getFacetRanges().get(0)).getCounts();
        for (RangeFacet.Count monthFacet : monthFacets) {
            Date monthStartDate;
            String monthStart = monthFacet.getValue();
            try {
                monthStartDate = SOLR_DATE_FORMAT_NO_MS.get().parse(monthStart);
            }
            catch (ParseException e) {
                throw new SolrImportExportException("Could not read start of month batch as date: " + monthStart, e);
            }
            int docsThisMonth = monthFacet.getCount();
            SolrQuery monthQuery = new SolrQuery("*:*");
            monthQuery.setRows(Integer.valueOf(10000));
            monthQuery.set("wt", new String[]{"csv"});
            monthQuery.set("fl", new String[]{"*"});
            monthQuery.addFilterQuery(new String[]{timeField + ":[" + monthStart + " TO " + monthStart + "+1MONTH]"});
            for (int i = 0; i < docsThisMonth; i += 10000) {
                monthQuery.setStart(Integer.valueOf(i));
                URL url = new URL(solrUrl + "/select?" + monthQuery.toString());
                File file = new File(toDir.getCanonicalPath(), SolrImportExport.makeExportFilename(indexName, monthStartDate, docsThisMonth, i));
                if (!file.createNewFile()) {
                    throw new SolrImportExportException("Could not create file " + file.getCanonicalPath() + " while exporting index " + indexName + ", month" + monthStart + ", batch " + i);
                }
                FileUtils.copyURLToFile((URL)url, (File)file);
                log.info((Object)("Exported batch " + i + " to " + file.getCanonicalPath()));
            }
        }
    }

    private static String makeFilterQuery(String timeField, String lastValue) {
        if ("m".equals(lastValue)) {
            return timeField + ":[NOW/MONTH-1MONTH TO NOW/MONTH]";
        }
        int days = DIRECTORY_OPTION.equals(lastValue) ? 1 : Integer.valueOf(lastValue);
        return timeField + ":[NOW/DAY-" + days + "DAYS TO " + SOLR_DATE_FORMAT.get().format(new Date()) + "]";
    }

    private static String makeDirectoryName(String directoryValue) {
        if (StringUtils.isNotBlank((String)directoryValue)) {
            return directoryValue;
        }
        return ConfigurationManager.getProperty("dspace.dir") + File.separator + "solr-export" + File.separator;
    }

    private static String makeExportFilename(String indexName, Date exportStart, long totalRecords, int index) {
        String exportFileNumber = "";
        if (totalRecords > 10000L) {
            exportFileNumber = StringUtils.leftPad((String)("" + index / 10000), (int)((int)Math.ceil(Math.log10(totalRecords / 10000L))), (String)"0");
        }
        return indexName + "_export_" + EXPORT_DATE_FORMAT.get().format(exportStart) + (StringUtils.isNotBlank((String)exportFileNumber) ? "_" + exportFileNumber : "") + ".csv";
    }

    private static String makeSolrUrl(String indexName) {
        if (indexName.startsWith("statistics")) {
            return ConfigurationManager.getProperty("solr-statistics", "server") + indexName.replaceFirst("statistics", "");
        }
        if ("authority".equals(indexName)) {
            return ConfigurationManager.getProperty("solr.authority.server");
        }
        return "http://localhost:8080/solr/" + indexName;
    }

    private static String makeTimeField(String indexName) {
        if (indexName.startsWith("statistics")) {
            return "time";
        }
        if ("authority".equals(indexName)) {
            return "last_modified_date";
        }
        return null;
    }

    private static void printHelpAndExit(Options options, int exitCode) {
        HelpFormatter myhelp = new HelpFormatter();
        myhelp.printHelp(SolrImportExport.class.getSimpleName() + "\n", options);
        System.exit(exitCode);
    }
}

