/**
 * The contents of this file are subject to the license and copyright
 * detailed in the LICENSE and NOTICE files at the root of the source
 * tree and available online at
 *
 * http://www.dspace.org/license/
 */
package org.dspace.content;

import java.util.Objects;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.dspace.core.ReloadableEntity;

/**
 * Class representing an EntityType
 * This class contains an Integer ID that will be the unique value for this class and also the primary key
 * This also has a label that will be used to identify what kind of EntityType this object is
 */
@Entity
@Table(name = "entity_type")
public class EntityType implements ReloadableEntity<Integer> {

    /**
     * The Integer ID used as a primary key for this database object.
     * This is generated by a sequence
     */
    @Id
    @GeneratedValue(strategy = GenerationType.SEQUENCE, generator = "entity_type_id_seq")
    @SequenceGenerator(name = "entity_type_id_seq", sequenceName = "entity_type_id_seq", allocationSize = 1)
    @Column(name = "id", unique = true, nullable = false, insertable = true, updatable = false)
    protected Integer id;

    /**
     * The String label field for the entity type
     * This cannot be null
     */
    @Column(name = "label", nullable = false)
    private String label;

    /**
     * The standard setter for the ID of this EntityType
     *
     * @param id The ID that this EntityType's ID will be set to
     */
    public void setId(Integer id) {
        this.id = id;
    }

    /**
     * The standard getter for the label of this EntityType
     *
     * @return The label for this EntityType
     */
    public String getLabel() {
        return label;
    }

    /**
     * The standard setter for the label of this EntityType
     *
     * @param label The label that this EntityType's label will be set to
     */
    public void setLabel(String label) {
        this.label = label;
    }

    /**
     * The standard getter for the ID of this EntityType
     *
     * @return The ID for this EntityType
     */
    @Override
    public Integer getID() {
        return id;
    }

    /**
     * Determines whether two entity types are equal based on the id and the label
     * @param obj   object to be compared
     * @return
     */
    @Override
    public boolean equals(Object obj) {
        if (!(obj instanceof EntityType)) {
            return false;
        }
        EntityType entityType = (EntityType) obj;

        if (!Objects.equals(this.getID(), entityType.getID())) {
            return false;
        }

        return StringUtils.equals(this.getLabel(), entityType.getLabel());
    }

    /**
     * Returns a hash code value for the object.
     * @return  hash code value
     */
    @Override
    public int hashCode() {
        return new HashCodeBuilder().append(getID()).toHashCode();
    }
}
