/*
 * Decompiled with CFR 0.152.
 */
package org.dspace.storage.bitstore;

import com.amazonaws.AmazonClientException;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.regions.Region;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.amazonaws.services.s3.transfer.Upload;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.function.Supplier;
import javax.validation.constraints.NotNull;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.dspace.content.Bitstream;
import org.dspace.curate.Utils;
import org.dspace.services.ConfigurationService;
import org.dspace.services.factory.DSpaceServicesFactory;
import org.dspace.storage.bitstore.BaseBitStoreService;
import org.dspace.storage.bitstore.factory.StorageServiceFactory;
import org.dspace.storage.bitstore.service.BitstreamStorageService;
import org.dspace.util.FunctionalUtils;
import org.springframework.beans.factory.annotation.Autowired;

public class S3BitStoreService
extends BaseBitStoreService {
    protected static final String DEFAULT_BUCKET_PREFIX = "dspace-asset-";
    protected final String REGISTERED_FLAG = "-R";
    private static final Logger log = LogManager.getLogger(S3BitStoreService.class);
    private static final String CSA = "MD5";
    protected static final int digitsPerLevel = 2;
    protected static final int directoryLevels = 3;
    private boolean enabled = false;
    private String awsAccessKey;
    private String awsSecretKey;
    private String awsRegionName;
    private boolean useRelativePath;
    private String bucketName = null;
    private String subfolder = null;
    private AmazonS3 s3Service = null;
    private TransferManager tm = null;
    private static final ConfigurationService configurationService = DSpaceServicesFactory.getInstance().getConfigurationService();

    protected static Supplier<AmazonS3> amazonClientBuilderBy(@NotNull Regions regions, @NotNull AWSCredentials awsCredentials) {
        return () -> (AmazonS3)((AmazonS3ClientBuilder)((AmazonS3ClientBuilder)AmazonS3ClientBuilder.standard().withCredentials((AWSCredentialsProvider)new AWSStaticCredentialsProvider(awsCredentials))).withRegion(regions)).build();
    }

    public S3BitStoreService() {
    }

    protected S3BitStoreService(AmazonS3 s3Service, TransferManager tm) {
        this.s3Service = s3Service;
        this.tm = tm;
    }

    @Override
    public boolean isEnabled() {
        return this.enabled;
    }

    @Override
    public void init() throws IOException {
        if (this.isInitialized()) {
            return;
        }
        try {
            if (StringUtils.isNotBlank((CharSequence)this.getAwsAccessKey()) && StringUtils.isNotBlank((CharSequence)this.getAwsSecretKey())) {
                log.warn("Use local defined S3 credentials");
                Regions regions = Regions.DEFAULT_REGION;
                if (StringUtils.isNotBlank((CharSequence)this.awsRegionName)) {
                    try {
                        regions = Regions.fromName((String)this.awsRegionName);
                    }
                    catch (IllegalArgumentException e) {
                        log.warn("Invalid aws_region: " + this.awsRegionName);
                    }
                }
                this.s3Service = FunctionalUtils.getDefaultOrBuild(this.s3Service, S3BitStoreService.amazonClientBuilderBy(regions, (AWSCredentials)new BasicAWSCredentials(this.getAwsAccessKey(), this.getAwsSecretKey())));
                log.warn("S3 Region set to: " + regions.getName());
            } else {
                log.info("Using a IAM role or aws environment credentials");
                this.s3Service = FunctionalUtils.getDefaultOrBuild(this.s3Service, AmazonS3ClientBuilder::defaultClient);
            }
            if (StringUtils.isEmpty((CharSequence)this.bucketName)) {
                String hostname = org.dspace.core.Utils.getHostName(configurationService.getProperty("dspace.ui.url"));
                this.bucketName = DEFAULT_BUCKET_PREFIX + hostname;
                log.warn("S3 BucketName is not configured, setting default: " + this.bucketName);
            }
            try {
                if (!this.s3Service.doesBucketExist(this.bucketName)) {
                    this.s3Service.createBucket(this.bucketName);
                    log.info("Creating new S3 Bucket: " + this.bucketName);
                }
            }
            catch (AmazonClientException e) {
                throw new IOException(e);
            }
            this.initialized = true;
            log.info("AWS S3 Assetstore ready to go! bucket:" + this.bucketName);
        }
        catch (Exception e) {
            this.initialized = false;
            log.error("Can't initialize this store!", (Throwable)e);
        }
        log.info("AWS S3 Assetstore ready to go! bucket:" + this.bucketName);
        this.tm = FunctionalUtils.getDefaultOrBuild(this.tm, () -> TransferManagerBuilder.standard().withAlwaysCalculateMultipartMd5(true).withS3Client(this.s3Service).build());
    }

    @Override
    public String generateId() {
        return org.dspace.core.Utils.generateKey();
    }

    @Override
    public InputStream get(Bitstream bitstream) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        if (this.isRegisteredBitstream(key)) {
            key = key.substring("-R".length());
        }
        try {
            S3Object object = this.s3Service.getObject(new GetObjectRequest(this.bucketName, key));
            return object != null ? object.getObjectContent() : null;
        }
        catch (AmazonClientException e) {
            log.error("get(" + key + ")", (Throwable)e);
            throw new IOException(e);
        }
    }

    @Override
    public void put(Bitstream bitstream, InputStream in) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        File scratchFile = File.createTempFile(bitstream.getInternalId(), "s3bs");
        try {
            FileUtils.copyInputStreamToFile((InputStream)in, (File)scratchFile);
            long contentLength = scratchFile.length();
            String localChecksum = Utils.checksum(scratchFile, CSA);
            Upload upload = this.tm.upload(this.bucketName, key, scratchFile);
            upload.waitForUploadResult();
            bitstream.setSizeBytes(contentLength);
            bitstream.setChecksum(localChecksum);
            bitstream.setChecksumAlgorithm(CSA);
        }
        catch (AmazonClientException | IOException | InterruptedException e) {
            log.error("put(" + bitstream.getInternalId() + ", is)", e);
            throw new IOException(e);
        }
        finally {
            if (!scratchFile.delete()) {
                scratchFile.deleteOnExit();
            }
        }
    }

    @Override
    public Map about(Bitstream bitstream, Map attrs) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        if (this.isRegisteredBitstream(key)) {
            key = key.substring("-R".length());
        }
        try {
            ObjectMetadata objectMetadata = this.s3Service.getObjectMetadata(this.bucketName, key);
            if (objectMetadata != null) {
                return this.about(objectMetadata, attrs);
            }
        }
        catch (AmazonS3Exception e) {
            if (e.getStatusCode() == 404) {
                return null;
            }
        }
        catch (AmazonClientException e) {
            log.error("about(" + key + ", attrs)", (Throwable)e);
            throw new IOException(e);
        }
        return null;
    }

    public Map about(ObjectMetadata objectMetadata, Map attrs) {
        if (objectMetadata != null) {
            this.putValueIfExistsKey(attrs, "size_bytes", objectMetadata.getContentLength());
            this.putValueIfExistsKey(attrs, "checksum", objectMetadata.getETag());
            this.putEntryIfExistsKey(attrs, "checksum", Map.entry("checksum_algorithm", CSA));
            this.putValueIfExistsKey(attrs, "modified", String.valueOf(objectMetadata.getLastModified().getTime()));
        }
        return attrs;
    }

    @Override
    public void remove(Bitstream bitstream) throws IOException {
        String key = this.getFullKey(bitstream.getInternalId());
        try {
            this.s3Service.deleteObject(this.bucketName, key);
        }
        catch (AmazonClientException e) {
            log.error("remove(" + key + ")", (Throwable)e);
            throw new IOException(e);
        }
    }

    public String getFullKey(String id) {
        StringBuilder bufFilename = new StringBuilder();
        if (StringUtils.isNotEmpty((CharSequence)this.subfolder)) {
            bufFilename.append(this.subfolder);
            this.appendSeparator(bufFilename);
        }
        if (this.useRelativePath) {
            bufFilename.append(this.getRelativePath(id));
        } else {
            bufFilename.append(id);
        }
        if (log.isDebugEnabled()) {
            log.debug("S3 filepath for " + id + " is " + bufFilename.toString());
        }
        return bufFilename.toString();
    }

    public String getRelativePath(String sInternalId) {
        BitstreamStorageService bitstreamStorageService = StorageServiceFactory.getInstance().getBitstreamStorageService();
        String sIntermediatePath = "";
        if (bitstreamStorageService.isRegisteredBitstream(sInternalId)) {
            sInternalId = sInternalId.substring("-R".length());
        } else {
            sInternalId = this.sanitizeIdentifier(sInternalId);
            sIntermediatePath = this.getIntermediatePath(sInternalId);
        }
        return sIntermediatePath + sInternalId;
    }

    public void setEnabled(boolean enabled) {
        this.enabled = enabled;
    }

    public String getAwsAccessKey() {
        return this.awsAccessKey;
    }

    @Autowired(required=true)
    public void setAwsAccessKey(String awsAccessKey) {
        this.awsAccessKey = awsAccessKey;
    }

    public String getAwsSecretKey() {
        return this.awsSecretKey;
    }

    @Autowired(required=true)
    public void setAwsSecretKey(String awsSecretKey) {
        this.awsSecretKey = awsSecretKey;
    }

    public String getAwsRegionName() {
        return this.awsRegionName;
    }

    public void setAwsRegionName(String awsRegionName) {
        this.awsRegionName = awsRegionName;
    }

    @Autowired(required=true)
    public String getBucketName() {
        return this.bucketName;
    }

    public void setBucketName(String bucketName) {
        this.bucketName = bucketName;
    }

    public String getSubfolder() {
        return this.subfolder;
    }

    public void setSubfolder(String subfolder) {
        this.subfolder = subfolder;
    }

    public boolean isUseRelativePath() {
        return this.useRelativePath;
    }

    public void setUseRelativePath(boolean useRelativePath) {
        this.useRelativePath = useRelativePath;
    }

    public static void main(String[] args) throws Exception {
        CommandLine command;
        Options options = new Options();
        Option option = Option.builder((String)"a").desc("access key").hasArg().required().build();
        options.addOption(option);
        option = Option.builder((String)"s").desc("secret key").hasArg().required().build();
        options.addOption(option);
        option = Option.builder((String)"f").desc("asset file name").hasArg().required().build();
        options.addOption(option);
        DefaultParser parser = new DefaultParser();
        try {
            command = parser.parse(options, args);
        }
        catch (ParseException e) {
            System.err.println(e.getMessage());
            new HelpFormatter().printHelp(S3BitStoreService.class.getSimpleName() + "options", options);
            return;
        }
        String accessKey = command.getOptionValue("a");
        String secretKey = command.getOptionValue("s");
        String assetFile = command.getOptionValue("f");
        S3BitStoreService store = new S3BitStoreService();
        BasicAWSCredentials awsCredentials = new BasicAWSCredentials(accessKey, secretKey);
        store.s3Service = new AmazonS3Client((AWSCredentials)awsCredentials);
        Region usEast1 = Region.getRegion((Regions)Regions.US_EAST_1);
        store.s3Service.setRegion(usEast1);
        String hostname = org.dspace.core.Utils.getHostName(configurationService.getProperty("dspace.ui.url"));
        store.bucketName = DEFAULT_BUCKET_PREFIX + hostname + ".s3test";
        store.s3Service.createBucket(store.bucketName);
    }

    public boolean isRegisteredBitstream(String internalId) {
        return internalId.startsWith("-R");
    }
}

