/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.io.ByteArrayInputStream;
import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.Certificate;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.PKIXParameters;
import java.security.cert.TrustAnchor;
import java.security.cert.X509Certificate;
import java.security.spec.X509EncodedKeySpec;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.security.auth.x500.X500Principal;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.util.DatagramReader;
import org.eclipse.californium.scandium.util.DatagramWriter;

public final class CertificateMessage
extends HandshakeMessage {
    private static final String CERTIFICATE_TYPE_X509 = "X.509";
    private static final Logger LOGGER = Logger.getLogger(CertificateMessage.class.getCanonicalName());
    private static final int CERTIFICATE_LENGTH_BITS = 24;
    private static final int CERTIFICATE_LIST_LENGTH = 24;
    private X509Certificate[] certificateChain;
    private List<byte[]> encodedChain;
    private byte[] rawPublicKeyBytes;
    private int length = 3;

    public CertificateMessage(X509Certificate[] certificateChain, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (certificateChain == null) {
            throw new NullPointerException("Certificate chain must not be null");
        }
        this.setCertificateChain(certificateChain);
        this.calculateLength(certificateChain);
    }

    public CertificateMessage(byte[] rawPublicKeyBytes, InetSocketAddress peerAddress) {
        super(peerAddress);
        if (rawPublicKeyBytes == null) {
            throw new NullPointerException("Raw public key byte array must not be null");
        }
        this.rawPublicKeyBytes = Arrays.copyOf(rawPublicKeyBytes, rawPublicKeyBytes.length);
        this.length += this.rawPublicKeyBytes.length;
    }

    private void setCertificateChain(X509Certificate[] chain) {
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>();
        X500Principal issuer = null;
        for (X509Certificate cert : chain) {
            LOGGER.log(Level.FINER, "Current Subject DN: {0}", cert.getSubjectX500Principal().getName());
            if (issuer != null && !issuer.equals(cert.getSubjectX500Principal())) {
                LOGGER.log(Level.FINER, "Actual Issuer DN: {0}", cert.getSubjectX500Principal().getName());
                throw new IllegalArgumentException("Given certificates do not form a chain");
            }
            if (cert.getIssuerX500Principal().equals(cert.getSubjectX500Principal())) continue;
            certificates.add(cert);
            issuer = cert.getIssuerX500Principal();
            LOGGER.log(Level.FINER, "Expected Issuer DN: {0}", issuer.getName());
        }
        this.certificateChain = certificates.toArray(new X509Certificate[0]);
    }

    @Override
    public HandshakeType getMessageType() {
        return HandshakeType.CERTIFICATE;
    }

    private void calculateLength(Certificate[] certificateChain) {
        if (certificateChain != null && this.encodedChain == null) {
            this.encodedChain = new ArrayList<byte[]>(certificateChain.length);
            try {
                for (Certificate cert : certificateChain) {
                    byte[] encoded = cert.getEncoded();
                    this.encodedChain.add(encoded);
                    this.length += 3 + encoded.length;
                }
            }
            catch (CertificateEncodingException e) {
                this.encodedChain = null;
                LOGGER.log(Level.SEVERE, "Could not encode certificate chain", e);
            }
        }
    }

    @Override
    public int getMessageLength() {
        return this.length;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        if (this.rawPublicKeyBytes == null && this.certificateChain != null) {
            sb.append("\t\tCertificate chain length: ").append(this.getMessageLength() - 3).append("\n");
            int index = 0;
            for (X509Certificate cert : this.certificateChain) {
                sb.append("\t\t\tCertificate Length: ").append(this.encodedChain.get(index).length).append("\n");
                sb.append("\t\t\tCertificate: ").append(cert).append("\n");
                ++index;
            }
        } else if (this.rawPublicKeyBytes != null && this.certificateChain == null) {
            sb.append("\t\tRaw Public Key: ");
            sb.append(this.getPublicKey().toString());
            sb.append("\n");
        }
        return sb.toString();
    }

    public X509Certificate[] getCertificateChain() {
        if (this.certificateChain != null) {
            return Arrays.copyOf(this.certificateChain, this.certificateChain.length);
        }
        return null;
    }

    private static Set<TrustAnchor> getTrustAnchors(X509Certificate[] trustedCertificates) {
        HashSet<TrustAnchor> result = new HashSet<TrustAnchor>();
        if (trustedCertificates != null) {
            for (X509Certificate cert : trustedCertificates) {
                result.add(new TrustAnchor(cert, null));
            }
        }
        return result;
    }

    public void verifyCertificate(X509Certificate[] trustedCertificates) throws HandshakeException {
        if (this.rawPublicKeyBytes == null) {
            Set<TrustAnchor> trustAnchors = CertificateMessage.getTrustAnchors(trustedCertificates);
            try {
                CertificateFactory certFactory = CertificateFactory.getInstance(CERTIFICATE_TYPE_X509);
                CertPath certPath = certFactory.generateCertPath(Arrays.asList(this.certificateChain));
                PKIXParameters params = new PKIXParameters(trustAnchors);
                params.setRevocationEnabled(false);
                CertPathValidator validator = CertPathValidator.getInstance("PKIX");
                validator.validate(certPath, params);
            }
            catch (GeneralSecurityException e) {
                if (LOGGER.isLoggable(Level.FINEST)) {
                    LOGGER.log(Level.FINEST, "Certificate validation failed", e);
                } else if (LOGGER.isLoggable(Level.FINE)) {
                    LOGGER.log(Level.FINE, "Certificate validation failed due to {0}", e.getMessage());
                }
                AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_CERTIFICATE, this.getPeer());
                throw new HandshakeException("Certificate chain could not be validated", alert);
            }
        }
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        if (this.rawPublicKeyBytes == null) {
            writer.write(this.getMessageLength() - 3, 24);
            for (byte[] encoded : this.encodedChain) {
                writer.write(encoded.length, 24);
                writer.writeBytes(encoded);
            }
        } else {
            writer.write(this.rawPublicKeyBytes.length, 24);
            writer.writeBytes(this.rawPublicKeyBytes);
        }
        return writer.toByteArray();
    }

    public static CertificateMessage fromByteArray(byte[] byteArray, boolean useRawPublicKey, InetSocketAddress peerAddress) throws HandshakeException {
        DatagramReader reader = new DatagramReader(byteArray);
        if (useRawPublicKey) {
            LOGGER.log(Level.FINER, "Parsing RawPublicKey CERTIFICATE message");
            int certificateLength = reader.read(24);
            byte[] rawPublicKey = reader.readBytes(certificateLength);
            return new CertificateMessage(rawPublicKey, peerAddress);
        }
        return CertificateMessage.readX509CertificateMessage(reader, peerAddress);
    }

    private static CertificateMessage readX509CertificateMessage(DatagramReader reader, InetSocketAddress peerAddress) throws HandshakeException {
        LOGGER.log(Level.FINER, "Parsing X.509 CERTIFICATE message");
        int certificateChainLength = reader.read(24);
        ArrayList<Certificate> certs = new ArrayList<Certificate>();
        CertificateFactory certificateFactory = null;
        while (certificateChainLength > 0) {
            int certificateLength = reader.read(24);
            byte[] certificate = reader.readBytes(certificateLength);
            certificateChainLength -= 3 + certificateLength;
            try {
                if (certificateFactory == null) {
                    certificateFactory = CertificateFactory.getInstance(CERTIFICATE_TYPE_X509);
                }
                certs.add(certificateFactory.generateCertificate(new ByteArrayInputStream(certificate)));
            }
            catch (CertificateException e) {
                throw new HandshakeException("Cannot parse X.509 certificate chain provided by peer", new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.BAD_CERTIFICATE, peerAddress), e);
            }
        }
        return new CertificateMessage(certs.toArray(new X509Certificate[certs.size()]), peerAddress);
    }

    public PublicKey getPublicKey() {
        PublicKey publicKey = null;
        if (this.rawPublicKeyBytes == null) {
            if (this.certificateChain != null && this.certificateChain.length > 0) {
                publicKey = this.certificateChain[0].getPublicKey();
            }
        } else {
            X509EncodedKeySpec publicKeySpec = new X509EncodedKeySpec(this.rawPublicKeyBytes);
            try {
                publicKey = KeyFactory.getInstance("EC").generatePublic(publicKeySpec);
            }
            catch (GeneralSecurityException e) {
                LOGGER.log(Level.SEVERE, "Could not reconstruct the peer's public key", e);
            }
        }
        return publicKey;
    }
}

