/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.nio.channels.ClosedByInterruptException;
import java.security.GeneralSecurityException;
import java.security.Principal;
import java.security.SecureRandom;
import java.security.cert.Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.crypto.Mac;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.californium.elements.Connector;
import org.eclipse.californium.elements.CorrelationContext;
import org.eclipse.californium.elements.DtlsCorrelationContext;
import org.eclipse.californium.elements.RawData;
import org.eclipse.californium.elements.RawDataChannel;
import org.eclipse.californium.scandium.DTLSConnectorConfig;
import org.eclipse.californium.scandium.ErrorHandler;
import org.eclipse.californium.scandium.config.DtlsConnectorConfig;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.ApplicationMessage;
import org.eclipse.californium.scandium.dtls.ClientHandshaker;
import org.eclipse.californium.scandium.dtls.ClientHello;
import org.eclipse.californium.scandium.dtls.CompressionMethod;
import org.eclipse.californium.scandium.dtls.Connection;
import org.eclipse.californium.scandium.dtls.ConnectionStore;
import org.eclipse.californium.scandium.dtls.ContentType;
import org.eclipse.californium.scandium.dtls.DTLSFlight;
import org.eclipse.californium.scandium.dtls.DTLSMessage;
import org.eclipse.californium.scandium.dtls.DTLSSession;
import org.eclipse.californium.scandium.dtls.DtlsHandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.HandshakeType;
import org.eclipse.californium.scandium.dtls.Handshaker;
import org.eclipse.californium.scandium.dtls.HelloRequest;
import org.eclipse.californium.scandium.dtls.HelloVerifyRequest;
import org.eclipse.californium.scandium.dtls.InMemoryConnectionStore;
import org.eclipse.californium.scandium.dtls.MaxFragmentLengthExtension;
import org.eclipse.californium.scandium.dtls.ProtocolVersion;
import org.eclipse.californium.scandium.dtls.Record;
import org.eclipse.californium.scandium.dtls.RecordLayer;
import org.eclipse.californium.scandium.dtls.ResumingClientHandshaker;
import org.eclipse.californium.scandium.dtls.ResumingServerHandshaker;
import org.eclipse.californium.scandium.dtls.ResumptionSupportingConnectionStore;
import org.eclipse.californium.scandium.dtls.ServerHandshaker;
import org.eclipse.californium.scandium.dtls.SessionAdapter;
import org.eclipse.californium.scandium.dtls.SessionListener;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.util.ByteArrayUtils;

public class DTLSConnector
implements Connector {
    private static final Logger LOGGER = Logger.getLogger(DTLSConnector.class.getCanonicalName());
    private static final int MAX_PLAINTEXT_FRAGMENT_LENGTH = 16384;
    private static final int MAX_CIPHERTEXT_EXPANSION = CipherSuite.TLS_PSK_WITH_AES_128_CBC_SHA256.getMaxCiphertextExpansion();
    private static final int MAX_DATAGRAM_BUFFER_SIZE = 16409 + MAX_CIPHERTEXT_EXPANSION;
    static final ThreadGroup SCANDIUM_THREAD_GROUP = new ThreadGroup("Californium/Scandium");
    private InetSocketAddress lastBindAddress;
    private int maximumTransmissionUnit = 1280;
    private int inboundDatagramBufferSize = MAX_DATAGRAM_BUFFER_SIZE;
    private Object cookieMacKeyLock = new Object();
    private long lastGenerationDate = System.currentTimeMillis();
    private SecretKey cookieMacKey = new SecretKeySpec(DTLSConnector.randomBytes(), "MAC");
    private final DtlsConnectorConfig config;
    private DatagramSocket socket;
    private ScheduledExecutorService timer;
    private Worker receiver;
    private Worker sender;
    private final ConnectionStore connectionStore;
    private final BlockingQueue<RawData> outboundMessages;
    private boolean running;
    private RawDataChannel messageHandler;
    private ErrorHandler errorHandler;

    public DTLSConnector(DtlsConnectorConfig configuration) {
        this(configuration, null);
    }

    public DTLSConnector(DtlsConnectorConfig configuration, ConnectionStore connectionStore) {
        if (configuration == null) {
            throw new NullPointerException("Configuration must not be null");
        }
        this.config = configuration;
        this.outboundMessages = new LinkedBlockingQueue<RawData>(this.config.getOutboundMessageBufferSize());
        this.connectionStore = connectionStore != null ? connectionStore : new InMemoryConnectionStore();
    }

    @Deprecated
    public DTLSConnector(InetSocketAddress address) {
        this(address, null);
    }

    @Deprecated
    public DTLSConnector(InetSocketAddress address, Certificate[] rootCertificates) {
        this(address, rootCertificates, null, null);
    }

    @Deprecated
    public DTLSConnector(InetSocketAddress address, Certificate[] rootCertificates, ConnectionStore connectionStore, DTLSConnectorConfig config) {
        DtlsConnectorConfig.Builder builder = new DtlsConnectorConfig.Builder(address);
        if (config != null) {
            builder.setMaxRetransmissions(config.getMaxRetransmit());
            builder.setRetransmissionTimeout(config.getRetransmissionTimeout());
            if (rootCertificates != null) {
                builder.setTrustStore(rootCertificates);
            }
            if (config.pskStore != null) {
                builder.setPskStore(config.pskStore);
            } else if (config.certChain != null) {
                builder.setIdentity(config.privateKey, config.certChain, config.sendRawKey);
            } else {
                builder.setIdentity(config.privateKey, config.publicKey);
            }
        }
        this.config = builder.build();
        this.outboundMessages = new LinkedBlockingQueue<RawData>(this.config.getOutboundMessageBufferSize());
        this.connectionStore = connectionStore != null ? connectionStore : new InMemoryConnectionStore();
    }

    public final void close(InetSocketAddress peerAddress) {
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null && connection.getEstablishedSession() != null) {
            this.terminateConnection(connection, new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY, peerAddress), connection.getEstablishedSession());
        }
    }

    public final synchronized void start() throws IOException {
        this.start(this.config.getAddress());
    }

    final synchronized void restart() throws IOException {
        if (this.lastBindAddress == null) {
            throw new IllegalStateException("Connector has never been started before");
        }
        this.start(this.lastBindAddress);
    }

    private void start(InetSocketAddress bindAddress) throws IOException {
        NetworkInterface ni;
        if (this.running) {
            return;
        }
        this.timer = Executors.newSingleThreadScheduledExecutor(new ThreadFactory(){
            private final AtomicInteger index = new AtomicInteger(1);

            @Override
            public Thread newThread(Runnable r) {
                Thread ret = new Thread(SCANDIUM_THREAD_GROUP, r, "DTLS RetransmitTask " + this.index.getAndIncrement(), 0L);
                ret.setDaemon(true);
                ret.setPriority(5);
                return ret;
            }
        });
        this.socket = new DatagramSocket(null);
        this.socket.setReuseAddress(true);
        this.socket.bind(bindAddress);
        if (!(this.lastBindAddress == null || this.socket.getLocalAddress().equals(this.lastBindAddress.getAddress()) && this.socket.getLocalPort() == this.lastBindAddress.getPort())) {
            if (this.connectionStore instanceof ResumptionSupportingConnectionStore) {
                ((ResumptionSupportingConnectionStore)this.connectionStore).markAllAsResumptionRequired();
            } else {
                this.connectionStore.clear();
            }
        }
        if ((ni = NetworkInterface.getByInetAddress(bindAddress.getAddress())) != null && ni.getMTU() > 0) {
            this.maximumTransmissionUnit = ni.getMTU();
        } else {
            LOGGER.config("Cannot determine MTU of network interface, using minimum MTU [1280] of IPv6 instead");
            this.maximumTransmissionUnit = 1280;
        }
        if (this.config.getMaxFragmentLengthCode() != null) {
            MaxFragmentLengthExtension.Length lengthCode = MaxFragmentLengthExtension.Length.fromCode(this.config.getMaxFragmentLengthCode());
            this.inboundDatagramBufferSize = lengthCode.length() + MAX_CIPHERTEXT_EXPANSION + 25;
        }
        this.lastBindAddress = new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
        this.running = true;
        this.sender = new Worker("DTLS-Sender-" + this.lastBindAddress){

            @Override
            public void doWork() throws Exception {
                DTLSConnector.this.sendNextMessageOverNetwork();
            }
        };
        this.receiver = new Worker("DTLS-Receiver-" + this.lastBindAddress){

            @Override
            public void doWork() throws Exception {
                DTLSConnector.this.receiveNextDatagramFromNetwork();
            }
        };
        this.receiver.start();
        this.sender.start();
        LOGGER.log(Level.INFO, "DTLS connector listening on [{0}] with MTU [{1}] using (inbound) datagram buffer size [{2} bytes]", new Object[]{this.lastBindAddress, this.maximumTransmissionUnit, this.inboundDatagramBufferSize});
    }

    public final synchronized void forceResumeSessionFor(InetSocketAddress peer) {
        Connection peerConnection = this.connectionStore.get(peer);
        if (peerConnection != null && peerConnection.getEstablishedSession() != null) {
            peerConnection.setResumptionRequired(true);
        }
    }

    final synchronized void releaseSocket() {
        this.running = false;
        this.sender.interrupt();
        this.outboundMessages.clear();
        if (this.socket != null) {
            this.socket.close();
            this.socket = null;
        }
        this.maximumTransmissionUnit = 0;
    }

    public final synchronized void stop() {
        if (!this.running) {
            return;
        }
        LOGGER.log(Level.INFO, "Stopping DTLS connector on [{0}]", this.lastBindAddress);
        this.timer.shutdownNow();
        this.releaseSocket();
    }

    public final synchronized void destroy() {
        this.stop();
        this.connectionStore.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void receiveNextDatagramFromNetwork() throws IOException {
        byte[] buffer = new byte[this.inboundDatagramBufferSize];
        DatagramPacket packet = new DatagramPacket(buffer, buffer.length);
        DatagramSocket datagramSocket = this.socket;
        synchronized (datagramSocket) {
            this.socket.receive(packet);
        }
        if (packet.getLength() == 0) {
            return;
        }
        InetSocketAddress peerAddress = new InetSocketAddress(packet.getAddress(), packet.getPort());
        byte[] data = Arrays.copyOfRange(packet.getData(), packet.getOffset(), packet.getLength());
        List<Record> records = Record.fromByteArray(data, peerAddress);
        LOGGER.log(Level.FINER, "Received {0} DTLS records using a {1} byte datagram buffer", new Object[]{records.size(), this.inboundDatagramBufferSize});
        for (Record record : records) {
            try {
                LOGGER.log(Level.FINEST, "Received DTLS record of type [{0}]", (Object)record.getType());
                switch (record.getType()) {
                    case APPLICATION_DATA: {
                        this.processApplicationDataRecord(record);
                        break;
                    }
                    case ALERT: {
                        this.processAlertRecord(record);
                        break;
                    }
                    case CHANGE_CIPHER_SPEC: {
                        this.processChangeCipherSpecRecord(record);
                        break;
                    }
                    case HANDSHAKE: {
                        this.processHandshakeRecord(record);
                        break;
                    }
                    default: {
                        LOGGER.log(Level.FINE, "Discarding record of unsupported type [{0}] from peer [{1}]", new Object[]{record.getType(), record.getPeerAddress()});
                        break;
                    }
                }
            }
            catch (RuntimeException e) {
                LOGGER.log(Level.INFO, String.format("Unexpected error occurred while processing record from peer [%s]", peerAddress), e);
                this.terminateConnection(peerAddress, e, AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR);
                break;
            }
        }
    }

    private void terminateOngoingHandshake(InetSocketAddress peerAddress, Throwable cause, AlertMessage.AlertDescription description) {
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null && connection.hasOngoingHandshake()) {
            if (LOGGER.isLoggable(Level.FINEST)) {
                LOGGER.log(Level.FINEST, String.format("Aborting handshake with peer [%s]: ", peerAddress), cause);
            } else if (LOGGER.isLoggable(Level.INFO)) {
                LOGGER.log(Level.INFO, "Aborting handshake with peer [{0}]: {1}", new Object[]{peerAddress, cause.getMessage()});
            }
            DTLSSession session = connection.getOngoingHandshake().getSession();
            AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, description, peerAddress);
            if (!connection.hasEstablishedSession()) {
                this.terminateConnection(connection, alert, session);
            } else {
                this.send(alert, session);
                connection.terminateOngoingHandshake();
            }
        }
    }

    private void terminateConnection(InetSocketAddress peerAddress) {
        if (peerAddress != null) {
            this.terminateConnection(this.connectionStore.get(peerAddress));
        }
    }

    private void terminateConnection(Connection connection) {
        if (connection != null) {
            connection.cancelPendingFlight();
            this.connectionClosed(connection.getPeerAddress());
        }
    }

    private void terminateConnection(InetSocketAddress peerAddress, Throwable cause, AlertMessage.AlertLevel level, AlertMessage.AlertDescription description) {
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection != null) {
            if (connection.hasEstablishedSession()) {
                this.terminateConnection(connection, new AlertMessage(level, description, peerAddress), connection.getEstablishedSession());
            } else if (connection.hasOngoingHandshake()) {
                this.terminateConnection(connection, new AlertMessage(level, description, peerAddress), connection.getOngoingHandshake().getSession());
            }
        }
    }

    private void terminateConnection(Connection connection, AlertMessage alert, DTLSSession session) {
        if (alert != null && session == null) {
            throw new IllegalArgumentException("Session must not be NULL if alert message is to be sent");
        }
        connection.cancelPendingFlight();
        if (alert == null) {
            LOGGER.log(Level.FINE, "Terminating connection with peer [{0}]", connection.getPeerAddress());
        } else {
            LOGGER.log(Level.FINE, "Terminating connection with peer [{0}], reason [{1}]", new Object[]{connection.getPeerAddress(), alert.getDescription()});
            this.send(alert, session);
        }
        this.connectionClosed(connection.getPeerAddress());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void processApplicationDataRecord(Record record) {
        Connection connection = this.connectionStore.get(record.getPeerAddress());
        if (connection != null && connection.hasEstablishedSession()) {
            DTLSSession session;
            DTLSSession dTLSSession = session = connection.getEstablishedSession();
            synchronized (dTLSSession) {
                if (session.isRecordProcessable(record.getEpoch(), record.getSequenceNumber())) {
                    try {
                        record.setSession(session);
                        ApplicationMessage message = (ApplicationMessage)record.getFragment();
                        connection.handshakeCompleted(record.getPeerAddress());
                        session.markRecordAsRead(record.getEpoch(), record.getSequenceNumber());
                        this.handleApplicationMessage(message, session);
                    }
                    catch (GeneralSecurityException | HandshakeException e) {
                        DTLSConnector.discardRecord(record, e);
                    }
                } else {
                    LOGGER.log(Level.FINER, "Discarding duplicate APPLICATION_DATA record received from peer [{0}]", record.getPeerAddress());
                }
            }
        } else {
            LOGGER.log(Level.FINER, "Discarding APPLICATION_DATA record received from peer [{0}] without an active session", new Object[]{record.getPeerAddress()});
        }
    }

    private void handleApplicationMessage(ApplicationMessage message, DTLSSession session) {
        if (this.messageHandler != null) {
            DtlsCorrelationContext context = new DtlsCorrelationContext(session.getSessionIdentifier().toString(), String.valueOf(session.getReadEpoch()), session.getReadStateCipher());
            this.messageHandler.receiveData(RawData.inbound((byte[])message.getData(), (InetSocketAddress)message.getPeer(), (Principal)session.getPeerIdentity(), (CorrelationContext)context, (boolean)false));
        }
    }

    private void processAlertRecord(Record record) {
        Connection connection = this.connectionStore.get(record.getPeerAddress());
        if (connection == null) {
            LOGGER.log(Level.FINER, "Discarding ALERT record from [{0}] received without existing connection", record.getPeerAddress());
        } else {
            this.processAlertRecord(record, connection);
        }
    }

    private void processAlertRecord(Record record, Connection connection) {
        if (connection.hasEstablishedSession() && connection.getEstablishedSession().getReadEpoch() == record.getEpoch()) {
            this.processAlertRecord(record, connection, connection.getEstablishedSession());
        } else if (connection.hasOngoingHandshake() && connection.getOngoingHandshake().getSession().getReadEpoch() == record.getEpoch()) {
            this.processAlertRecord(record, connection, connection.getOngoingHandshake().getSession());
        } else {
            LOGGER.log(Level.FINER, "Epoch of ALERT record [epoch=%d] from [%s] does not match expected epoch(s), discarding ...", new Object[]{record.getEpoch(), record.getPeerAddress()});
        }
    }

    private void processAlertRecord(Record record, Connection connection, DTLSSession session) {
        record.setSession(session);
        try {
            AlertMessage alert = (AlertMessage)record.getFragment();
            LOGGER.log(Level.FINEST, "Processing {0} ALERT from [{1}]: {2}", new Object[]{alert.getLevel(), alert.getPeer(), alert.getDescription()});
            if (AlertMessage.AlertDescription.CLOSE_NOTIFY.equals((Object)alert.getDescription())) {
                this.terminateConnection(connection, new AlertMessage(AlertMessage.AlertLevel.WARNING, AlertMessage.AlertDescription.CLOSE_NOTIFY, alert.getPeer()), session);
            } else if (AlertMessage.AlertLevel.FATAL.equals((Object)alert.getLevel())) {
                this.terminateConnection(connection);
            }
            if (this.errorHandler != null) {
                this.errorHandler.onError(alert.getPeer(), alert.getLevel(), alert.getDescription());
            }
        }
        catch (GeneralSecurityException | HandshakeException e) {
            DTLSConnector.discardRecord(record, e);
        }
    }

    private void processChangeCipherSpecRecord(Record record) {
        Connection connection = this.connectionStore.get(record.getPeerAddress());
        if (connection != null && connection.hasOngoingHandshake()) {
            try {
                connection.getOngoingHandshake().processMessage(record);
            }
            catch (HandshakeException e) {
                this.handleExceptionDuringHandshake(e, e.getAlert().getLevel(), e.getAlert().getDescription(), record);
            }
        } else {
            LOGGER.log(Level.FINE, "Received CHANGE_CIPHER_SPEC record from peer [{0}] with no handshake going on", record.getPeerAddress());
        }
    }

    private void processHandshakeRecord(Record record) {
        LOGGER.log(Level.FINE, "Received {0} record from peer [{1}]", new Object[]{record.getType(), record.getPeerAddress()});
        Connection con = this.connectionStore.get(record.getPeerAddress());
        try {
            if (con == null) {
                this.processHandshakeRecordWithoutConnection(record);
            } else {
                this.processHandshakeRecordWithConnection(record, con);
            }
        }
        catch (HandshakeException e) {
            this.handleExceptionDuringHandshake(e, e.getAlert().getLevel(), e.getAlert().getDescription(), record);
        }
    }

    private void processHandshakeRecordWithoutConnection(Record record) throws HandshakeException {
        if (record.getEpoch() > 0) {
            LOGGER.log(Level.FINE, "Discarding unexpected handshake message [epoch={0}] received from peer [{1}] without existing connection", new Object[]{record.getEpoch(), record.getPeerAddress()});
        } else {
            try {
                HandshakeMessage handshakeMessage = (HandshakeMessage)record.getFragment();
                if (HandshakeType.CLIENT_HELLO.equals((Object)handshakeMessage.getMessageType())) {
                    this.processClientHello((ClientHello)handshakeMessage, record);
                } else {
                    LOGGER.log(Level.FINE, "Discarding unexpected {0} message from peer [{1}]", new Object[]{handshakeMessage.getMessageType(), handshakeMessage.getPeer()});
                }
            }
            catch (GeneralSecurityException e) {
                DTLSConnector.discardRecord(record, e);
            }
        }
    }

    private void processHandshakeRecordWithConnection(Record record, Connection connection) throws HandshakeException {
        if (connection.hasOngoingHandshake() && connection.getOngoingHandshake().getSession().getReadEpoch() == record.getEpoch()) {
            record.setSession(connection.getOngoingHandshake().getSession());
        } else if (connection.hasEstablishedSession() && connection.getEstablishedSession().getReadEpoch() == record.getEpoch()) {
            record.setSession(connection.getEstablishedSession());
        } else if (record.getEpoch() != 0) {
            LOGGER.log(Level.FINE, "Discarding HANDSHAKE message [epoch={0}] from peer [{1}] which does not match expected epoch(s)", new Object[]{record.getEpoch(), record.getPeerAddress()});
            return;
        }
        try {
            HandshakeMessage handshakeMessage = (HandshakeMessage)record.getFragment();
            this.processDecryptedHandshakeMessage(handshakeMessage, record, connection);
        }
        catch (GeneralSecurityException e) {
            DTLSConnector.discardRecord(record, e);
        }
    }

    private void processDecryptedHandshakeMessage(HandshakeMessage handshakeMessage, Record record, Connection connection) throws HandshakeException {
        switch (handshakeMessage.getMessageType()) {
            case CLIENT_HELLO: {
                this.processClientHello((ClientHello)handshakeMessage, record, connection);
                break;
            }
            case HELLO_REQUEST: {
                this.processHelloRequest((HelloRequest)handshakeMessage, connection);
                break;
            }
            default: {
                this.processOngoingHandshakeMessage(handshakeMessage, record, connection);
            }
        }
    }

    private void processOngoingHandshakeMessage(HandshakeMessage message, Record record, Connection connection) throws HandshakeException {
        if (connection.hasOngoingHandshake()) {
            connection.getOngoingHandshake().processMessage(record);
        } else {
            LOGGER.log(Level.FINE, "Discarding {0} message received from peer [{1}] with no handshake going on", new Object[]{message.getMessageType(), message.getPeer()});
        }
    }

    private void processHelloRequest(HelloRequest helloRequest, Connection connection) throws HandshakeException {
        if (connection.hasOngoingHandshake()) {
            LOGGER.log(Level.FINE, "Ignoring {0} received from [{1}] while already in an ongoing handshake with peer", new Object[]{helloRequest.getMessageType(), helloRequest.getPeer()});
        } else {
            DTLSSession session = connection.getEstablishedSession();
            if (session == null) {
                session = new DTLSSession(helloRequest.getPeer(), true);
            }
            ClientHandshaker handshaker = new ClientHandshaker(session, this.getRecordLayerForPeer(connection), connection, this.config, this.maximumTransmissionUnit);
            ((Handshaker)handshaker).startHandshake();
        }
    }

    private void processClientHello(ClientHello clientHello, Record record) throws HandshakeException {
        if (LOGGER.isLoggable(Level.FINE)) {
            StringBuilder msg = new StringBuilder("Processing CLIENT_HELLO from peer [").append(record.getPeerAddress()).append("]");
            if (LOGGER.isLoggable(Level.FINEST)) {
                msg.append(":").append(System.lineSeparator()).append(record);
            }
            LOGGER.fine(msg.toString());
        }
        if (this.isClientInControlOfSourceIpAddress(clientHello, record)) {
            if (clientHello.hasSessionId()) {
                this.resumeExistingSession(clientHello, record);
            } else {
                this.startNewHandshake(clientHello, record);
            }
        }
    }

    private void processClientHello(ClientHello clientHello, Record record, Connection connection) throws HandshakeException {
        if (LOGGER.isLoggable(Level.FINE)) {
            StringBuilder msg = new StringBuilder("Processing CLIENT_HELLO from peer [").append(record.getPeerAddress()).append("]");
            if (LOGGER.isLoggable(Level.FINEST)) {
                msg.append(":").append(System.lineSeparator()).append(record);
            }
            LOGGER.fine(msg.toString());
        }
        if (this.isClientInControlOfSourceIpAddress(clientHello, record)) {
            if (DTLSConnector.isHandshakeAlreadyStartedForMessage(clientHello, connection)) {
                this.processOngoingHandshakeMessage(clientHello, record, connection);
            } else if (clientHello.hasSessionId()) {
                this.resumeExistingSession(clientHello, record);
            } else {
                this.terminateConnection(connection);
                this.startNewHandshake(clientHello, record);
            }
        }
    }

    private static boolean isHandshakeAlreadyStartedForMessage(ClientHello clientHello, Connection connection) {
        return connection != null && connection.hasOngoingHandshake() && connection.getOngoingHandshake().hasBeenStartedByMessage(clientHello);
    }

    private boolean isClientInControlOfSourceIpAddress(ClientHello clientHello, Record record) {
        byte[] expectedCookie = this.generateCookie(clientHello);
        if (Arrays.equals(expectedCookie, clientHello.getCookie())) {
            return true;
        }
        this.sendHelloVerify(clientHello, record, expectedCookie);
        return false;
    }

    private void startNewHandshake(ClientHello clientHello, Record record) throws HandshakeException {
        Connection peerConnection = new Connection(record.getPeerAddress());
        this.connectionStore.put(peerConnection);
        DTLSSession newSession = new DTLSSession(record.getPeerAddress(), false, record.getSequenceNumber());
        ServerHandshaker handshaker = new ServerHandshaker(clientHello.getMessageSeq(), newSession, this.getRecordLayerForPeer(peerConnection), (SessionListener)peerConnection, this.config, this.maximumTransmissionUnit);
        handshaker.processMessage(record);
    }

    private void resumeExistingSession(ClientHello clientHello, Record record) throws HandshakeException {
        LOGGER.log(Level.FINER, "Client [{0}] wants to resume session with ID [{1}]", new Object[]{clientHello.getPeer(), clientHello.getSessionId()});
        final Connection previousConnection = this.connectionStore.find(clientHello.getSessionId());
        if (previousConnection != null && previousConnection.hasEstablishedSession()) {
            DTLSSession resumableSession = new DTLSSession(record.getPeerAddress(), previousConnection.getEstablishedSession(), record.getSequenceNumber());
            Connection peerConnection = new Connection(record.getPeerAddress());
            ResumingServerHandshaker handshaker = new ResumingServerHandshaker(clientHello.getMessageSeq(), resumableSession, this.getRecordLayerForPeer(peerConnection), (SessionListener)peerConnection, this.config, this.maximumTransmissionUnit);
            if (!previousConnection.getPeerAddress().equals(peerConnection.getPeerAddress())) {
                handshaker.addSessionListener(new SessionAdapter(){

                    @Override
                    public void sessionEstablished(Handshaker handshaker, DTLSSession establishedSession) throws HandshakeException {
                        LOGGER.log(Level.FINER, "Discarding existing connection to [{0}] after successful resumption of session [ID={1}] by peer [{2}]", new Object[]{previousConnection.getPeerAddress(), establishedSession.getSessionIdentifier(), establishedSession.getPeer()});
                        DTLSConnector.this.terminateConnection(previousConnection);
                    }
                });
            } else {
                this.terminateConnection(previousConnection);
            }
            this.connectionStore.put(peerConnection);
            handshaker.processMessage(record);
        } else {
            LOGGER.log(Level.FINER, "Client [{0}] tries to resume non-existing session [ID={1}], performing full handshake instead ...", new Object[]{clientHello.getPeer(), clientHello.getSessionId()});
            this.terminateConnection(clientHello.getPeer());
            this.startNewHandshake(clientHello, record);
        }
    }

    private void sendHelloVerify(ClientHello clientHello, Record record, byte[] expectedCookie) {
        LOGGER.log(Level.FINER, "Verifying client IP address [{0}] using HELLO_VERIFY_REQUEST", record.getPeerAddress());
        HelloVerifyRequest msg = new HelloVerifyRequest(new ProtocolVersion(), expectedCookie, record.getPeerAddress());
        msg.setMessageSeq(clientHello.getMessageSeq());
        Record helloVerify = new Record(ContentType.HANDSHAKE, 0, record.getSequenceNumber(), (DTLSMessage)msg, record.getPeerAddress());
        this.sendRecord(helloVerify);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private SecretKey getMacKeyForCookies() {
        Object object = this.cookieMacKeyLock;
        synchronized (object) {
            if (System.currentTimeMillis() - this.lastGenerationDate > TimeUnit.MINUTES.toMillis(5L)) {
                this.cookieMacKey = new SecretKeySpec(DTLSConnector.randomBytes(), "MAC");
                this.lastGenerationDate = System.currentTimeMillis();
            }
            return this.cookieMacKey;
        }
    }

    private byte[] generateCookie(ClientHello clientHello) {
        try {
            Mac hmac = Mac.getInstance("HmacSHA256");
            hmac.init(this.getMacKeyForCookies());
            hmac.update(clientHello.getPeer().toString().getBytes());
            hmac.update((byte)clientHello.getClientVersion().getMajor());
            hmac.update((byte)clientHello.getClientVersion().getMinor());
            hmac.update(clientHello.getRandom().getRandomBytes());
            hmac.update(clientHello.getSessionId().getId());
            hmac.update(CipherSuite.listToByteArray(clientHello.getCipherSuites()));
            hmac.update(CompressionMethod.listToByteArray(clientHello.getCompressionMethods()));
            return hmac.doFinal();
        }
        catch (GeneralSecurityException e) {
            throw new DtlsHandshakeException("Cannot compute cookie for peer", AlertMessage.AlertDescription.INTERNAL_ERROR, AlertMessage.AlertLevel.FATAL, clientHello.getPeer(), e);
        }
    }

    void send(AlertMessage alert, DTLSSession session) {
        if (alert == null) {
            throw new IllegalArgumentException("Alert must not be NULL");
        }
        if (session == null) {
            throw new IllegalArgumentException("Session must not be NULL");
        }
        try {
            this.sendRecord(new Record(ContentType.ALERT, session.getWriteEpoch(), session.getSequenceNumber(), (DTLSMessage)alert, session));
        }
        catch (GeneralSecurityException e) {
            LOGGER.log(Level.FINE, String.format("Cannot create ALERT message for peer [%s]", session.getPeer()), e);
        }
    }

    public final void send(RawData msg) {
        boolean queueFull;
        if (msg == null) {
            throw new NullPointerException("Message must not be null");
        }
        if (msg.getBytes().length > 16384) {
            throw new IllegalArgumentException("Message data must not exceed 16384 bytes");
        }
        boolean bl = queueFull = !this.outboundMessages.offer(msg);
        if (queueFull) {
            LOGGER.log(Level.WARNING, "Outbound message queue is full! Dropping outbound message to peer [{0}]", msg.getInetSocketAddress());
        }
    }

    private void sendNextMessageOverNetwork() throws HandshakeException {
        try {
            RawData message = this.outboundMessages.take();
            this.sendMessage(message);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    private void sendMessage(RawData message) throws HandshakeException {
        DTLSSession session;
        InetSocketAddress peerAddress = message.getInetSocketAddress();
        LOGGER.log(Level.FINER, "Sending application layer message to peer [{0}]", peerAddress);
        Connection connection = this.connectionStore.get(peerAddress);
        if (connection == null) {
            connection = new Connection(peerAddress);
            this.connectionStore.put(connection);
        }
        if ((session = connection.getEstablishedSession()) == null) {
            ClientHandshaker handshaker = new ClientHandshaker(new DTLSSession(peerAddress, true), this.getRecordLayerForPeer(connection), connection, this.config, this.maximumTransmissionUnit);
            handshaker.addSessionListener(this.newDeferredMessageSender(message));
            ((Handshaker)handshaker).startHandshake();
        } else if (connection.isResumptionRequired()) {
            DTLSSession resumableSession = new DTLSSession(peerAddress, session, 0L);
            Connection newConnection = new Connection(peerAddress);
            this.terminateConnection(connection, null, null);
            this.connectionStore.put(newConnection);
            ResumingClientHandshaker handshaker = new ResumingClientHandshaker(resumableSession, this.getRecordLayerForPeer(newConnection), newConnection, this.config, this.maximumTransmissionUnit);
            handshaker.addSessionListener(this.newDeferredMessageSender(message));
            ((Handshaker)handshaker).startHandshake();
        } else {
            this.sendMessage(message, session);
        }
    }

    private void sendMessage(RawData message, DTLSSession session) {
        try {
            Record record = new Record(ContentType.APPLICATION_DATA, session.getWriteEpoch(), session.getSequenceNumber(), (DTLSMessage)new ApplicationMessage(message.getBytes(), message.getInetSocketAddress()), session);
            if (message.getMessageCallback() != null) {
                DtlsCorrelationContext ctx = new DtlsCorrelationContext(session.getSessionIdentifier().toString(), String.valueOf(session.getWriteEpoch()), session.getWriteStateCipher());
                message.getMessageCallback().onContextEstablished((CorrelationContext)ctx);
            }
            this.sendRecord(record);
        }
        catch (GeneralSecurityException e) {
            LOGGER.log(Level.FINE, String.format("Cannot send APPLICATION record to peer [%s]", message.getInetSocketAddress()), e);
        }
    }

    private SessionListener newDeferredMessageSender(final RawData message) {
        return new SessionListener(){

            @Override
            public void sessionEstablished(Handshaker handshaker, DTLSSession establishedSession) throws HandshakeException {
                LOGGER.log(Level.FINE, "Session with [{0}] established, now sending deferred message", establishedSession.getPeer());
                DTLSConnector.this.sendMessage(message, establishedSession);
            }

            @Override
            public void handshakeStarted(Handshaker handshaker) throws HandshakeException {
            }

            @Override
            public void handshakeCompleted(InetSocketAddress peer) {
            }
        };
    }

    public final DTLSSession getSessionByAddress(InetSocketAddress address) {
        if (address == null) {
            return null;
        }
        Connection connection = this.connectionStore.get(address);
        if (connection != null) {
            return connection.getEstablishedSession();
        }
        return null;
    }

    private void sendHandshakeFlight(DTLSFlight flight, Connection connection) {
        if (flight != null) {
            connection.cancelPendingFlight();
            if (flight.isRetransmissionNeeded()) {
                connection.setPendingFlight(flight);
                this.scheduleRetransmission(flight);
            }
            this.sendFlight(flight);
        }
    }

    private void sendFlight(DTLSFlight flight) {
        byte[] payload = new byte[]{};
        int maxDatagramSize = this.maximumTransmissionUnit;
        if (flight.getSession() != null) {
            maxDatagramSize = flight.getSession().getMaxDatagramSize();
        }
        ArrayList<DatagramPacket> datagrams = new ArrayList<DatagramPacket>();
        try {
            for (Record record : flight.getMessages()) {
                byte[] recordBytes = record.toByteArray();
                if (recordBytes.length > maxDatagramSize) {
                    LOGGER.log(Level.INFO, "{0} record of {1} bytes for peer [{2}] exceeds max. datagram size [{3}], discarding...", new Object[]{record.getType(), recordBytes.length, record.getPeerAddress(), maxDatagramSize});
                    continue;
                }
                LOGGER.log(Level.FINEST, "Sending record of {2} bytes to peer [{0}]:\n{1}", new Object[]{flight.getPeerAddress(), record, recordBytes.length});
                if (payload.length + recordBytes.length > maxDatagramSize) {
                    DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
                    datagrams.add(datagram);
                    payload = new byte[]{};
                }
                payload = ByteArrayUtils.concatenate(payload, recordBytes);
            }
            DatagramPacket datagram = new DatagramPacket(payload, payload.length, flight.getPeerAddress().getAddress(), flight.getPeerAddress().getPort());
            datagrams.add(datagram);
            LOGGER.log(Level.FINER, "Sending flight of {0} message(s) to peer [{1}] using {2} datagram(s) of max. {3} bytes", new Object[]{flight.getMessages().size(), flight.getPeerAddress(), datagrams.size(), maxDatagramSize});
            for (DatagramPacket datagramPacket : datagrams) {
                this.sendDatagram(datagramPacket);
            }
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Could not send datagram", e);
        }
    }

    private void sendRecord(Record record) {
        try {
            byte[] recordBytes = record.toByteArray();
            DatagramPacket datagram = new DatagramPacket(recordBytes, recordBytes.length, record.getPeerAddress());
            this.sendDatagram(datagram);
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Could not send record", e);
        }
    }

    private void sendDatagram(DatagramPacket datagramPacket) throws IOException {
        if (!this.socket.isClosed()) {
            this.socket.send(datagramPacket);
        } else {
            LOGGER.log(Level.FINE, "Socket [{0}] is closed, discarding packet ...", this.config.getAddress());
        }
    }

    private void handleTimeout(DTLSFlight flight) {
        int max = this.config.getMaxRetransmissions();
        if (flight.getTries() < max) {
            LOGGER.log(Level.FINE, "Re-transmitting flight for [{0}], [{1}] retransmissions left", new Object[]{flight.getPeerAddress(), max - flight.getTries() - 1});
            try {
                flight.incrementTries();
                flight.setNewSequenceNumbers();
                this.sendFlight(flight);
                this.scheduleRetransmission(flight);
            }
            catch (GeneralSecurityException e) {
                LOGGER.log(Level.INFO, String.format("Cannot retransmit flight to peer [%s]", flight.getPeerAddress()), e);
            }
        } else {
            LOGGER.log(Level.FINE, "Flight for [{0}] has reached maximum no. [{1}] of retransmissions, discarding ...", new Object[]{flight.getPeerAddress(), max});
        }
    }

    private void scheduleRetransmission(DTLSFlight flight) {
        flight.cancelRetransmission();
        if (flight.isRetransmissionNeeded()) {
            if (flight.getTimeout() == 0) {
                flight.setTimeout(this.config.getRetransmissionTimeout());
            } else {
                flight.incrementTimeout();
            }
            ScheduledFuture<?> f = this.timer.schedule(new RetransmitTask(flight), (long)flight.getTimeout(), TimeUnit.MILLISECONDS);
            flight.setRetransmitTask(f);
        }
    }

    public final int getMaximumTransmissionUnit() {
        return this.maximumTransmissionUnit;
    }

    public final int getMaximumFragmentLength(InetSocketAddress peer) {
        Connection con = this.connectionStore.get(peer);
        if (con != null && con.getEstablishedSession() != null) {
            return con.getEstablishedSession().getMaxFragmentLength();
        }
        return this.maximumTransmissionUnit - 53;
    }

    public final InetSocketAddress getAddress() {
        if (this.socket == null) {
            return this.config.getAddress();
        }
        return new InetSocketAddress(this.socket.getLocalAddress(), this.socket.getLocalPort());
    }

    public final boolean isRunning() {
        return this.running;
    }

    private RecordLayer getRecordLayerForPeer(final Connection connection) {
        return new RecordLayer(){

            @Override
            public void sendRecord(Record record) {
                this.sendRecord(record);
            }

            @Override
            public void sendFlight(DTLSFlight flight) {
                DTLSConnector.this.sendHandshakeFlight(flight, connection);
            }
        };
    }

    public void setRawDataReceiver(RawDataChannel messageHandler) {
        this.messageHandler = messageHandler;
    }

    public final void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    private void connectionClosed(InetSocketAddress peerAddress) {
        if (peerAddress != null) {
            this.connectionStore.remove(peerAddress);
        }
    }

    private static byte[] randomBytes() {
        SecureRandom rng = new SecureRandom();
        byte[] result = new byte[32];
        rng.nextBytes(result);
        return result;
    }

    private void handleExceptionDuringHandshake(Throwable cause, AlertMessage.AlertLevel level, AlertMessage.AlertDescription description, Record record) {
        if (AlertMessage.AlertLevel.FATAL.equals((Object)level)) {
            this.terminateOngoingHandshake(record.getPeerAddress(), cause, description);
        } else {
            DTLSConnector.discardRecord(record, cause);
        }
    }

    private static void discardRecord(Record record, Throwable cause) {
        if (LOGGER.isLoggable(Level.FINEST)) {
            LOGGER.log(Level.FINEST, String.format("Discarding %s record from peer [%s]: ", new Object[]{record.getType(), record.getPeerAddress()}), cause);
        } else if (LOGGER.isLoggable(Level.FINE)) {
            LOGGER.log(Level.FINE, "Discarding {0} record from peer [{1}]: {2}", new Object[]{record.getType(), record.getPeerAddress(), cause.getMessage()});
        }
    }

    private abstract class Worker
    extends Thread {
        protected Worker(String name) {
            super(SCANDIUM_THREAD_GROUP, name);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            try {
                LOGGER.log(Level.CONFIG, "Starting worker thread [{0}]", this.getName());
                while (DTLSConnector.this.running) {
                    try {
                        this.doWork();
                    }
                    catch (ClosedByInterruptException e) {
                        LOGGER.log(Level.CONFIG, "Worker thread [{0}] has been interrupted", this.getName());
                    }
                    catch (Exception e) {
                        if (!DTLSConnector.this.running) continue;
                        LOGGER.log(Level.FINE, "Exception thrown by worker thread [" + this.getName() + "]", e);
                    }
                }
            }
            finally {
                LOGGER.log(Level.CONFIG, "Worker thread [{0}] has terminated", this.getName());
            }
        }

        protected abstract void doWork() throws Exception;
    }

    private class RetransmitTask
    implements Runnable {
        private DTLSFlight flight;

        RetransmitTask(DTLSFlight flight) {
            this.flight = flight;
        }

        @Override
        public void run() {
            DTLSConnector.this.handleTimeout(this.flight);
        }
    }
}

