/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.net.InetSocketAddress;
import java.security.GeneralSecurityException;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.interfaces.ECPublicKey;
import java.security.spec.ECParameterSpec;
import java.security.spec.ECPoint;
import java.security.spec.ECPublicKeySpec;
import java.util.Arrays;
import org.eclipse.californium.elements.util.DatagramReader;
import org.eclipse.californium.elements.util.DatagramWriter;
import org.eclipse.californium.elements.util.StringUtil;
import org.eclipse.californium.scandium.dtls.AlertMessage;
import org.eclipse.californium.scandium.dtls.HandshakeException;
import org.eclipse.californium.scandium.dtls.HandshakeMessage;
import org.eclipse.californium.scandium.dtls.Random;
import org.eclipse.californium.scandium.dtls.ServerKeyExchange;
import org.eclipse.californium.scandium.dtls.SignatureAndHashAlgorithm;
import org.eclipse.californium.scandium.dtls.cipher.ECDHECryptography;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ECDHServerKeyExchange
extends ServerKeyExchange {
    private static final String MSG_UNKNOWN_CURVE_TYPE = "Unknown curve type [{}]";
    private static final Logger LOGGER = LoggerFactory.getLogger((String)ECDHServerKeyExchange.class.getCanonicalName());
    private static final int CURVE_TYPE_BITS = 8;
    private static final int NAMED_CURVE_BITS = 16;
    private static final int PUBLIC_LENGTH_BITS = 8;
    private static final int HASH_ALGORITHM_BITS = 8;
    private static final int SIGNATURE_ALGORITHM_BITS = 8;
    private static final int SIGNATURE_LENGTH_BITS = 16;
    private static final String KEYPAIR_GENERATOR_INSTANCE = "EC";
    private static final int EXPLICIT_PRIME = 1;
    private static final int EXPLICIT_CHAR2 = 2;
    private static final int NAMED_CURVE = 3;
    private ECPublicKey publicKey = null;
    private ECPoint point = null;
    private byte[] pointEncoded = null;
    private final int curveId;
    private byte[] signatureEncoded = null;
    private final SignatureAndHashAlgorithm signatureAndHashAlgorithm;
    private int curveType = 3;

    public ECDHServerKeyExchange(SignatureAndHashAlgorithm signatureAndHashAlgorithm, ECDHECryptography ecdhe, PrivateKey serverPrivateKey, Random clientRandom, Random serverRandom, int namedCurveId, InetSocketAddress peerAddress) throws GeneralSecurityException {
        this(signatureAndHashAlgorithm, namedCurveId, peerAddress);
        this.publicKey = ecdhe.getPublicKey();
        ECParameterSpec parameters = this.publicKey.getParams();
        this.point = this.publicKey.getW();
        this.pointEncoded = ECDHECryptography.encodePoint(this.point, parameters.getCurve());
        Signature signature = Signature.getInstance(this.signatureAndHashAlgorithm.jcaName());
        signature.initSign(serverPrivateKey);
        this.updateSignature(signature, clientRandom, serverRandom);
        this.signatureEncoded = signature.sign();
    }

    private ECDHServerKeyExchange(SignatureAndHashAlgorithm signatureAndHashAlgorithm, int curveId, byte[] pointEncoded, byte[] signatureEncoded, InetSocketAddress peerAddress) throws HandshakeException {
        this(signatureAndHashAlgorithm, curveId, peerAddress);
        this.pointEncoded = Arrays.copyOf(pointEncoded, pointEncoded.length);
        this.signatureEncoded = Arrays.copyOf(signatureEncoded, signatureEncoded.length);
        ECDHECryptography.SupportedGroup group = ECDHECryptography.SupportedGroup.fromId(curveId);
        if (group == null || !group.isUsable()) {
            throw new HandshakeException(String.format("Server used unsupported elliptic curve (%d) for ECDH", curveId), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.HANDSHAKE_FAILURE, peerAddress));
        }
        try {
            this.point = ECDHECryptography.decodePoint(pointEncoded, group.getEcParams().getCurve());
            KeyFactory keyFactory = KeyFactory.getInstance(KEYPAIR_GENERATOR_INSTANCE);
            this.publicKey = (ECPublicKey)keyFactory.generatePublic(new ECPublicKeySpec(this.point, group.getEcParams()));
        }
        catch (GeneralSecurityException e) {
            LOGGER.debug("Cannot re-create server's public key from params", (Throwable)e);
            throw new HandshakeException(String.format("Cannot re-create server's public key from params: %s", e.getMessage()), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.INTERNAL_ERROR, peerAddress));
        }
    }

    private ECDHServerKeyExchange(SignatureAndHashAlgorithm signatureAndHashAlgorithm, int namedCurveId, InetSocketAddress peerAddress) {
        super(peerAddress);
        this.signatureAndHashAlgorithm = signatureAndHashAlgorithm;
        this.curveId = namedCurveId;
    }

    @Override
    public byte[] fragmentToByteArray() {
        DatagramWriter writer = new DatagramWriter();
        switch (this.curveType) {
            case 1: 
            case 2: {
                break;
            }
            case 3: {
                this.writeNamedCurve(writer);
                break;
            }
            default: {
                LOGGER.warn(MSG_UNKNOWN_CURVE_TYPE, (Object)this.curveType);
            }
        }
        return writer.toByteArray();
    }

    private void writeNamedCurve(DatagramWriter writer) {
        writer.write(3, 8);
        writer.write(this.curveId, 16);
        writer.write(this.pointEncoded.length, 8);
        writer.writeBytes(this.pointEncoded);
        if (this.signatureEncoded != null) {
            writer.write(this.signatureAndHashAlgorithm.getHash().getCode(), 8);
            writer.write(this.signatureAndHashAlgorithm.getSignature().getCode(), 8);
            writer.write(this.signatureEncoded.length, 16);
            writer.writeBytes(this.signatureEncoded);
        }
    }

    public static HandshakeMessage fromByteArray(byte[] byteArray, InetSocketAddress peerAddress) throws HandshakeException {
        DatagramReader reader = new DatagramReader(byteArray);
        int curveType = reader.read(8);
        switch (curveType) {
            case 3: {
                return ECDHServerKeyExchange.readNamedCurve(reader, peerAddress);
            }
        }
        throw new HandshakeException(String.format("Curve type [%s] received in ServerKeyExchange message from peer [%s] is unsupported", curveType, peerAddress), new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.HANDSHAKE_FAILURE, peerAddress));
    }

    private static ECDHServerKeyExchange readNamedCurve(DatagramReader reader, InetSocketAddress peerAddress) throws HandshakeException {
        int curveId = reader.read(16);
        int length = reader.read(8);
        byte[] pointEncoded = reader.readBytes(length);
        byte[] bytesLeft = reader.readBytesLeft();
        SignatureAndHashAlgorithm signAndHash = new SignatureAndHashAlgorithm(SignatureAndHashAlgorithm.HashAlgorithm.SHA256, SignatureAndHashAlgorithm.SignatureAlgorithm.ECDSA);
        byte[] signatureEncoded = null;
        if (bytesLeft.length > 0) {
            DatagramReader remainder = new DatagramReader(bytesLeft);
            int hashAlgorithm = remainder.read(8);
            int signatureAlgorithm = remainder.read(8);
            signAndHash = new SignatureAndHashAlgorithm(hashAlgorithm, signatureAlgorithm);
            length = remainder.read(16);
            signatureEncoded = remainder.readBytes(length);
        }
        return new ECDHServerKeyExchange(signAndHash, curveId, pointEncoded, signatureEncoded, peerAddress);
    }

    @Override
    public int getMessageLength() {
        int length = 0;
        switch (this.curveType) {
            case 1: 
            case 2: {
                break;
            }
            case 3: {
                int signatureLength = this.signatureEncoded == null ? 0 : 4 + this.signatureEncoded.length;
                length = 4 + this.pointEncoded.length + signatureLength;
                break;
            }
            default: {
                LOGGER.warn(MSG_UNKNOWN_CURVE_TYPE, (Object)this.curveType);
            }
        }
        return length;
    }

    public void verifySignature(PublicKey serverPublicKey, Random clientRandom, Random serverRandom) throws HandshakeException {
        if (this.signatureEncoded == null) {
            return;
        }
        boolean verified = false;
        try {
            Signature signature = Signature.getInstance(this.signatureAndHashAlgorithm.jcaName());
            signature.initVerify(serverPublicKey);
            this.updateSignature(signature, clientRandom, serverRandom);
            verified = signature.verify(this.signatureEncoded);
        }
        catch (GeneralSecurityException e) {
            LOGGER.error("Could not verify the server's signature.", (Throwable)e);
        }
        if (!verified) {
            String message = "The server's ECDHE key exchange message's signature could not be verified.";
            AlertMessage alert = new AlertMessage(AlertMessage.AlertLevel.FATAL, AlertMessage.AlertDescription.HANDSHAKE_FAILURE, this.getPeer());
            throw new HandshakeException(message, alert);
        }
    }

    private void updateSignature(Signature signature, Random clientRandom, Random serverRandom) throws SignatureException {
        signature.update(clientRandom.getRandomBytes());
        signature.update(serverRandom.getRandomBytes());
        switch (this.curveType) {
            case 1: 
            case 2: {
                break;
            }
            case 3: {
                this.updateSignatureForNamedCurve(signature);
                break;
            }
            default: {
                LOGGER.warn(MSG_UNKNOWN_CURVE_TYPE, (Object)this.curveType);
            }
        }
    }

    private void updateSignatureForNamedCurve(Signature signature) throws SignatureException {
        signature.update((byte)3);
        signature.update((byte)(this.curveId >> 8));
        signature.update((byte)this.curveId);
        signature.update((byte)this.pointEncoded.length);
        signature.update(this.pointEncoded);
    }

    public ECPublicKey getPublicKey() {
        return this.publicKey;
    }

    public int getCurveId() {
        return this.curveId;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(super.toString());
        sb.append("\t\tDiffie-Hellman public key: ");
        sb.append(this.getPublicKey().toString());
        sb.append(StringUtil.lineSeparator());
        return sb.toString();
    }
}

