/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.californium.scandium.dtls;

import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.eclipse.californium.elements.util.JceProviderUtil;
import org.eclipse.californium.scandium.dtls.cipher.CipherSuite;
import org.eclipse.californium.scandium.dtls.cipher.ThreadLocalSignature;
import org.eclipse.californium.scandium.util.ListUtils;

public final class SignatureAndHashAlgorithm {
    public static final SignatureAndHashAlgorithm SHA1_WITH_ECDSA;
    public static final SignatureAndHashAlgorithm SHA256_WITH_ECDSA;
    public static final SignatureAndHashAlgorithm SHA384_WITH_ECDSA;
    public static final SignatureAndHashAlgorithm SHA256_WITH_RSA;
    public static final SignatureAndHashAlgorithm INTRINSIC_WITH_ED25519;
    public static final SignatureAndHashAlgorithm INTRINSIC_WITH_ED448;
    public static final List<SignatureAndHashAlgorithm> DEFAULT;
    private final String jcaName;
    private final HashAlgorithm hash;
    private final SignatureAlgorithm signature;
    private final int hashAlgorithmCode;
    private final int signatureAlgorithmCode;
    private final boolean supported;

    public static ThreadLocalSignature getThreadLocalSignature(String algorithm) {
        if (algorithm == null) {
            algorithm = "UNKNOWN";
        }
        return ThreadLocalSignature.SIGNATURES.get(algorithm);
    }

    public static SignatureAndHashAlgorithm valueOf(String jcaName) {
        int index = jcaName.indexOf("with");
        if (index < 0) {
            index = jcaName.indexOf("WITH");
        }
        HashAlgorithm hashAlgorithm = null;
        SignatureAlgorithm signatureAlgorithm = null;
        if (0 < index) {
            String hash = jcaName.substring(0, index);
            String signature = jcaName.substring(index + 4, jcaName.length());
            try {
                hashAlgorithm = HashAlgorithm.valueOf(hash);
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
            try {
                signatureAlgorithm = SignatureAlgorithm.valueOf(signature);
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
            if (hashAlgorithm == null && signatureAlgorithm == null) {
                throw new IllegalArgumentException(jcaName + " is unknown!");
            }
            if (hashAlgorithm == null) {
                throw new IllegalArgumentException(jcaName + " uses a unknown hash-algorithm!");
            }
            if (signatureAlgorithm == null) {
                throw new IllegalArgumentException(jcaName + " uses a unknown signature-algorithm!");
            }
        } else {
            hashAlgorithm = HashAlgorithm.INTRINSIC;
            signatureAlgorithm = SignatureAlgorithm.intrinsicValueOf(jcaName);
        }
        return new SignatureAndHashAlgorithm(hashAlgorithm, signatureAlgorithm);
    }

    public static List<SignatureAndHashAlgorithm> getSignatureAlgorithms(List<X509Certificate> certificateChain) {
        ArrayList<SignatureAndHashAlgorithm> result = new ArrayList<SignatureAndHashAlgorithm>();
        if (certificateChain != null && !certificateChain.isEmpty()) {
            for (X509Certificate certificate : certificateChain) {
                String sigAlgName = certificate.getSigAlgName();
                SignatureAndHashAlgorithm signature = SignatureAndHashAlgorithm.valueOf(sigAlgName);
                if (!signature.isSupported()) {
                    throw new IllegalArgumentException(sigAlgName + " is not supported by JCE!");
                }
                ListUtils.addIfAbsent(result, signature);
            }
        }
        return result;
    }

    public static void ensureSignatureAlgorithm(List<SignatureAndHashAlgorithm> algorithms, PublicKey publicKey) {
        if (publicKey == null) {
            throw new NullPointerException("Public key must not be null!");
        }
        SignatureAndHashAlgorithm signAndHash = SignatureAndHashAlgorithm.getSupportedSignatureAlgorithm(DEFAULT, publicKey);
        if (signAndHash != null) {
            ListUtils.addIfAbsent(algorithms, signAndHash);
            return;
        }
        if (algorithms == null) {
            throw new NullPointerException("The defaults list must not be null!");
        }
        if (SignatureAndHashAlgorithm.getSupportedSignatureAlgorithm(algorithms, publicKey) != null) {
            return;
        }
        boolean keyAlgorithmSupported = false;
        for (SignatureAlgorithm signatureAlgorithm : SignatureAlgorithm.values()) {
            if (!signatureAlgorithm.isSupported(publicKey.getAlgorithm())) continue;
            keyAlgorithmSupported = true;
            if (signatureAlgorithm.isIntrinsic()) {
                signAndHash = new SignatureAndHashAlgorithm(HashAlgorithm.INTRINSIC, signatureAlgorithm);
                if (!signAndHash.isSupported(publicKey)) continue;
                ListUtils.addIfAbsent(algorithms, signAndHash);
                return;
            }
            for (HashAlgorithm hashAlgorithm : HashAlgorithm.values()) {
                if (hashAlgorithm == HashAlgorithm.INTRINSIC || !hashAlgorithm.isRecommended() || !(signAndHash = new SignatureAndHashAlgorithm(hashAlgorithm, signatureAlgorithm)).isSupported(publicKey)) continue;
                ListUtils.addIfAbsent(algorithms, signAndHash);
                return;
            }
        }
        if (keyAlgorithmSupported) {
            throw new IllegalArgumentException(publicKey.getAlgorithm() + " public key is not supported!");
        }
        throw new IllegalArgumentException(publicKey.getAlgorithm() + " is not supported!");
    }

    public static List<SignatureAndHashAlgorithm> getCommonSignatureAlgorithms(List<SignatureAndHashAlgorithm> proposedSignatureAndHashAlgorithms, List<SignatureAndHashAlgorithm> supportedSignatureAndHashAlgorithms) {
        ArrayList<SignatureAndHashAlgorithm> result = new ArrayList<SignatureAndHashAlgorithm>();
        for (SignatureAndHashAlgorithm algo : proposedSignatureAndHashAlgorithms) {
            if (!supportedSignatureAndHashAlgorithms.contains(algo)) continue;
            ListUtils.addIfAbsent(result, algo);
        }
        return result;
    }

    public static SignatureAndHashAlgorithm getSupportedSignatureAlgorithm(List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, PublicKey key) {
        if (key == null) {
            throw new NullPointerException("Public key must not be null!");
        }
        for (SignatureAndHashAlgorithm supportedAlgorithm : supportedSignatureAlgorithms) {
            if (!supportedAlgorithm.isSupported(key)) continue;
            return supportedAlgorithm;
        }
        return null;
    }

    public static List<SignatureAndHashAlgorithm> getCompatibleSignatureAlgorithms(List<SignatureAndHashAlgorithm> signatureAndHashAlgorithms, List<CipherSuite.CertificateKeyAlgorithm> certificatekeyAlgorithms) {
        ArrayList<SignatureAndHashAlgorithm> result = new ArrayList<SignatureAndHashAlgorithm>();
        block0: for (SignatureAndHashAlgorithm algo : signatureAndHashAlgorithms) {
            for (CipherSuite.CertificateKeyAlgorithm certificateKeyAlgorithm : certificatekeyAlgorithms) {
                if (!algo.isSupported(certificateKeyAlgorithm)) continue;
                result.add(algo);
                continue block0;
            }
        }
        return result;
    }

    public static boolean isSupportedAlgorithm(List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, CipherSuite.CertificateKeyAlgorithm certificatekeyAlgorithm) {
        for (SignatureAndHashAlgorithm supportedAlgorithm : supportedSignatureAlgorithms) {
            if (!supportedAlgorithm.isSupported(certificatekeyAlgorithm)) continue;
            return true;
        }
        return false;
    }

    public static boolean isSupportedAlgorithm(List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, String keyAlgorithm) {
        for (SignatureAndHashAlgorithm supportedAlgorithm : supportedSignatureAlgorithms) {
            if (!supportedAlgorithm.isSupported(keyAlgorithm)) continue;
            return true;
        }
        return false;
    }

    public static boolean isSignedWithSupportedAlgorithms(List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, List<X509Certificate> certificateChain) {
        for (X509Certificate certificate : certificateChain) {
            if (SignatureAndHashAlgorithm.isSignedWithSupportedAlgorithm(supportedSignatureAlgorithms, certificate)) continue;
            return false;
        }
        return true;
    }

    public static boolean isSignedWithSupportedAlgorithm(List<SignatureAndHashAlgorithm> supportedSignatureAlgorithms, X509Certificate certificate) {
        String sigAlgName = certificate.getSigAlgName();
        String sigEdDsa = JceProviderUtil.getEdDsaStandardAlgorithmName((String)sigAlgName, null);
        if (sigEdDsa != null) {
            if (SignatureAlgorithm.ED25519.isSupported(sigEdDsa)) {
                return supportedSignatureAlgorithms.contains(INTRINSIC_WITH_ED25519);
            }
            if (SignatureAlgorithm.ED448.isSupported(sigEdDsa)) {
                return supportedSignatureAlgorithms.contains(INTRINSIC_WITH_ED448);
            }
            return false;
        }
        for (SignatureAndHashAlgorithm supportedAlgorithm : supportedSignatureAlgorithms) {
            if (!sigAlgName.equalsIgnoreCase(supportedAlgorithm.getJcaName())) continue;
            return true;
        }
        return false;
    }

    public SignatureAndHashAlgorithm(HashAlgorithm hashAlgorithm, SignatureAlgorithm signatureAlgorithm) {
        if (hashAlgorithm == null) {
            throw new NullPointerException("Hash Algorithm must not be null!");
        }
        if (signatureAlgorithm == null) {
            throw new NullPointerException("Signature Algorithm must not be null!");
        }
        this.hash = hashAlgorithm;
        this.signature = signatureAlgorithm;
        this.hashAlgorithmCode = hashAlgorithm.getCode();
        this.signatureAlgorithmCode = signatureAlgorithm.getCode();
        this.jcaName = this.buildJcaName();
        this.supported = this.jcaName != null && SignatureAndHashAlgorithm.getThreadLocalSignature(this.jcaName).isSupported();
    }

    public SignatureAndHashAlgorithm(int hashAlgorithmCode, int signatureAlgorithmCode) {
        this.hashAlgorithmCode = hashAlgorithmCode;
        this.signatureAlgorithmCode = signatureAlgorithmCode;
        this.signature = SignatureAlgorithm.getAlgorithmByCode(signatureAlgorithmCode);
        this.hash = HashAlgorithm.getAlgorithmByCode(hashAlgorithmCode);
        this.jcaName = this.buildJcaName();
        this.supported = this.jcaName != null && SignatureAndHashAlgorithm.getThreadLocalSignature(this.jcaName).isSupported();
    }

    private String buildJcaName() {
        if (this.hash != null && this.signature != null) {
            StringBuilder name = new StringBuilder();
            if (this.hash != HashAlgorithm.INTRINSIC) {
                name.append((Object)this.hash);
                name.append("with");
            }
            name.append((Object)this.signature);
            return name.toString();
        }
        return null;
    }

    public SignatureAlgorithm getSignature() {
        return this.signature;
    }

    public HashAlgorithm getHash() {
        return this.hash;
    }

    public String getJcaName() {
        return this.jcaName;
    }

    public boolean isRecommended() {
        return this.signature != null && this.hash != null && this.hash.isRecommended();
    }

    public boolean isSupported() {
        return this.supported;
    }

    public boolean isSupported(String keyAlgorithm) {
        if (this.supported) {
            return this.signature.isSupported(keyAlgorithm);
        }
        return false;
    }

    public boolean isSupported(CipherSuite.CertificateKeyAlgorithm certificateKeyAlgorithm) {
        if (this.supported) {
            return this.signature.isSupported(certificateKeyAlgorithm);
        }
        return false;
    }

    public boolean isSupported(PublicKey publicKey) {
        Signature signature;
        if (this.supported && this.signature.isSupported(publicKey.getAlgorithm()) && (signature = (Signature)this.getThreadLocalSignature().current()) != null) {
            try {
                signature.initVerify(publicKey);
                return true;
            }
            catch (InvalidKeyException invalidKeyException) {
                // empty catch block
            }
        }
        return false;
    }

    public String toString() {
        if (this.jcaName != null) {
            return this.jcaName;
        }
        StringBuilder result = new StringBuilder();
        if (this.hash != null) {
            result.append((Object)this.hash);
        } else {
            result.append(String.format("0x%02x", this.hashAlgorithmCode));
        }
        result.append("with");
        if (this.signature != null) {
            result.append((Object)this.signature);
        } else {
            result.append(String.format("0x%02x", this.signatureAlgorithmCode));
        }
        return result.toString();
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (this.getClass() != obj.getClass()) {
            return false;
        }
        SignatureAndHashAlgorithm other = (SignatureAndHashAlgorithm)obj;
        return this.signatureAlgorithmCode == other.signatureAlgorithmCode && this.hashAlgorithmCode == other.hashAlgorithmCode;
    }

    public int hashCode() {
        return this.hashAlgorithmCode * 256 + this.signatureAlgorithmCode;
    }

    public ThreadLocalSignature getThreadLocalSignature() {
        return SignatureAndHashAlgorithm.getThreadLocalSignature(this.getJcaName());
    }

    static {
        JceProviderUtil.init();
        SHA1_WITH_ECDSA = new SignatureAndHashAlgorithm(HashAlgorithm.SHA1, SignatureAlgorithm.ECDSA);
        SHA256_WITH_ECDSA = new SignatureAndHashAlgorithm(HashAlgorithm.SHA256, SignatureAlgorithm.ECDSA);
        SHA384_WITH_ECDSA = new SignatureAndHashAlgorithm(HashAlgorithm.SHA384, SignatureAlgorithm.ECDSA);
        SHA256_WITH_RSA = new SignatureAndHashAlgorithm(HashAlgorithm.SHA256, SignatureAlgorithm.RSA);
        INTRINSIC_WITH_ED25519 = new SignatureAndHashAlgorithm(HashAlgorithm.INTRINSIC, SignatureAlgorithm.ED25519);
        INTRINSIC_WITH_ED448 = new SignatureAndHashAlgorithm(HashAlgorithm.INTRINSIC, SignatureAlgorithm.ED448);
        DEFAULT = Collections.unmodifiableList(Arrays.asList(SHA256_WITH_ECDSA, SHA256_WITH_RSA));
    }

    public static enum SignatureAlgorithm {
        ANONYMOUS(0, null),
        RSA(1, CipherSuite.CertificateKeyAlgorithm.RSA),
        DSA(2, CipherSuite.CertificateKeyAlgorithm.DSA),
        ECDSA(3, CipherSuite.CertificateKeyAlgorithm.EC, "EC", false),
        ED25519(7, CipherSuite.CertificateKeyAlgorithm.EC, "OID.1.3.101.112", true),
        ED448(8, CipherSuite.CertificateKeyAlgorithm.EC, "OID.1.3.101.113", true);

        private final int code;
        private final CipherSuite.CertificateKeyAlgorithm certificateKeyAlgorithm;
        private final String keyAlgorithm;
        private final boolean isIntrinsic;

        private SignatureAlgorithm(int code, CipherSuite.CertificateKeyAlgorithm certificateKeyAlgorithm) {
            this.code = code;
            this.certificateKeyAlgorithm = certificateKeyAlgorithm;
            this.keyAlgorithm = this.name();
            this.isIntrinsic = false;
        }

        private SignatureAlgorithm(int code, CipherSuite.CertificateKeyAlgorithm certificateKeyAlgorithm, String keyAlgorithm, boolean intrinsic) {
            this.code = code;
            this.certificateKeyAlgorithm = certificateKeyAlgorithm;
            this.keyAlgorithm = keyAlgorithm;
            this.isIntrinsic = intrinsic;
        }

        public static SignatureAlgorithm getAlgorithmByCode(int code) {
            switch (code) {
                case 0: {
                    return ANONYMOUS;
                }
                case 1: {
                    return RSA;
                }
                case 2: {
                    return DSA;
                }
                case 3: {
                    return ECDSA;
                }
                case 7: {
                    return ED25519;
                }
                case 8: {
                    return ED448;
                }
            }
            return null;
        }

        public int getCode() {
            return this.code;
        }

        public boolean isSupported(String keyAlgorithm) {
            String key;
            if (this.keyAlgorithm.equalsIgnoreCase(keyAlgorithm)) {
                return JceProviderUtil.isSupported((String)keyAlgorithm);
            }
            if ((ED25519 == this || ED448 == this) && (key = JceProviderUtil.getEdDsaStandardAlgorithmName((String)keyAlgorithm, null)) != null) {
                if (ED25519 == this) {
                    if ("OID.1.3.101.112" == key || "EdDSA" == key) {
                        return JceProviderUtil.isSupported((String)"Ed25519");
                    }
                } else if ("OID.1.3.101.113" == key || "EdDSA" == key) {
                    return JceProviderUtil.isSupported((String)"Ed448");
                }
            }
            return false;
        }

        public boolean isSupported(CipherSuite.CertificateKeyAlgorithm certificateKeyAlgorithm) {
            return this.certificateKeyAlgorithm == certificateKeyAlgorithm;
        }

        public boolean isIntrinsic() {
            return this.isIntrinsic;
        }

        public static SignatureAlgorithm intrinsicValueOf(String algorithmName) {
            String standardAlgorithmName = JceProviderUtil.getEdDsaStandardAlgorithmName((String)algorithmName, null);
            if (standardAlgorithmName != null) {
                for (SignatureAlgorithm algorithm : SignatureAlgorithm.values()) {
                    if (!algorithm.isIntrinsic || !algorithm.isSupported(standardAlgorithmName)) continue;
                    return algorithm;
                }
                throw new IllegalArgumentException(algorithmName + " is no supported intrinsic algorithm!");
            }
            throw new IllegalArgumentException(algorithmName + " is unknown intrinsic algorithm!");
        }
    }

    public static enum HashAlgorithm {
        NONE(0, false),
        MD5(1, false),
        SHA1(2, false),
        SHA224(3, false),
        SHA256(4, true),
        SHA384(5, true),
        SHA512(6, true),
        INTRINSIC(8, true);

        private final int code;
        private final boolean recommended;

        private HashAlgorithm(int code, boolean recommended) {
            this.code = code;
            this.recommended = recommended;
        }

        public static HashAlgorithm getAlgorithmByCode(int code) {
            switch (code) {
                case 0: {
                    return NONE;
                }
                case 1: {
                    return MD5;
                }
                case 2: {
                    return SHA1;
                }
                case 3: {
                    return SHA224;
                }
                case 4: {
                    return SHA256;
                }
                case 5: {
                    return SHA384;
                }
                case 6: {
                    return SHA512;
                }
                case 8: {
                    return INTRINSIC;
                }
            }
            return null;
        }

        public int getCode() {
            return this.code;
        }

        public boolean isRecommended() {
            return this.recommended;
        }
    }
}

