/*
 * Copyright (c) 2016 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.stack.mutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.util.EmptyStackException;

import org.eclipse.collections.api.ByteIterable;
import org.eclipse.collections.api.LazyByteIterable;
import org.eclipse.collections.api.bag.primitive.MutableByteBag;
import org.eclipse.collections.api.block.function.primitive.ByteToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectByteIntToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectByteToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.BytePredicate;
import org.eclipse.collections.api.block.procedure.primitive.ByteIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.ByteProcedure;
import org.eclipse.collections.api.iterator.ByteIterator;
import org.eclipse.collections.api.list.primitive.ByteList;
import org.eclipse.collections.api.list.primitive.MutableByteList;
import org.eclipse.collections.api.set.primitive.MutableByteSet;
import org.eclipse.collections.api.stack.MutableStack;
import org.eclipse.collections.api.stack.primitive.ByteStack;
import org.eclipse.collections.api.stack.primitive.ImmutableByteStack;
import org.eclipse.collections.api.stack.primitive.MutableByteStack;
import org.eclipse.collections.impl.bag.mutable.primitive.ByteHashBag;
import org.eclipse.collections.impl.factory.primitive.ByteStacks;
import org.eclipse.collections.impl.iterator.UnmodifiableByteIterator;
import org.eclipse.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import org.eclipse.collections.impl.list.mutable.primitive.ByteArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.ByteHashSet;
import org.eclipse.collections.impl.stack.mutable.ArrayStack;
import net.jcip.annotations.NotThreadSafe;

/**
 * ByteArrayStack is similar to {@link ArrayStack}, and is memory-optimized for byte primitives.
 * This file was automatically generated from template file primitiveArrayStack.stg.
 */
@NotThreadSafe
public final class ByteArrayStack
        implements MutableByteStack, Externalizable
{
    private static final long serialVersionUID = 1L;

    private transient ByteArrayList delegate;

    public ByteArrayStack()
    {
        this.delegate = new ByteArrayList();
    }

    private ByteArrayStack(int size)
    {
        this.delegate = new ByteArrayList(size);
    }

    private ByteArrayStack(byte... items)
    {
        this.delegate = new ByteArrayList(items);
    }

    public static ByteArrayStack newStackFromTopToBottom(byte... items)
    {
        ByteArrayStack stack = new ByteArrayStack(items.length);
        for (int i = items.length - 1; i >= 0; i--)
        {
            stack.push(items[i]);
        }
        return stack;
    }

    public static ByteArrayStack newStackWith(byte... items)
    {
        return new ByteArrayStack(items);
    }

    public static ByteArrayStack newStack(ByteIterable items)
    {
        ByteArrayStack stack = new ByteArrayStack(items.size());
        stack.delegate = ByteArrayList.newList(items);
        return stack;
    }

    public static ByteArrayStack newStackFromTopToBottom(ByteIterable items)
    {
        ByteArrayStack stack = new ByteArrayStack(items.size());
        stack.delegate = ByteArrayList.newList(items).reverseThis();
        return stack;
    }

    public void push(byte item)
    {
        this.delegate.add(item);
    }

    public byte pop()
    {
        this.checkEmptyStack();
        return this.delegate.removeAtIndex(this.delegate.size() - 1);
    }

    private void checkEmptyStack()
    {
        if (this.delegate.isEmpty())
        {
            throw new EmptyStackException();
        }
    }

    public ByteList pop(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new ByteArrayList(0);
        }
        MutableByteList subList = new ByteArrayList(count);
        while (count > 0)
        {
            subList.add(this.pop());
            count--;
        }
        return subList;
    }

    private void checkPositiveValueForCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    public MutableByteStack select(BytePredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().select(predicate));
    }

    public MutableByteStack reject(BytePredicate predicate)
    {
        return newStackFromTopToBottom(this.delegate.asReversed().reject(predicate));
    }

    public byte peek()
    {
        this.checkEmptyStack();
        return this.delegate.getLast();
    }

    public ByteList peek(int count)
    {
        this.checkPositiveValueForCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new ByteArrayList(0);
        }
        MutableByteList subList = new ByteArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public byte peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public ByteIterator byteIterator()
    {
        return new UnmodifiableByteIterator(this.delegate.asReversed().byteIterator());
    }

    public void forEach(ByteProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(ByteProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return this.delegate.isEmpty();
    }

    public boolean notEmpty()
    {
        return this.delegate.notEmpty();
    }

    public int count(BytePredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(BytePredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(BytePredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(BytePredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public byte detectIfNone(BytePredicate predicate, byte ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> MutableStack<V> collect(ByteToObjectFunction<? extends V> function)
    {
        return ArrayStack.newStackFromTopToBottom(this.delegate.asReversed().collect(function));
    }

    public <V> V injectInto(V injectedValue, ObjectByteToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.asReversed().injectInto(injectedValue, function);
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public byte max()
    {
        return this.delegate.max();
    }

    public byte min()
    {
        return this.delegate.min();
    }

    public byte minIfEmpty(byte defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.min();
    }

    public byte maxIfEmpty(byte defaultValue)
    {
        if (this.isEmpty())
        {
            return defaultValue;
        }
        return this.max();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public byte[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public byte[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(byte value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(byte... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(ByteIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public void clear()
    {
        this.delegate.clear();
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof ByteStack))
        {
            return false;
        }
        ByteStack stack = (ByteStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        ByteIterable iterable = this.delegate.asReversed();
        ByteIterator iterator = iterable.byteIterator();
        while (iterator.hasNext())
        {
            byte item = iterator.next();
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    public MutableByteList toList()
    {
        return ByteArrayList.newList(this);
    }

    public MutableByteList toSortedList()
    {
        return ByteArrayList.newList(this).sortThis();
    }

    public MutableByteSet toSet()
    {
        return ByteHashSet.newSet(this);
    }

    public MutableByteBag toBag()
    {
        return ByteHashBag.newBag(this);
    }

    public byte getFirst()
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".getFirst() not implemented yet");
    }

    public int indexOf(byte value)
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".indexOf() not implemented yet");
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectByteIntToObjectFunction<? super T, ? extends T> function)
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".injectIntoWithIndex() not implemented yet");
    }

    public void forEachWithIndex(ByteIntProcedure procedure)
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".forEachWithIndex() not implemented yet");
    }

    public LazyByteIterable asLazy()
    {
        return new LazyByteIterableAdapter(this);
    }

    public MutableByteStack asUnmodifiable()
    {
        return new UnmodifiableByteStack(this);
    }

    public MutableByteStack asSynchronized()
    {
        return new SynchronizedByteStack(this);
    }

    public ImmutableByteStack toImmutable()
    {
        return ByteStacks.immutable.withAll(this.delegate);
    }

    public void writeExternal(ObjectOutput out) throws IOException
    {
        out.writeInt(this.size());
        ByteIterator iterator = this.delegate.asReversed().byteIterator();
        while (iterator.hasNext())
        {
            byte each = iterator.next();
            out.writeByte(each);
        }
    }

    public void readExternal(ObjectInput in) throws IOException
    {
        int size = in.readInt();
        byte[] array = new byte[size];
        for (int i = size - 1; i >= 0; i--)
        {
            array[i] = in.readByte();
        }
        this.delegate = ByteArrayList.newListWith(array);
    }
}
