/*
 * Copyright (c) 2016 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.stack.immutable.primitive;

import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectInput;
import java.io.ObjectOutput;
import java.io.Serializable;

import org.eclipse.collections.api.LazyCharIterable;
import org.eclipse.collections.api.CharIterable;
import org.eclipse.collections.api.bag.primitive.MutableCharBag;
import org.eclipse.collections.api.block.function.primitive.CharToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectCharToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectCharIntToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.CharPredicate;
import org.eclipse.collections.api.block.procedure.primitive.CharIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.CharProcedure;
import org.eclipse.collections.api.iterator.CharIterator;
import org.eclipse.collections.api.list.primitive.CharList;
import org.eclipse.collections.api.list.primitive.MutableCharList;
import org.eclipse.collections.api.set.primitive.MutableCharSet;
import org.eclipse.collections.api.stack.ImmutableStack;
import org.eclipse.collections.api.stack.primitive.CharStack;
import org.eclipse.collections.api.stack.primitive.ImmutableCharStack;
import org.eclipse.collections.impl.bag.mutable.primitive.CharHashBag;
import org.eclipse.collections.impl.block.procedure.checked.primitive.CheckedCharProcedure;
import org.eclipse.collections.impl.factory.Stacks;
import org.eclipse.collections.impl.factory.primitive.CharStacks;
import org.eclipse.collections.impl.iterator.UnmodifiableCharIterator;
import org.eclipse.collections.impl.lazy.primitive.LazyCharIterableAdapter;
import org.eclipse.collections.impl.list.mutable.primitive.CharArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.CharHashSet;
import org.eclipse.collections.impl.stack.mutable.primitive.CharArrayStack;
import net.jcip.annotations.Immutable;

/**
 * ImmutableCharArrayStack is the non-modifiable equivalent of {@link CharArrayStack}.
 * This file was automatically generated from template file immutablePrimitiveArrayStack.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableCharArrayStack
        implements ImmutableCharStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final CharArrayList delegate;

    private ImmutableCharArrayStack(char[] newElements)
    {
        this.checkOptimizedSize(newElements.length);
        this.delegate = new CharArrayList(newElements);
    }

    private ImmutableCharArrayStack(CharArrayList newElements)
    {
        this.checkOptimizedSize(newElements.size());
        this.delegate = newElements;
    }

    private void checkOptimizedSize(int length)
    {
        if (length <= 1)
        {
            throw new IllegalArgumentException("Use CharStacks.immutable.with() to instantiate an optimized collection");
        }
    }

    public static ImmutableCharArrayStack newStack(CharIterable iterable)
    {
        return new ImmutableCharArrayStack(iterable.toArray());
    }

    public static ImmutableCharArrayStack newStackWith(char... elements)
    {
        char[] newArray = new char[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableCharArrayStack(newArray);
    }

    public static ImmutableCharArrayStack newStackFromTopToBottom(char... items)
    {
        return new ImmutableCharArrayStack(CharArrayList.newListWith(items).reverseThis());
    }

    public static ImmutableCharArrayStack newStackFromTopToBottom(CharIterable items)
    {
        return new ImmutableCharArrayStack(CharArrayList.newList(items).reverseThis());
    }

    public ImmutableCharStack push(char item)
    {
        CharArrayList newDelegate = CharArrayList.newList(this.delegate);
        newDelegate.add(item);
        return new ImmutableCharArrayStack(newDelegate);
    }

    public ImmutableCharStack pop()
    {
        CharArrayList newDelegate = CharArrayList.newList(this.delegate);
        newDelegate.removeAtIndex(this.delegate.size() - 1);
        return CharStacks.immutable.with(newDelegate.toArray());
    }

    public ImmutableCharStack pop(int count)
    {
        this.checkNegativeCount(count);
        if (count == 0)
        {
            return this;
        }
        this.checkSizeLessThanCount(count);
        CharArrayList newDelegate = CharArrayList.newList(this.delegate);
        while (count > 0)
        {
            newDelegate.removeAtIndex(newDelegate.size() - 1);
            count--;
        }
        return CharStacks.immutable.with(newDelegate.toArray());
    }

    private void checkNegativeCount(int count)
    {
        if (count < 0)
        {
            throw new IllegalArgumentException("Count must be positive but was " + count);
        }
    }

    public char peek()
    {
        return this.delegate.getLast();
    }

    public CharList peek(int count)
    {
        this.checkNegativeCount(count);
        this.checkSizeLessThanCount(count);
        if (count == 0)
        {
            return new CharArrayList();
        }
        MutableCharList subList = new CharArrayList(count);
        int index = this.delegate.size() - 1;
        for (int i = 0; i < count; i++)
        {
            subList.add(this.delegate.get(index - i));
        }
        return subList;
    }

    public char peekAt(int index)
    {
        this.rangeCheck(index);
        return this.delegate.get(this.delegate.size() - 1 - index);
    }

    private void rangeCheck(int index)
    {
        if (index < 0 || index > this.delegate.size() - 1)
        {
            throw new IllegalArgumentException("Index " + index + " out of range.Should be between 0 and " + (this.delegate.size() - 1));
        }
    }

    public CharIterator charIterator()
    {
        return new UnmodifiableCharIterator(this.delegate.asReversed().charIterator());
    }

    public void forEach(CharProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(CharProcedure procedure)
    {
        this.delegate.asReversed().forEach(procedure);
    }

    public int count(CharPredicate predicate)
    {
        return this.delegate.asReversed().count(predicate);
    }

    public boolean anySatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().anySatisfy(predicate);
    }

    public boolean allSatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().allSatisfy(predicate);
    }

    public boolean noneSatisfy(CharPredicate predicate)
    {
        return this.delegate.asReversed().noneSatisfy(predicate);
    }

    public ImmutableCharStack select(CharPredicate predicate)
    {
        return CharStacks.immutable.withAllReversed(this.delegate.asReversed().select(predicate));
    }

    public ImmutableCharStack reject(CharPredicate predicate)
    {
        return CharStacks.immutable.withAllReversed(this.delegate.asReversed().reject(predicate));
    }

    public char detectIfNone(CharPredicate predicate, char ifNone)
    {
        return this.delegate.asReversed().detectIfNone(predicate, ifNone);
    }

    public <V> ImmutableStack<V> collect(CharToObjectFunction<? extends V> function)
    {
        return Stacks.immutable.withAllReversed(this.delegate.asReversed().collect(function));
    }

    public long sum()
    {
        return this.delegate.sum();
    }

    public char max()
    {
        return this.delegate.max();
    }

    public char maxIfEmpty(char defaultValue)
    {
        return this.max();
    }

    public char min()
    {
        return this.delegate.min();
    }

    public char minIfEmpty(char defaultValue)
    {
        return this.min();
    }

    public double average()
    {
        return this.delegate.average();
    }

    public double median()
    {
        return this.delegate.median();
    }

    public char[] toSortedArray()
    {
        return this.delegate.toSortedArray();
    }

    public MutableCharList toSortedList()
    {
        return CharArrayList.newList(this).sortThis();
    }

    public char[] toArray()
    {
        return this.delegate.asReversed().toArray();
    }

    public boolean contains(char value)
    {
        return this.delegate.asReversed().contains(value);
    }

    public boolean containsAll(char... source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public boolean containsAll(CharIterable source)
    {
        return this.delegate.asReversed().containsAll(source);
    }

    public MutableCharList toList()
    {
        return CharArrayList.newList(this);
    }

    public MutableCharSet toSet()
    {
        return CharHashSet.newSet(this);
    }

    public MutableCharBag toBag()
    {
        return CharHashBag.newBag(this);
    }

    public <V> V injectInto(V injectedValue, ObjectCharToObjectFunction<? super V, ? extends V> function)
    {
        return this.delegate.toReversed().injectInto(injectedValue, function);
    }

    public LazyCharIterable asLazy()
    {
        return new LazyCharIterableAdapter(this);
    }

    public ImmutableCharStack toImmutable()
    {
        return this;
    }

    public int size()
    {
        return this.delegate.size();
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    @Override
    public boolean equals(Object otherStack)
    {
        if (otherStack == this)
        {
            return true;
        }
        if (!(otherStack instanceof CharStack))
        {
            return false;
        }
        CharStack stack = (CharStack) otherStack;
        if (this.size() != stack.size())
        {
            return false;
        }
        for (int i = 0; i < this.size(); i++)
        {
            if (this.peekAt(i) != stack.peekAt(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        CharIterable iterable = this.delegate.asReversed();
        CharIterator iterator = iterable.charIterator();
        while (iterator.hasNext())
        {
            char item = iterator.next();
            hashCode = 31 * hashCode + (int) item;
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.delegate.asReversed().toString();
    }

    public String makeString()
    {
        return this.delegate.asReversed().makeString();
    }

    public String makeString(String separator)
    {
        return this.delegate.asReversed().makeString(separator);
    }

    public String makeString(String start, String separator, String end)
    {
        return this.delegate.asReversed().makeString(start, separator, end);
    }

    public void appendString(Appendable appendable)
    {
        this.delegate.asReversed().appendString(appendable);
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.delegate.asReversed().appendString(appendable, separator);
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        this.delegate.asReversed().appendString(appendable, start, separator, end);
    }

    public char getFirst()
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".getFirst() not implemented yet");
    }

    public int indexOf(char value)
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".indexOf() not implemented yet");
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectCharIntToObjectFunction<? super T, ? extends T> function)
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".injectIntoWithIndex() not implemented yet");
    }

    public void forEachWithIndex(CharIntProcedure procedure)
    {
        throw new UnsupportedOperationException(this.getClass().getSimpleName() + ".forEachWithIndex() not implemented yet");
    }

    private void checkSizeLessThanCount(int count)
    {
        if (this.delegate.size() < count)
        {
            throw new IllegalArgumentException("Count must be less than size: Count = " + count + " Size = " + this.delegate.size());
        }
    }

    private Object writeReplace()
    {
        return new ImmutableCharStackSerializationProxy(this);
    }

    private static class ImmutableCharStackSerializationProxy implements Externalizable
    {
        private static final long serialVersionUID = 1L;

        private CharStack stack;

        @SuppressWarnings("UnusedDeclaration")
        public ImmutableCharStackSerializationProxy()
        {
            // Empty constructor for Externalizable class
        }

        protected ImmutableCharStackSerializationProxy(CharStack stack)
        {
            this.stack = stack;
        }

        public void writeExternal(final ObjectOutput out) throws IOException
        {
            out.writeInt(this.stack.size());
            try
            {
                this.stack.forEach(new CheckedCharProcedure()
                {
                    @Override
                    public void safeValue(char item) throws IOException
                    {
                        out.writeChar(item);
                    }
                });
            }
            catch (RuntimeException e)
            {
                if (e.getCause() instanceof IOException)
                {
                    throw (IOException) e.getCause();
                }
                throw e;
            }
        }

        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException
        {
            int size = in.readInt();
            CharArrayList deserializedDelegate = new CharArrayList(size);

            for (int i = 0; i < size; i++)
            {
                deserializedDelegate.add(in.readChar());
            }

            this.stack = ImmutableCharArrayStack.newStackFromTopToBottom(deserializedDelegate);
        }

        protected Object readResolve()
        {
            return this.stack;
        }
    }
}
