/*
 * Copyright (c) 2016 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.stack.mutable.primitive;

import java.io.Serializable;
import java.util.Collection;
import java.util.Collections;

import org.eclipse.collections.api.DoubleIterable;
import org.eclipse.collections.api.LazyDoubleIterable;
import org.eclipse.collections.api.bag.primitive.MutableDoubleBag;
import org.eclipse.collections.api.block.function.primitive.DoubleToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectDoubleIntToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.DoublePredicate;
import org.eclipse.collections.api.block.procedure.primitive.DoubleIntProcedure;
import org.eclipse.collections.api.block.procedure.primitive.DoubleProcedure;
import org.eclipse.collections.api.iterator.DoubleIterator;
import org.eclipse.collections.api.list.primitive.DoubleList;
import org.eclipse.collections.api.list.primitive.MutableDoubleList;
import org.eclipse.collections.api.set.primitive.MutableDoubleSet;
import org.eclipse.collections.api.stack.MutableStack;
import org.eclipse.collections.api.stack.primitive.ImmutableDoubleStack;
import org.eclipse.collections.api.stack.primitive.MutableDoubleStack;
import org.eclipse.collections.impl.factory.primitive.DoubleStacks;
import org.eclipse.collections.impl.iterator.UnmodifiableDoubleIterator;
import org.eclipse.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableDoubleStack}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link DoubleIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveStack.stg.
 *
 * @see MutableDoubleStack#asSynchronized()
 * @see MutableStack#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedDoubleStack
        implements MutableDoubleStack, Serializable
{
    private static final long serialVersionUID = 1L;
    private final Object lock;
    @GuardedBy("this.lock")
    private final MutableDoubleStack stack;

    SynchronizedDoubleStack(MutableDoubleStack stack)
    {
        this(stack, null);
    }

    SynchronizedDoubleStack(MutableDoubleStack stack, Object newLock)
    {
        this.stack = stack;
        this.lock = newLock == null ? this : newLock;
    }

    public void push(double item)
    {
        synchronized (this.lock)
        {
            this.stack.push(item);
        }
    }

    public double pop()
    {
        synchronized (this.lock)
        {
            return this.stack.pop();
        }
    }

    public DoubleList pop(int count)
    {
        synchronized (this.lock)
        {
            return this.stack.pop(count);
        }
    }

    public double peek()
    {
        synchronized (this.lock)
        {
            return this.stack.peek();
        }
    }

    public DoubleList peek(int count)
    {
        synchronized (this.lock)
        {
            return this.stack.peek(count);
        }
    }

    public double peekAt(int index)
    {
        synchronized (this.lock)
        {
            return this.stack.peekAt(index);
        }
    }

    public int size()
    {
        synchronized (this.lock)
        {
            return this.stack.size();
        }
    }

    public boolean isEmpty()
    {
        synchronized (this.lock)
        {
            return this.stack.isEmpty();
        }
    }

    public boolean notEmpty()
    {
        synchronized (this.lock)
        {
            return this.stack.notEmpty();
        }
    }

    public void clear()
    {
        synchronized (this.lock)
        {
            this.stack.clear();
        }
    }

    public boolean contains(double value)
    {
        synchronized (this.lock)
        {
            return this.stack.contains(value);
        }
    }

    public boolean containsAll(double... source)
    {
        synchronized (this.lock)
        {
            return this.stack.containsAll(source);
        }
    }

    public boolean containsAll(DoubleIterable source)
    {
        synchronized (this.lock)
        {
            return this.stack.containsAll(source);
        }
    }

    /**
     * Must be called in a synchronized block.
     */
    public DoubleIterator doubleIterator()
    {
        return new UnmodifiableDoubleIterator(this.stack.doubleIterator());
    }

    public void forEach(DoubleProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(DoubleProcedure procedure)
    {
        synchronized (this.lock)
        {
            this.stack.forEach(procedure);
        }
    }

    public int count(DoublePredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.count(predicate);
        }
    }

    public boolean anySatisfy(DoublePredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.anySatisfy(predicate);
        }
    }

    public boolean allSatisfy(DoublePredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.allSatisfy(predicate);
        }
    }

    public boolean noneSatisfy(DoublePredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.noneSatisfy(predicate);
        }
    }

    public double detectIfNone(DoublePredicate predicate, double ifNone)
    {
        synchronized (this.lock)
        {
            return this.stack.detectIfNone(predicate, ifNone);
        }
    }

    public MutableDoubleStack select(DoublePredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.select(predicate);
        }
    }

    public MutableDoubleStack reject(DoublePredicate predicate)
    {
        synchronized (this.lock)
        {
            return this.stack.reject(predicate);
        }
    }

    public <V> MutableStack<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        synchronized (this.lock)
        {
            return this.stack.collect(function);
        }
    }

    public double sum()
    {
        synchronized (this.lock)
        {
            return this.stack.sum();
        }
    }

    public double max()
    {
        synchronized (this.lock)
        {
            return this.stack.max();
        }
    }

    public double min()
    {
        synchronized (this.lock)
        {
            return this.stack.min();
        }
    }

    public double minIfEmpty(double defaultValue)
    {
        synchronized (this.lock)
        {
            return this.stack.minIfEmpty(defaultValue);
        }
    }

    public double maxIfEmpty(double defaultValue)
    {
        synchronized (this.lock)
        {
            return this.stack.maxIfEmpty(defaultValue);
        }
    }

    public double average()
    {
        synchronized (this.lock)
        {
            return this.stack.average();
        }
    }

    public double median()
    {
        synchronized (this.lock)
        {
            return this.stack.median();
        }
    }

    public MutableDoubleList toSortedList()
    {
        synchronized (this.lock)
        {
            return this.stack.toSortedList();
        }
    }

    public double[] toSortedArray()
    {
        synchronized (this.lock)
        {
            return this.stack.toSortedArray();
        }
    }

    public double[] toArray()
    {
        synchronized (this.lock)
        {
            return this.stack.toArray();
        }
    }

    @Override
    public String toString()
    {
        synchronized (this.lock)
        {
            return this.stack.toString();
        }
    }

    public String makeString()
    {
        synchronized (this.lock)
        {
            return this.stack.makeString();
        }
    }

    public String makeString(String separator)
    {
        synchronized (this.lock)
        {
            return this.stack.makeString(separator);
        }
    }

    public String makeString(String start, String separator, String end)
    {
        synchronized (this.lock)
        {
            return this.stack.makeString(start, separator, end);
        }
    }

    public void appendString(Appendable appendable)
    {
        synchronized (this.lock)
        {
            this.stack.appendString(appendable);
        }
    }

    public void appendString(Appendable appendable, String separator)
    {
        synchronized (this.lock)
        {
            this.stack.appendString(appendable, separator);
        }
    }

    public void appendString(
            Appendable appendable,
            String start,
            String separator,
            String end)
    {
        synchronized (this.lock)
        {
            this.stack.appendString(appendable, start, separator, end);
        }
    }

    public MutableDoubleList toList()
    {
        synchronized (this.lock)
        {
            return this.stack.toList();
        }
    }

    public MutableDoubleSet toSet()
    {
        synchronized (this.lock)
        {
            return this.stack.toSet();
        }
    }

    public MutableDoubleBag toBag()
    {
        synchronized (this.lock)
        {
            return this.stack.toBag();
        }
    }

    @Override
    public boolean equals(Object otherStack)
    {
        synchronized (this.lock)
        {
            return this.stack.equals(otherStack);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.lock)
        {
            return this.stack.hashCode();
        }
    }

    public LazyDoubleIterable asLazy()
    {
        synchronized (this.lock)
        {
            return new LazyDoubleIterableAdapter(this);
        }
    }

    public MutableDoubleStack asUnmodifiable()
    {
        return new UnmodifiableDoubleStack(this);
    }

    public MutableDoubleStack asSynchronized()
    {
        return this;
    }

    public ImmutableDoubleStack toImmutable()
    {
        return DoubleStacks.immutable.withAllReversed(this);
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.lock)
        {
            return this.stack.injectInto(injectedValue, function);
        }
    }

    public double getFirst()
    {
        synchronized (this.lock)
        {
            return this.stack.getFirst();
        }
    }

    public int indexOf(double value)
    {
        synchronized (this.lock)
        {
            return this.stack.indexOf(value);
        }
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectDoubleIntToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.lock)
        {
            return this.stack.injectIntoWithIndex(injectedValue, function);
        }
    }

    public void forEachWithIndex(DoubleIntProcedure procedure)
    {
        synchronized (this.lock)
        {
            this.stack.forEachWithIndex(procedure);
        }
    }
}
