/*
 * Copyright (c) 2017 Goldman Sachs.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v. 1.0 which accompany this distribution.
 * The Eclipse Public License is available at http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 */

package org.eclipse.collections.impl.list.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.NoSuchElementException;

import org.eclipse.collections.api.LongIterable;
import org.eclipse.collections.api.LazyLongIterable;
import org.eclipse.collections.api.bag.primitive.MutableLongBag;
import org.eclipse.collections.api.block.function.primitive.LongToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import org.eclipse.collections.api.block.function.primitive.ObjectLongIntToObjectFunction;
import org.eclipse.collections.api.block.predicate.primitive.LongPredicate;
import org.eclipse.collections.api.block.procedure.primitive.LongProcedure;
import org.eclipse.collections.api.block.procedure.primitive.LongIntProcedure;
import org.eclipse.collections.api.collection.primitive.MutableLongCollection;
import org.eclipse.collections.api.iterator.LongIterator;
import org.eclipse.collections.api.list.ImmutableList;
import org.eclipse.collections.api.list.primitive.ImmutableLongList;
import org.eclipse.collections.api.list.primitive.LongList;
import org.eclipse.collections.api.list.primitive.MutableLongList;
import org.eclipse.collections.api.set.primitive.MutableLongSet;
import org.eclipse.collections.impl.bag.mutable.primitive.LongHashBag;
import org.eclipse.collections.impl.factory.primitive.LongLists;
import org.eclipse.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import org.eclipse.collections.impl.lazy.primitive.ReverseLongIterable;
import org.eclipse.collections.impl.list.mutable.FastList;
import org.eclipse.collections.impl.list.mutable.primitive.LongArrayList;
import org.eclipse.collections.impl.set.mutable.primitive.LongHashSet;
import net.jcip.annotations.Immutable;

/**
 * ImmutableLongArrayList is the non-modifiable equivalent of {@link LongArrayList}.
 * It wraps a Java long array.
 * This file was automatically generated from template file immutablePrimitiveArrayList.stg.
 *
 * @since 3.2.
 */
@Immutable
final class ImmutableLongArrayList
        implements ImmutableLongList, Serializable
{
    private static final long serialVersionUID = 1L;
    private final long[] items;

    private ImmutableLongArrayList(long[] newElements)
    {
        if (newElements.length <= 1)
        {
            throw new IllegalArgumentException("Use LongLists.immutable.with() to instantiate an optimized collection");
        }
        this.items = newElements;
    }

    public static ImmutableLongArrayList newList(LongIterable iterable)
    {
        return new ImmutableLongArrayList(iterable.toArray());
    }

    public static ImmutableLongArrayList newListWith(long... elements)
    {
        long[] newArray = new long[elements.length];
        System.arraycopy(elements, 0, newArray, 0, elements.length);
        return new ImmutableLongArrayList(newArray);
    }

    public long get(int index)
    {
        return this.items[index];
    }

    public long getFirst()
    {
        return this.items[0];
    }

    public long getLast()
    {
        return this.items[this.items.length - 1];
    }

    public int indexOf(long value)
    {
        for (int i = 0; i < this.items.length; i++)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public int lastIndexOf(long value)
    {
        for (int i = this.items.length - 1; i >= 0; i--)
        {
            if (this.items[i] == value)
            {
                return i;
            }
        }
        return -1;
    }

    public LongIterator longIterator()
    {
        return new InternalLongIterator();
    }

    public void forEach(LongProcedure procedure)
    {
        this.each(procedure);
    }

    /**
     * @since 7.0.
     */
    public void each(LongProcedure procedure)
    {
        for (long item : this.items)
        {
            procedure.value(item);
        }
    }

    public void forEachWithIndex(LongIntProcedure procedure)
    {
        for (int i = 0; i < this.items.length; i++)
        {
            procedure.value(this.items[i], i);
        }
    }

    public int count(LongPredicate predicate)
    {
        int count = 0;
        for (long item : this.items)
        {
            if (predicate.accept(item))
            {
                count++;
            }
        }
        return count;
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        for (long item : this.items)
        {
            if (predicate.accept(item))
            {
                return true;
            }
        }
        return false;
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        for (long item : this.items)
        {
            if (!predicate.accept(item))
            {
                return false;
            }
        }
        return true;
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        return !this.anySatisfy(predicate);
    }

    public ImmutableLongList select(LongPredicate predicate)
    {
        return this.select(predicate, new LongArrayList()).toImmutable();
    }

    /**
     * @since 8.1.
     */
    public <R extends MutableLongCollection> R select(LongPredicate predicate, R target)
    {
        for (long item : this.items)
        {
            if (predicate.accept(item))
            {
                target.add(item);
            }
        }
        return target;
    }

    public ImmutableLongList reject(LongPredicate predicate)
    {
        return this.reject(predicate, new LongArrayList()).toImmutable();
    }

    /**
     * @since 8.1.
     */
    public <R extends MutableLongCollection> R reject(LongPredicate predicate, R target)
    {
        for (long item : this.items)
        {
            if (!predicate.accept(item))
            {
                target.add(item);
            }
        }
        return target;
    }

    public long detectIfNone(LongPredicate predicate, long ifNone)
    {
        for (long item : this.items)
        {
            if (predicate.accept(item))
            {
                return item;
            }
        }
        return ifNone;
    }

    public <V> ImmutableList<V> collect(LongToObjectFunction<? extends V> function)
    {
        return this.collect(function, FastList.newList(this.items.length)).toImmutable();
    }

    /**
     * @since 8.1.
     */
    public <V, R extends Collection<V>> R collect(LongToObjectFunction<? extends V> function, R target)
    {
        for (long item : this.items)
        {
            target.add(function.valueOf(item));
        }
        return target;
    }

    public long sum()
    {
        long result = 0L;
        for (long item : this.items)
        {
            result += item;
        }
        return result;
    }

    public long max()
    {
        long max = this.items[0];
        for (int i = 1; i < this.items.length; i++)
        {
            long value = this.items[i];
            if (max < value)
            {
                max = value;
            }
        }
        return max;
    }

    public long maxIfEmpty(long defaultValue)
    {
        return this.max();
    }

    public long min()
    {
        long min = this.items[0];
        for (int i = 1; i < this.items.length; i++)
        {
            long value = this.items[i];
            if (value < min)
            {
                min = value;
            }
        }
        return min;
    }

    public long minIfEmpty(long defaultValue)
    {
        return this.min();
    }

    public double average()
    {
        return (double) this.sum() / (double) this.size();
    }

    public double median()
    {
        long[] sortedArray = this.toSortedArray();
        int middleIndex = sortedArray.length >> 1;
        if (sortedArray.length > 1 && (sortedArray.length & 1) == 0)
        {
            long first = sortedArray[middleIndex];
            long second = sortedArray[middleIndex - 1];
            return ((double) first + (double) second) / 2.0;
        }
        return (double) sortedArray[middleIndex];
    }

    public long[] toSortedArray()
    {
        long[] array = this.toArray();
        Arrays.sort(array);
        return array;
    }

    public long dotProduct(LongList list)
    {
        if (this.size() != list.size())
        {
            throw new IllegalArgumentException("Lists used in dotProduct must be the same size");
        }
        long sum = 0L;
        for (int i = 0; i < this.size(); i++)
        {
            sum += this.items[i] * list.get(i);
        }
        return sum;
    }

    public LazyLongIterable asReversed()
    {
        return ReverseLongIterable.adapt(this);
    }

    public MutableLongList toSortedList()
    {
        return LongArrayList.newList(this).sortThis();
    }

    public int binarySearch(long value)
    {
        return Arrays.binarySearch(this.items, value);
    }

    public long[] toArray()
    {
        long[] newItems = new long[this.items.length];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        return newItems;
    }

    public boolean contains(long value)
    {
        for (long item : this.items)
        {
            if (item == value)
            {
                return true;
            }
        }
        return false;
    }

    public boolean containsAll(long... source)
    {
        for (long value : source)
        {
            if (!this.contains(value))
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(LongIterable source)
    {
        for (LongIterator iterator = source.longIterator(); iterator.hasNext(); )
        {
            if (!this.contains(iterator.next()))
            {
                return false;
            }
        }
        return true;
    }

    public MutableLongList toList()
    {
        return LongArrayList.newList(this);
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSet(this);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBag(this);
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public ImmutableLongList toImmutable()
    {
        return this;
    }

    public ImmutableLongArrayList toReversed()
    {
        return ImmutableLongArrayList.newList(this.asReversed());
    }

    public ImmutableLongList newWith(long element)
    {
        long[] newItems = new long[this.items.length + 1];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        newItems[this.items.length] = element;
        return new ImmutableLongArrayList(newItems);
    }

    public ImmutableLongList newWithout(long element)
    {
        int index = this.indexOf(element);
        if (index != -1)
        {
            long[] newItems = new long[this.items.length - 1];
            System.arraycopy(this.items, 0, newItems, 0, index);
            System.arraycopy(this.items, index + 1, newItems, index, this.items.length - index - 1);
            return LongLists.immutable.with(newItems);
        }
        return this;
    }

    public ImmutableLongList newWithAll(LongIterable elements)
    {
        long[] newItems = new long[this.items.length + elements.size()];
        System.arraycopy(this.items, 0, newItems, 0, this.items.length);
        int index = 0;
        for (LongIterator iterator = elements.longIterator(); iterator.hasNext(); index++)
        {
            newItems[this.items.length + index] = iterator.next();
        }
        return new ImmutableLongArrayList(newItems);
    }

    public ImmutableLongList newWithoutAll(LongIterable elements)
    {
        MutableLongList mutableLongList = this.toList();
        mutableLongList.removeAll(elements);
        return mutableLongList.toImmutable();
    }

    public int size()
    {
        return this.items.length;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.items.length; i++)
        {
            result = function.valueOf(result, this.items[i]);
        }
        return result;
    }

    public <T> T injectIntoWithIndex(T injectedValue, ObjectLongIntToObjectFunction<? super T, ? extends T> function)
    {
        T result = injectedValue;
        for (int i = 0; i < this.items.length; i++)
        {
            result = function.valueOf(result, this.items[i], i);
        }
        return result;
    }

    @Override
    public boolean equals(Object otherList)
    {
        if (otherList == this)
        {
            return true;
        }
        if (!(otherList instanceof LongList))
        {
            return false;
        }
        LongList list = (LongList) otherList;
        if (this.items.length != list.size())
        {
            return false;
        }
        for (int i = 0; i < this.items.length; i++)
        {
            if (this.items[i] != list.get(i))
            {
                return false;
            }
        }
        return true;
    }

    @Override
    public int hashCode()
    {
        int hashCode = 1;
        for (long item : this.items)
        {
            hashCode = 31 * hashCode + (int) (item ^ item >>> 32);
        }
        return hashCode;
    }

    @Override
    public String toString()
    {
        return this.makeString("[", ", ", "]");
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            for (int i = 0; i < this.items.length; i++)
            {
                if (i > 0)
                {
                    appendable.append(separator);
                }
                long value = this.items[i];
                appendable.append(String.valueOf(value));
            }
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    /**
     * @since 6.0.
     */
    public ImmutableLongList distinct()
    {
        LongArrayList target = new LongArrayList();
        MutableLongSet seenSoFar = new LongHashSet(this.size());

        for (long each : this.items)
        {
            if (seenSoFar.add(each))
            {
                target.add(each);
            }
        }
        return target.toImmutable();
    }

    public ImmutableLongList subList(int fromIndex, int toIndex)
    {
        throw new UnsupportedOperationException("subList not yet implemented!");
    }

    private class InternalLongIterator implements LongIterator
    {
        /**
         * Index of element to be returned by subsequent call to next.
         */
        private int currentIndex;

        public boolean hasNext()
        {
            return this.currentIndex != ImmutableLongArrayList.this.items.length;
        }

        public long next()
        {
            if (!this.hasNext())
            {
                throw new NoSuchElementException();
            }
            long next = ImmutableLongArrayList.this.items[this.currentIndex];
            this.currentIndex++;
            return next;
        }
    }
}
