//
// ========================================================================
// Copyright (c) 1995-2020 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under
// the terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0
//
// This Source Code may also be made available under the following
// Secondary Licenses when the conditions for such availability set
// forth in the Eclipse Public License, v. 2.0 are satisfied:
// the Apache License v2.0 which is available at
// https://www.apache.org/licenses/LICENSE-2.0
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.http2;

import java.util.Comparator;

import org.eclipse.jetty.util.Index;

public class HTTP2Cipher
{
    public static final Comparator<String> COMPARATOR = new CipherComparator();

    private static final Index<Boolean> __blackProtocols = new Index.Builder<Boolean>()
        .caseSensitive(false)
        .with("TLSv1.2", Boolean.TRUE)
        .with("TLSv1.1", Boolean.TRUE)
        .with("TLSv1", Boolean.TRUE)
        .with("SSL", Boolean.TRUE)
        .with("SSLv2", Boolean.TRUE)
        .with("SSLv3", Boolean.TRUE)
        .build();

    private static final Index<Boolean> __blackCiphers = new Index.Builder<Boolean>()
        .caseSensitive(false)
        .with("TLS_NULL_WITH_NULL_NULL", Boolean.TRUE)
        .with("TLS_RSA_WITH_NULL_MD5", Boolean.TRUE)
        .with("TLS_RSA_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_RSA_EXPORT_WITH_RC4_40_MD5", Boolean.TRUE)
        .with("TLS_RSA_WITH_RC4_128_MD5", Boolean.TRUE)
        .with("TLS_RSA_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_RSA_EXPORT_WITH_RC2_CBC_40_MD5", Boolean.TRUE)
        .with("TLS_RSA_WITH_IDEA_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_EXPORT_WITH_DES40_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_EXPORT_WITH_DES40_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_EXPORT_WITH_DES40_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_EXPORT_WITH_DES40_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_EXPORT_WITH_DES40_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_EXPORT_WITH_RC4_40_MD5", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_RC4_128_MD5", Boolean.TRUE)
        .with("TLS_DH_anon_EXPORT_WITH_DES40_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_KRB5_WITH_DES_CBC_SHA", Boolean.TRUE)
        .with("TLS_KRB5_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_KRB5_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_KRB5_WITH_IDEA_CBC_SHA", Boolean.TRUE)
        .with("TLS_KRB5_WITH_DES_CBC_MD5", Boolean.TRUE)
        .with("TLS_KRB5_WITH_3DES_EDE_CBC_MD5", Boolean.TRUE)
        .with("TLS_KRB5_WITH_RC4_128_MD5", Boolean.TRUE)
        .with("TLS_KRB5_WITH_IDEA_CBC_MD5", Boolean.TRUE)
        .with("TLS_KRB5_EXPORT_WITH_DES_CBC_40_SHA", Boolean.TRUE)
        .with("TLS_KRB5_EXPORT_WITH_RC2_CBC_40_SHA", Boolean.TRUE)
        .with("TLS_KRB5_EXPORT_WITH_RC4_40_SHA", Boolean.TRUE)
        .with("TLS_KRB5_EXPORT_WITH_DES_CBC_40_MD5", Boolean.TRUE)
        .with("TLS_KRB5_EXPORT_WITH_RC2_CBC_40_MD5", Boolean.TRUE)
        .with("TLS_KRB5_EXPORT_WITH_RC4_40_MD5", Boolean.TRUE)
        .with("TLS_PSK_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_NULL_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_CAMELLIA_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_AES_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_AES_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_AES_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_AES_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_AES_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_CAMELLIA_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_PSK_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_PSK_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_SEED_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_SEED_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_SEED_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_SEED_CBC_SHA", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_SEED_CBC_SHA", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_SEED_CBC_SHA", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_NULL_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_NULL_SHA384", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_NULL_SHA256", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_NULL_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_NULL_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_NULL_SHA384", Boolean.TRUE)
        .with("TLS_RSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_CAMELLIA_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_CAMELLIA_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_CAMELLIA_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_CAMELLIA_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_CAMELLIA_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_CAMELLIA_256_CBC_SHA256", Boolean.TRUE)
        .with("TLS_EMPTY_RENEGOTIATION_INFO_SCSV", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_anon_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_ECDH_anon_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_ECDH_anon_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_anon_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDH_anon_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_RSA_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_DSS_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_RSA_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_DSS_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_RSA_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_SRP_SHA_DSS_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_AES_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_AES_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_RC4_128_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_3DES_EDE_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_AES_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_AES_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_NULL_SHA", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_NULL_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_NULL_SHA384", Boolean.TRUE)
        .with("TLS_RSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_DSS_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_RSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_RSA_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_ARIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_ARIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_ARIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_ARIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_ECDSA_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_RSA_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_RSA_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_RSA_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_RSA_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_DSS_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_DH_anon_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_ECDSA_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_ECDH_RSA_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_CAMELLIA_128_GCM_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_CAMELLIA_256_GCM_SHA384", Boolean.TRUE)
        .with("TLS_PSK_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_PSK_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_DHE_PSK_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_RSA_PSK_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_CAMELLIA_128_CBC_SHA256", Boolean.TRUE)
        .with("TLS_ECDHE_PSK_WITH_CAMELLIA_256_CBC_SHA384", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_128_CCM", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_256_CCM", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_128_CCM_8", Boolean.TRUE)
        .with("TLS_RSA_WITH_AES_256_CCM_8", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_128_CCM", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_256_CCM", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_128_CCM_8", Boolean.TRUE)
        .with("TLS_PSK_WITH_AES_256_CCM_8", Boolean.TRUE)
        .build();

    public static boolean isBlackListProtocol(String tlsProtocol)
    {
        return __blackProtocols.get(tlsProtocol) != null;
    }

    public static boolean isBlackListCipher(String tlsCipher)
    {
        return __blackCiphers.get(tlsCipher) != null;
    }

    /**
     * Comparator that orders non blacklisted ciphers before blacklisted ones.
     */
    public static class CipherComparator implements Comparator<String>
    {
        @Override
        public int compare(String c1, String c2)
        {
            boolean b1 = isBlackListCipher(c1);
            boolean b2 = isBlackListCipher(c2);
            if (b1 == b2)
                return 0;
            if (b1)
                return 1;
            return -1;
        }
    }
}
