/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http2;

import java.io.IOException;
import java.nio.channels.ClosedChannelException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.eclipse.jetty.http2.CloseState;
import org.eclipse.jetty.http2.ErrorCode;
import org.eclipse.jetty.http2.FlowControlStrategy;
import org.eclipse.jetty.http2.HTTP2Flusher;
import org.eclipse.jetty.http2.HTTP2Stream;
import org.eclipse.jetty.http2.ISession;
import org.eclipse.jetty.http2.IStream;
import org.eclipse.jetty.http2.api.Session;
import org.eclipse.jetty.http2.api.Stream;
import org.eclipse.jetty.http2.frames.DataFrame;
import org.eclipse.jetty.http2.frames.DisconnectFrame;
import org.eclipse.jetty.http2.frames.FailureFrame;
import org.eclipse.jetty.http2.frames.Frame;
import org.eclipse.jetty.http2.frames.FrameType;
import org.eclipse.jetty.http2.frames.GoAwayFrame;
import org.eclipse.jetty.http2.frames.HeadersFrame;
import org.eclipse.jetty.http2.frames.PingFrame;
import org.eclipse.jetty.http2.frames.PriorityFrame;
import org.eclipse.jetty.http2.frames.PushPromiseFrame;
import org.eclipse.jetty.http2.frames.ResetFrame;
import org.eclipse.jetty.http2.frames.SettingsFrame;
import org.eclipse.jetty.http2.frames.WindowUpdateFrame;
import org.eclipse.jetty.http2.generator.Generator;
import org.eclipse.jetty.http2.hpack.HpackException;
import org.eclipse.jetty.http2.parser.Parser;
import org.eclipse.jetty.io.ByteBufferPool;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.WriteFlusher;
import org.eclipse.jetty.util.AtomicBiInteger;
import org.eclipse.jetty.util.Atomics;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.CountingCallback;
import org.eclipse.jetty.util.MathUtils;
import org.eclipse.jetty.util.Promise;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.thread.Invocable;
import org.eclipse.jetty.util.thread.Scheduler;

@ManagedObject
public abstract class HTTP2Session
extends ContainerLifeCycle
implements ISession,
Parser.Listener {
    private static final Logger LOG = Log.getLogger(HTTP2Session.class);
    private final ConcurrentMap<Integer, IStream> streams = new ConcurrentHashMap<Integer, IStream>();
    private final AtomicInteger localStreamIds = new AtomicInteger();
    private final AtomicInteger lastRemoteStreamId = new AtomicInteger();
    private final AtomicInteger localStreamCount = new AtomicInteger();
    private final AtomicBiInteger remoteStreamCount = new AtomicBiInteger();
    private final AtomicInteger sendWindow = new AtomicInteger();
    private final AtomicInteger recvWindow = new AtomicInteger();
    private final AtomicReference<CloseState> closed = new AtomicReference<CloseState>(CloseState.NOT_CLOSED);
    private final AtomicLong bytesWritten = new AtomicLong();
    private final Scheduler scheduler;
    private final EndPoint endPoint;
    private final Generator generator;
    private final Session.Listener listener;
    private final FlowControlStrategy flowControl;
    private final HTTP2Flusher flusher;
    private int maxLocalStreams;
    private int maxRemoteStreams;
    private long streamIdleTimeout;
    private int initialSessionRecvWindow;
    private int writeThreshold;
    private boolean pushEnabled;
    private long idleTime;
    private GoAwayFrame closeFrame;

    public HTTP2Session(Scheduler scheduler, EndPoint endPoint, Generator generator, Session.Listener listener, FlowControlStrategy flowControl, int initialStreamId) {
        this.scheduler = scheduler;
        this.endPoint = endPoint;
        this.generator = generator;
        this.listener = listener;
        this.flowControl = flowControl;
        this.flusher = new HTTP2Flusher(this);
        this.maxLocalStreams = -1;
        this.maxRemoteStreams = -1;
        this.localStreamIds.set(initialStreamId);
        this.lastRemoteStreamId.set(HTTP2Session.isClientStream(initialStreamId) ? 0 : -1);
        this.streamIdleTimeout = endPoint.getIdleTimeout();
        this.sendWindow.set(65535);
        this.recvWindow.set(65535);
        this.writeThreshold = 32768;
        this.pushEnabled = true;
        this.idleTime = System.nanoTime();
        this.addBean(flowControl);
        this.addBean((Object)this.flusher);
    }

    protected void doStop() throws Exception {
        super.doStop();
        this.close(ErrorCode.NO_ERROR.code, "stop", new Callback(){

            public void succeeded() {
                HTTP2Session.this.disconnect();
            }

            public void failed(Throwable x) {
                HTTP2Session.this.disconnect();
            }

            public Invocable.InvocationType getInvocationType() {
                return Invocable.InvocationType.NON_BLOCKING;
            }
        });
    }

    @ManagedAttribute(value="The flow control strategy", readonly=true)
    public FlowControlStrategy getFlowControlStrategy() {
        return this.flowControl;
    }

    public int getMaxLocalStreams() {
        return this.maxLocalStreams;
    }

    public void setMaxLocalStreams(int maxLocalStreams) {
        this.maxLocalStreams = maxLocalStreams;
    }

    public int getMaxRemoteStreams() {
        return this.maxRemoteStreams;
    }

    public void setMaxRemoteStreams(int maxRemoteStreams) {
        this.maxRemoteStreams = maxRemoteStreams;
    }

    @ManagedAttribute(value="The stream's idle timeout")
    public long getStreamIdleTimeout() {
        return this.streamIdleTimeout;
    }

    public void setStreamIdleTimeout(long streamIdleTimeout) {
        this.streamIdleTimeout = streamIdleTimeout;
    }

    @ManagedAttribute(value="The initial size of session's flow control receive window")
    public int getInitialSessionRecvWindow() {
        return this.initialSessionRecvWindow;
    }

    public void setInitialSessionRecvWindow(int initialSessionRecvWindow) {
        this.initialSessionRecvWindow = initialSessionRecvWindow;
    }

    public int getWriteThreshold() {
        return this.writeThreshold;
    }

    public void setWriteThreshold(int writeThreshold) {
        this.writeThreshold = writeThreshold;
    }

    public EndPoint getEndPoint() {
        return this.endPoint;
    }

    public Generator getGenerator() {
        return this.generator;
    }

    @Override
    public long getBytesWritten() {
        return this.bytesWritten.get();
    }

    @Override
    public void onData(DataFrame frame) {
        this.onData(frame, Callback.NOOP);
    }

    @Override
    public void onData(DataFrame frame, Callback callback) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        int streamId = frame.getStreamId();
        IStream stream = this.getStream(streamId);
        int flowControlLength = frame.remaining() + frame.padding();
        this.flowControl.onDataReceived(this, stream, flowControlLength);
        if (stream != null) {
            if (this.getRecvWindow() < 0) {
                this.onConnectionFailure(ErrorCode.FLOW_CONTROL_ERROR.code, "session_window_exceeded", callback);
            } else {
                stream.process(frame, (Callback)new DataCallback(callback, stream, flowControlLength));
            }
        } else {
            boolean closed;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Stream #{} not found", (long)streamId);
            }
            this.flowControl.onDataConsumed(this, null, flowControlLength);
            boolean local = (streamId & 1) == (this.localStreamIds.get() & 1);
            boolean bl = closed = local ? this.isLocalStreamClosed(streamId) : this.isRemoteStreamClosed(streamId);
            if (closed) {
                this.reset(new ResetFrame(streamId, ErrorCode.STREAM_CLOSED_ERROR.code), callback);
            } else {
                this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "unexpected_data_frame", callback);
            }
        }
    }

    protected boolean isLocalStreamClosed(int streamId) {
        return streamId <= this.localStreamIds.get();
    }

    protected boolean isRemoteStreamClosed(int streamId) {
        return streamId <= this.getLastRemoteStreamId();
    }

    @Override
    public abstract void onHeaders(HeadersFrame var1);

    @Override
    public void onPriority(PriorityFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
    }

    @Override
    public void onReset(ResetFrame frame) {
        int streamId;
        IStream stream;
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        if ((stream = this.getStream(streamId = frame.getStreamId())) != null) {
            stream.process(frame, new OnResetCallback());
        } else {
            this.onResetForUnknownStream(frame);
        }
    }

    protected abstract void onResetForUnknownStream(ResetFrame var1);

    @Override
    public void onSettings(SettingsFrame frame) {
        this.onSettings(frame, true);
    }

    public void onSettings(SettingsFrame frame, boolean reply) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        if (frame.isReply()) {
            return;
        }
        block8: for (Map.Entry<Integer, Integer> entry : frame.getSettings().entrySet()) {
            int key = entry.getKey();
            int value = entry.getValue();
            switch (key) {
                case 1: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Updating HPACK header table size to {} for {}", new Object[]{value, this});
                    }
                    this.generator.setHeaderTableSize(value);
                    continue block8;
                }
                case 2: {
                    boolean enabled;
                    boolean bl = enabled = value == 1;
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("{} push for {}", new Object[]{enabled ? "Enabling" : "Disabling", this});
                    }
                    this.pushEnabled = enabled;
                    continue block8;
                }
                case 3: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Updating max local concurrent streams to {} for {}", new Object[]{value, this});
                    }
                    this.maxLocalStreams = value;
                    continue block8;
                }
                case 4: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Updating initial window size to {} for {}", new Object[]{value, this});
                    }
                    this.flowControl.updateInitialStreamWindow(this, value, false);
                    continue block8;
                }
                case 5: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Updating max frame size to {} for {}", new Object[]{value, this});
                    }
                    this.generator.setMaxFrameSize(value);
                    continue block8;
                }
                case 6: {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Updating max header list size to {} for {}", new Object[]{value, this});
                    }
                    this.generator.setMaxHeaderListSize(value);
                    continue block8;
                }
            }
            if (!LOG.isDebugEnabled()) continue;
            LOG.debug("Unknown setting {}:{} for {}", new Object[]{key, value, this});
        }
        this.notifySettings(this, frame);
        if (reply) {
            SettingsFrame replyFrame = new SettingsFrame(Collections.emptyMap(), true);
            this.settings(replyFrame, Callback.NOOP);
        }
    }

    @Override
    public void onPing(PingFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        if (frame.isReply()) {
            this.notifyPing(this, frame);
        } else {
            PingFrame reply = new PingFrame(frame.getPayload(), true);
            this.control(null, Callback.NOOP, reply);
        }
    }

    @Override
    public void onGoAway(GoAwayFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.REMOTELY_CLOSED)) continue block3;
                    this.closeFrame = frame;
                    this.notifyClose(this, frame, new DisconnectCallback());
                    return;
                }
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignored {}, already closed", new Object[]{frame});
        }
    }

    @Override
    public void onWindowUpdate(WindowUpdateFrame frame) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Received {}", new Object[]{frame});
        }
        int streamId = frame.getStreamId();
        int windowDelta = frame.getWindowDelta();
        if (streamId > 0) {
            IStream stream = this.getStream(streamId);
            if (stream != null) {
                int streamSendWindow = stream.updateSendWindow(0);
                if (MathUtils.sumOverflows((int)streamSendWindow, (int)windowDelta)) {
                    this.reset(new ResetFrame(streamId, ErrorCode.FLOW_CONTROL_ERROR.code), Callback.NOOP);
                } else {
                    stream.process(frame, Callback.NOOP);
                    this.onWindowUpdate(stream, frame);
                }
            } else if (!this.isRemoteStreamClosed(streamId)) {
                this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "unexpected_window_update_frame");
            }
        } else {
            int sessionSendWindow = this.updateSendWindow(0);
            if (MathUtils.sumOverflows((int)sessionSendWindow, (int)windowDelta)) {
                this.onConnectionFailure(ErrorCode.FLOW_CONTROL_ERROR.code, "invalid_flow_control_window");
            } else {
                this.onWindowUpdate(null, frame);
            }
        }
    }

    @Override
    public void onStreamFailure(int streamId, int error, String reason) {
        ResetCallback callback = new ResetCallback(streamId, error, Callback.NOOP);
        IStream stream = this.getStream(streamId);
        if (stream != null) {
            stream.process(new FailureFrame(error, reason), (Callback)callback);
        } else {
            callback.succeeded();
        }
    }

    @Override
    public void onConnectionFailure(int error, String reason) {
        this.onConnectionFailure(error, reason, Callback.NOOP);
    }

    protected void onConnectionFailure(int error, String reason, Callback callback) {
        this.notifyFailure(this, new IOException(String.format("%d/%s", error, reason)), (Callback)new CloseCallback(error, reason, callback));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void newStream(HeadersFrame frame, Promise<Stream> promise, Stream.Listener listener) {
        try {
            boolean queued;
            HTTP2Session hTTP2Session = this;
            synchronized (hTTP2Session) {
                int streamId = frame.getStreamId();
                if (streamId <= 0) {
                    streamId = this.localStreamIds.getAndAdd(2);
                    PriorityFrame priority = frame.getPriority();
                    priority = priority == null ? null : new PriorityFrame(streamId, priority.getParentStreamId(), priority.getWeight(), priority.isExclusive());
                    frame = new HeadersFrame(streamId, frame.getMetaData(), priority, frame.isEndStream());
                }
                IStream stream = this.createLocalStream(streamId);
                stream.setListener(listener);
                ControlEntry entry = new ControlEntry(frame, stream, new StreamPromiseCallback(promise, stream));
                queued = this.flusher.append(entry);
            }
            if (queued) {
                this.flusher.iterate();
            }
        }
        catch (Throwable x) {
            promise.failed(x);
        }
    }

    @Override
    public int priority(PriorityFrame frame, Callback callback) {
        int streamId = frame.getStreamId();
        IStream stream = (IStream)this.streams.get(streamId);
        if (stream == null) {
            streamId = this.localStreamIds.getAndAdd(2);
            frame = new PriorityFrame(streamId, frame.getParentStreamId(), frame.getWeight(), frame.isExclusive());
        }
        this.control(stream, callback, frame);
        return streamId;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void push(IStream stream, Promise<Stream> promise, PushPromiseFrame frame, Stream.Listener listener) {
        try {
            boolean queued;
            HTTP2Session hTTP2Session = this;
            synchronized (hTTP2Session) {
                int streamId = this.localStreamIds.getAndAdd(2);
                frame = new PushPromiseFrame(frame.getStreamId(), streamId, frame.getMetaData());
                IStream pushStream = this.createLocalStream(streamId);
                pushStream.setListener(listener);
                ControlEntry entry = new ControlEntry(frame, pushStream, new StreamPromiseCallback(promise, pushStream));
                queued = this.flusher.append(entry);
            }
            if (queued) {
                this.flusher.iterate();
            }
        }
        catch (Throwable x) {
            promise.failed(x);
        }
    }

    @Override
    public void settings(SettingsFrame frame, Callback callback) {
        this.control(null, callback, frame);
    }

    @Override
    public void ping(PingFrame frame, Callback callback) {
        if (frame.isReply()) {
            callback.failed((Throwable)new IllegalArgumentException());
        } else {
            this.control(null, callback, frame);
        }
    }

    protected void reset(ResetFrame frame, Callback callback) {
        this.control(this.getStream(frame.getStreamId()), callback, frame);
    }

    @Override
    public boolean close(int error, String reason, Callback callback) {
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.LOCALLY_CLOSED)) continue block3;
                    this.closeFrame = this.newGoAwayFrame(CloseState.LOCALLY_CLOSED, error, reason);
                    this.control(null, callback, this.closeFrame);
                    return true;
                }
            }
            break;
        }
        if (LOG.isDebugEnabled()) {
            LOG.debug("Ignoring close {}/{}, already closed", new Object[]{error, reason});
        }
        callback.succeeded();
        return false;
    }

    private GoAwayFrame newGoAwayFrame(CloseState closeState, int error, String reason) {
        byte[] payload = null;
        if (reason != null) {
            reason = reason.substring(0, Math.min(reason.length(), 32));
            payload = reason.getBytes(StandardCharsets.UTF_8);
        }
        return new GoAwayFrame(closeState, this.getLastRemoteStreamId(), error, payload);
    }

    @Override
    public boolean isClosed() {
        return this.closed.get() != CloseState.NOT_CLOSED;
    }

    private void control(IStream stream, Callback callback, Frame frame) {
        this.frames(stream, callback, frame, Frame.EMPTY_ARRAY);
    }

    @Override
    public void frames(IStream stream, Callback callback, Frame frame, Frame ... frames) {
        int length = frames.length;
        if (length == 0) {
            this.frame(new ControlEntry(frame, stream, callback), true);
        } else {
            callback = new CountingCallback(callback, 1 + length);
            this.frame(new ControlEntry(frame, stream, callback), false);
            for (int i = 1; i <= length; ++i) {
                this.frame(new ControlEntry(frames[i - 1], stream, callback), i == length);
            }
        }
    }

    @Override
    public void data(IStream stream, Callback callback, DataFrame frame) {
        this.frame(new DataEntry(frame, stream, callback), true);
    }

    private void frame(HTTP2Flusher.Entry entry, boolean flush) {
        boolean queued;
        if (LOG.isDebugEnabled()) {
            LOG.debug("{} {}", new Object[]{flush ? "Sending" : "Queueing", entry.frame});
        }
        boolean bl = queued = entry.frame.getType() == FrameType.PING ? this.flusher.prepend(entry) : this.flusher.append(entry);
        if (queued && flush) {
            if (entry.stream != null) {
                entry.stream.notIdle();
            }
            this.flusher.iterate();
        }
    }

    protected IStream createLocalStream(int streamId) {
        int localCount;
        do {
            localCount = this.localStreamCount.get();
            int maxCount = this.getMaxLocalStreams();
            if (maxCount < 0 || localCount < maxCount) continue;
            throw new IllegalStateException("Max local stream count " + maxCount + " exceeded" + System.lineSeparator() + this.dump());
        } while (!this.localStreamCount.compareAndSet(localCount, localCount + 1));
        IStream stream = this.newStream(streamId, true);
        if (this.streams.putIfAbsent(streamId, stream) == null) {
            stream.setIdleTimeout(this.getStreamIdleTimeout());
            this.flowControl.onStreamCreated(stream);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created local {}", new Object[]{stream});
            }
            return stream;
        }
        this.localStreamCount.decrementAndGet();
        throw new IllegalStateException("Duplicate stream " + streamId);
    }

    protected IStream createRemoteStream(int streamId) {
        int remoteClosing;
        int remoteCount;
        long encoded;
        do {
            encoded = this.remoteStreamCount.get();
            remoteCount = AtomicBiInteger.getHi((long)encoded);
            remoteClosing = AtomicBiInteger.getLo((long)encoded);
            int maxCount = this.getMaxRemoteStreams();
            if (maxCount < 0 || remoteCount - remoteClosing < maxCount) continue;
            this.reset(new ResetFrame(streamId, ErrorCode.REFUSED_STREAM_ERROR.code), Callback.NOOP);
            return null;
        } while (!this.remoteStreamCount.compareAndSet(encoded, remoteCount + 1, remoteClosing));
        IStream stream = this.newStream(streamId, false);
        if (this.streams.putIfAbsent(streamId, stream) == null) {
            this.updateLastRemoteStreamId(streamId);
            stream.setIdleTimeout(this.getStreamIdleTimeout());
            this.flowControl.onStreamCreated(stream);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Created remote {}", new Object[]{stream});
            }
            return stream;
        }
        this.remoteStreamCount.addAndGetHi(-1);
        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "duplicate_stream");
        return null;
    }

    void updateStreamCount(boolean local, int deltaStreams, int deltaClosing) {
        if (local) {
            this.localStreamCount.addAndGet(deltaStreams);
        } else {
            this.remoteStreamCount.add(deltaStreams, deltaClosing);
        }
    }

    protected IStream newStream(int streamId, boolean local) {
        return new HTTP2Stream(this.scheduler, this, streamId, local);
    }

    @Override
    public void removeStream(IStream stream) {
        IStream removed = (IStream)this.streams.remove(stream.getId());
        if (removed != null) {
            this.onStreamClosed(stream);
            this.flowControl.onStreamDestroyed(stream);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Removed {} {}", new Object[]{stream.isLocal() ? "local" : "remote", stream});
            }
        }
    }

    @Override
    public Collection<Stream> getStreams() {
        return new ArrayList<Stream>(this.streams.values());
    }

    @ManagedAttribute(value="The number of active streams")
    public int getStreamCount() {
        return this.streams.size();
    }

    @Override
    public IStream getStream(int streamId) {
        return (IStream)this.streams.get(streamId);
    }

    @ManagedAttribute(value="The flow control send window", readonly=true)
    public int getSendWindow() {
        return this.sendWindow.get();
    }

    @ManagedAttribute(value="The flow control receive window", readonly=true)
    public int getRecvWindow() {
        return this.recvWindow.get();
    }

    @Override
    public int updateSendWindow(int delta) {
        return this.sendWindow.getAndAdd(delta);
    }

    @Override
    public int updateRecvWindow(int delta) {
        return this.recvWindow.getAndAdd(delta);
    }

    @Override
    public void onWindowUpdate(IStream stream, WindowUpdateFrame frame) {
        this.flusher.window(stream, frame);
    }

    @Override
    @ManagedAttribute(value="Whether HTTP/2 push is enabled", readonly=true)
    public boolean isPushEnabled() {
        return this.pushEnabled;
    }

    @Override
    public void onShutdown() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Shutting down {}", new Object[]{this});
        }
        switch (this.closed.get()) {
            case NOT_CLOSED: {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Abrupt close for {}", new Object[]{this});
                }
                this.abort(new ClosedChannelException());
                break;
            }
            case LOCALLY_CLOSED: {
                this.control(null, Callback.NOOP, new DisconnectFrame());
                break;
            }
            case REMOTELY_CLOSED: {
                break;
            }
        }
    }

    @Override
    public boolean onIdleTimeout() {
        switch (this.closed.get()) {
            case NOT_CLOSED: {
                long elapsed = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - this.idleTime);
                if (elapsed < this.endPoint.getIdleTimeout()) {
                    return false;
                }
                return this.notifyIdleTimeout(this);
            }
            case LOCALLY_CLOSED: 
            case REMOTELY_CLOSED: {
                this.abort(new TimeoutException("Idle timeout " + this.endPoint.getIdleTimeout() + " ms"));
                return false;
            }
        }
        return false;
    }

    private void notIdle() {
        this.idleTime = System.nanoTime();
    }

    @Override
    public void onFrame(Frame frame) {
        this.onConnectionFailure(ErrorCode.PROTOCOL_ERROR.code, "upgrade");
    }

    protected void onStreamOpened(IStream stream) {
    }

    protected void onStreamClosed(IStream stream) {
    }

    @Override
    public void onFlushed(long bytes) throws IOException {
        this.flusher.onFlushed(bytes);
    }

    public void disconnect() {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Disconnecting {}", new Object[]{this});
        }
        this.endPoint.close();
    }

    private void terminate(Throwable cause) {
        block3: while (true) {
            CloseState current = this.closed.get();
            switch (current) {
                case NOT_CLOSED: 
                case LOCALLY_CLOSED: 
                case REMOTELY_CLOSED: {
                    if (!this.closed.compareAndSet(current, CloseState.CLOSED)) continue block3;
                    this.flusher.terminate(cause);
                    for (IStream stream : this.streams.values()) {
                        stream.close();
                    }
                    this.streams.clear();
                    this.disconnect();
                    return;
                }
            }
            break;
        }
    }

    protected void abort(Throwable failure) {
        this.notifyFailure(this, failure, new TerminateCallback(failure));
    }

    public boolean isDisconnected() {
        return !this.endPoint.isOpen();
    }

    protected int getLastRemoteStreamId() {
        return this.lastRemoteStreamId.get();
    }

    private void updateLastRemoteStreamId(int streamId) {
        Atomics.updateMax((AtomicInteger)this.lastRemoteStreamId, (int)streamId);
    }

    protected Stream.Listener notifyNewStream(Stream stream, HeadersFrame frame) {
        try {
            return this.listener.onNewStream(stream, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
            return null;
        }
    }

    protected void notifySettings(Session session, SettingsFrame frame) {
        try {
            this.listener.onSettings(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyPing(Session session, PingFrame frame) {
        try {
            this.listener.onPing(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyReset(Session session, ResetFrame frame) {
        try {
            this.listener.onReset(session, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyClose(Session session, GoAwayFrame frame, Callback callback) {
        try {
            this.listener.onClose(session, frame, callback);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected boolean notifyIdleTimeout(Session session) {
        try {
            return this.listener.onIdleTimeout(session);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
            return true;
        }
    }

    protected void notifyFailure(Session session, Throwable failure, Callback callback) {
        try {
            this.listener.onFailure(session, failure, callback);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + this.listener, x);
        }
    }

    protected void notifyHeaders(IStream stream, HeadersFrame frame) {
        Stream.Listener listener = stream.getListener();
        if (listener == null) {
            return;
        }
        try {
            listener.onHeaders(stream, frame);
        }
        catch (Throwable x) {
            LOG.info("Failure while notifying listener " + listener, x);
        }
    }

    protected static boolean isClientStream(int streamId) {
        return (streamId & 1) == 1;
    }

    public void dump(Appendable out, String indent) throws IOException {
        this.dumpObjects(out, indent, new Object[]{new DumpableCollection("streams", this.streams.values())});
    }

    public String toString() {
        return String.format("%s@%x{l:%s <-> r:%s,sendWindow=%s,recvWindow=%s,streams=%d,%s,%s}", this.getClass().getSimpleName(), this.hashCode(), this.getEndPoint().getLocalAddress(), this.getEndPoint().getRemoteAddress(), this.sendWindow, this.recvWindow, this.streams.size(), this.closed, this.closeFrame);
    }

    private class TerminateCallback
    implements Callback {
        private final Throwable failure;

        private TerminateCallback(Throwable failure) {
            this.failure = failure;
        }

        public void succeeded() {
            this.complete();
        }

        public void failed(Throwable x) {
            if (x != this.failure) {
                this.failure.addSuppressed(x);
            }
            this.complete();
        }

        public Invocable.InvocationType getInvocationType() {
            return Invocable.InvocationType.NON_BLOCKING;
        }

        private void complete() {
            HTTP2Session.this.terminate(this.failure);
        }
    }

    private class DisconnectCallback
    implements Callback {
        private DisconnectCallback() {
        }

        public void succeeded() {
            this.complete();
        }

        public void failed(Throwable x) {
            this.complete();
        }

        public Invocable.InvocationType getInvocationType() {
            return Invocable.InvocationType.NON_BLOCKING;
        }

        private void complete() {
            HTTP2Session.this.frames(null, Callback.NOOP, HTTP2Session.this.newGoAwayFrame(CloseState.CLOSED, ErrorCode.NO_ERROR.code, null), new DisconnectFrame());
        }
    }

    private class CloseCallback
    extends Callback.Nested {
        private final int error;
        private final String reason;

        private CloseCallback(int error, String reason, Callback callback) {
            super(callback);
            this.error = error;
            this.reason = reason;
        }

        public void succeeded() {
            this.complete();
        }

        public void failed(Throwable x) {
            this.complete();
        }

        private void complete() {
            HTTP2Session.this.close(this.error, this.reason, this.getCallback());
        }
    }

    private class OnResetCallback
    implements Callback {
        private OnResetCallback() {
        }

        public void succeeded() {
            this.complete();
        }

        public void failed(Throwable x) {
            this.complete();
        }

        public Invocable.InvocationType getInvocationType() {
            return Invocable.InvocationType.NON_BLOCKING;
        }

        private void complete() {
            HTTP2Session.this.flusher.iterate();
        }
    }

    private class ResetCallback
    extends Callback.Nested {
        private final int streamId;
        private final int error;

        private ResetCallback(int streamId, int error, Callback callback) {
            super(callback);
            this.streamId = streamId;
            this.error = error;
        }

        public void succeeded() {
            this.complete();
        }

        public void failed(Throwable x) {
            this.complete();
        }

        private void complete() {
            HTTP2Session.this.reset(new ResetFrame(this.streamId, this.error), this.getCallback());
        }
    }

    private class DataCallback
    extends Callback.Nested {
        private final IStream stream;
        private final int flowControlLength;

        public DataCallback(Callback callback, IStream stream, int flowControlLength) {
            super(callback);
            this.stream = stream;
            this.flowControlLength = flowControlLength;
        }

        public void succeeded() {
            this.complete();
            super.succeeded();
        }

        public void failed(Throwable x) {
            this.complete();
            super.failed(x);
        }

        private void complete() {
            HTTP2Session.this.notIdle();
            this.stream.notIdle();
            HTTP2Session.this.flowControl.onDataConsumed(HTTP2Session.this, this.stream, this.flowControlLength);
        }
    }

    private static class StreamPromiseCallback
    implements Callback {
        private final Promise<Stream> promise;
        private final IStream stream;

        private StreamPromiseCallback(Promise<Stream> promise, IStream stream) {
            this.promise = promise;
            this.stream = stream;
        }

        public void succeeded() {
            this.promise.succeeded((Object)this.stream);
        }

        public void failed(Throwable x) {
            this.promise.failed(x);
        }
    }

    private class DataEntry
    extends HTTP2Flusher.Entry {
        private int frameBytes;
        private int frameRemaining;
        private int dataBytes;
        private int dataRemaining;

        private DataEntry(DataFrame frame, IStream stream, Callback callback) {
            super(frame, stream, callback);
            this.dataRemaining = frame.remaining();
        }

        @Override
        public int getFrameBytesGenerated() {
            return this.frameBytes;
        }

        @Override
        public int getDataBytesRemaining() {
            return this.dataRemaining;
        }

        @Override
        protected boolean generate(ByteBufferPool.Lease lease) {
            int dataRemaining = this.getDataBytesRemaining();
            int sessionSendWindow = HTTP2Session.this.getSendWindow();
            int streamSendWindow = this.stream.updateSendWindow(0);
            int window = Math.min(streamSendWindow, sessionSendWindow);
            if (window <= 0 && dataRemaining > 0) {
                return false;
            }
            int length = Math.min(dataRemaining, window);
            DataFrame dataFrame = (DataFrame)this.frame;
            int frameBytes = HTTP2Session.this.generator.data(lease, dataFrame, length);
            this.frameBytes += frameBytes;
            this.frameRemaining += frameBytes;
            int dataBytes = frameBytes - 9;
            this.dataBytes += dataBytes;
            this.dataRemaining -= dataBytes;
            if (LOG.isDebugEnabled()) {
                LOG.debug("Generated {}, length/window/data={}/{}/{}", new Object[]{dataFrame, dataBytes, window, dataRemaining});
            }
            HTTP2Session.this.flowControl.onDataSending(this.stream, dataBytes);
            this.stream.updateClose(dataFrame.isEndStream(), CloseState.Event.BEFORE_SEND);
            return true;
        }

        @Override
        public long onFlushed(long bytes) throws IOException {
            long flushed = Math.min((long)this.frameRemaining, bytes);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Flushed {}/{} frame bytes for {}", new Object[]{flushed, bytes, this});
            }
            this.frameRemaining = (int)((long)this.frameRemaining - flushed);
            Object channel = this.stream.getAttachment();
            if (channel instanceof WriteFlusher.Listener) {
                ((WriteFlusher.Listener)channel).onFlushed(flushed);
            }
            return bytes - flushed;
        }

        public void succeeded() {
            HTTP2Session.this.bytesWritten.addAndGet(this.frameBytes);
            this.frameBytes = 0;
            this.frameRemaining = 0;
            HTTP2Session.this.flowControl.onDataSent(this.stream, this.dataBytes);
            this.dataBytes = 0;
            DataFrame dataFrame = (DataFrame)this.frame;
            if (this.getDataBytesRemaining() == 0) {
                if (this.stream.updateClose(dataFrame.isEndStream(), CloseState.Event.AFTER_SEND)) {
                    HTTP2Session.this.removeStream(this.stream);
                }
                super.succeeded();
            }
        }
    }

    private class ControlEntry
    extends HTTP2Flusher.Entry {
        private int frameBytes;

        private ControlEntry(Frame frame, IStream stream, Callback callback) {
            super(frame, stream, callback);
        }

        @Override
        public int getFrameBytesGenerated() {
            return this.frameBytes;
        }

        @Override
        protected boolean generate(ByteBufferPool.Lease lease) throws HpackException {
            this.frameBytes = HTTP2Session.this.generator.control(lease, this.frame);
            this.beforeSend();
            return true;
        }

        @Override
        public long onFlushed(long bytes) {
            long flushed = Math.min((long)this.frameBytes, bytes);
            if (LOG.isDebugEnabled()) {
                LOG.debug("Flushed {}/{} frame bytes for {}", new Object[]{flushed, bytes, this});
            }
            this.frameBytes = (int)((long)this.frameBytes - flushed);
            return bytes - flushed;
        }

        private void beforeSend() {
            switch (this.frame.getType()) {
                case HEADERS: {
                    HeadersFrame headersFrame = (HeadersFrame)this.frame;
                    this.stream.updateClose(headersFrame.isEndStream(), CloseState.Event.BEFORE_SEND);
                    break;
                }
                case SETTINGS: {
                    SettingsFrame settingsFrame = (SettingsFrame)this.frame;
                    Integer initialWindow = settingsFrame.getSettings().get(4);
                    if (initialWindow == null) break;
                    HTTP2Session.this.flowControl.updateInitialStreamWindow(HTTP2Session.this, initialWindow, true);
                    break;
                }
            }
        }

        public void succeeded() {
            HTTP2Session.this.bytesWritten.addAndGet(this.frameBytes);
            this.frameBytes = 0;
            switch (this.frame.getType()) {
                case HEADERS: {
                    HTTP2Session.this.onStreamOpened(this.stream);
                    HeadersFrame headersFrame = (HeadersFrame)this.frame;
                    if (!this.stream.updateClose(headersFrame.isEndStream(), CloseState.Event.AFTER_SEND)) break;
                    HTTP2Session.this.removeStream(this.stream);
                    break;
                }
                case RST_STREAM: {
                    if (this.stream == null) break;
                    this.stream.close();
                    HTTP2Session.this.removeStream(this.stream);
                    break;
                }
                case PUSH_PROMISE: {
                    this.stream.updateClose(true, CloseState.Event.RECEIVED);
                    break;
                }
                case GO_AWAY: {
                    HTTP2Session.this.getEndPoint().shutdownOutput();
                    break;
                }
                case WINDOW_UPDATE: {
                    HTTP2Session.this.flowControl.windowUpdate(HTTP2Session.this, this.stream, (WindowUpdateFrame)this.frame);
                    break;
                }
                case DISCONNECT: {
                    HTTP2Session.this.terminate(new ClosedChannelException());
                    break;
                }
            }
            super.succeeded();
        }

        @Override
        public void failed(Throwable x) {
            if (this.frame.getType() == FrameType.DISCONNECT) {
                HTTP2Session.this.terminate(new ClosedChannelException());
            }
            super.failed(x);
        }
    }
}

