//
// ========================================================================
// Copyright (c) 1995-2020 Mort Bay Consulting Pty Ltd and others.
//
// This program and the accompanying materials are made available under
// the terms of the Eclipse Public License 2.0 which is available at
// https://www.eclipse.org/legal/epl-2.0
//
// This Source Code may also be made available under the following
// Secondary Licenses when the conditions for such availability set
// forth in the Eclipse Public License, v. 2.0 are satisfied:
// the Apache License v2.0 which is available at
// https://www.apache.org/licenses/LICENSE-2.0
//
// SPDX-License-Identifier: EPL-2.0 OR Apache-2.0
// ========================================================================
//

package org.eclipse.jetty.client;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.Dumpable;

@ManagedObject
public class RoundRobinConnectionPool extends AbstractConnectionPool implements ConnectionPool.Multiplexable
{
    private final List<Entry> entries;
    private int maxMultiplex;
    private int index;

    public RoundRobinConnectionPool(HttpDestination destination, int maxConnections, Callback requester)
    {
        this(destination, maxConnections, requester, 1);
    }

    public RoundRobinConnectionPool(HttpDestination destination, int maxConnections, Callback requester, int maxMultiplex)
    {
        super(destination, maxConnections, requester);
        entries = new ArrayList<>(maxConnections);
        for (int i = 0; i < maxConnections; ++i)
        {
            entries.add(new Entry());
        }
        this.maxMultiplex = maxMultiplex;
    }

    @Override
    public int getMaxMultiplex()
    {
        synchronized (this)
        {
            return maxMultiplex;
        }
    }

    @Override
    public void setMaxMultiplex(int maxMultiplex)
    {
        synchronized (this)
        {
            this.maxMultiplex = maxMultiplex;
        }
    }

    /**
     * <p>Returns an idle connection, if available, following a round robin algorithm;
     * otherwise it always tries to create a new connection, up until the max connection count.</p>
     *
     * @param create this parameter is ignored and assumed to be always {@code true}
     * @return an idle connection or {@code null} if no idle connections are available
     */
    @Override
    public Connection acquire(boolean create)
    {
        // The nature of this connection pool is such that a
        // connection must always be present in the next slot.
        return super.acquire(true);
    }

    @Override
    protected void onCreated(Connection connection)
    {
        synchronized (this)
        {
            for (Entry entry : entries)
            {
                if (entry.connection == null)
                {
                    entry.connection = connection;
                    break;
                }
            }
        }
        idle(connection, false);
    }

    @Override
    protected Connection activate()
    {
        Connection connection = null;
        synchronized (this)
        {
            int offset = 0;
            int capacity = getMaxConnectionCount();
            while (offset < capacity)
            {
                int idx = index + offset;
                if (idx >= capacity)
                    idx -= capacity;

                Entry entry = entries.get(idx);

                if (entry.connection == null)
                    break;

                if (entry.active < getMaxMultiplex())
                {
                    ++entry.active;
                    ++entry.used;
                    connection = entry.connection;
                    index += offset + 1;
                    if (index >= capacity)
                        index -= capacity;
                    break;
                }

                ++offset;
            }
        }
        return connection == null ? null : active(connection);
    }

    @Override
    public boolean isActive(Connection connection)
    {
        synchronized (this)
        {
            for (Entry entry : entries)
            {
                if (entry.connection == connection)
                    return entry.active > 0;
            }
            return false;
        }
    }

    @Override
    public boolean release(Connection connection)
    {
        boolean found = false;
        boolean idle = false;
        synchronized (this)
        {
            for (Entry entry : entries)
            {
                if (entry.connection == connection)
                {
                    found = true;
                    int active = --entry.active;
                    idle = active == 0;
                    break;
                }
            }
        }
        if (!found)
            return false;
        released(connection);
        if (idle)
            return idle(connection, isClosed());
        return true;
    }

    @Override
    public boolean remove(Connection connection)
    {
        boolean found = false;
        synchronized (this)
        {
            for (Entry entry : entries)
            {
                if (entry.connection == connection)
                {
                    found = true;
                    entry.reset();
                    break;
                }
            }
        }
        if (found)
        {
            released(connection);
            removed(connection);
        }
        return found;
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException
    {
        List<Entry> connections;
        synchronized (this)
        {
            connections = new ArrayList<>(entries);
        }
        Dumpable.dumpObjects(out, indent, out, connections);
    }

    @Override
    public String toString()
    {
        int present = 0;
        int active = 0;
        synchronized (this)
        {
            for (Entry entry : entries)
            {
                if (entry.connection != null)
                {
                    ++present;
                    if (entry.active > 0)
                        ++active;
                }
            }
        }
        return String.format("%s@%x[c=%d/%d/%d,a=%d]",
            getClass().getSimpleName(),
            hashCode(),
            getPendingConnectionCount(),
            present,
            getMaxConnectionCount(),
            active
        );
    }

    private static class Entry
    {
        private Connection connection;
        private int active;
        private long used;

        private void reset()
        {
            connection = null;
            active = 0;
            used = 0;
        }

        @Override
        public String toString()
        {
            return String.format("{u=%d,c=%s}", used, connection);
        }
    }
}
