/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.IOException;
import java.nio.channels.AsynchronousCloseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.RequestNotifier;
import org.eclipse.jetty.client.ResponseNotifier;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.client.api.Request;
import org.eclipse.jetty.client.api.Response;
import org.eclipse.jetty.client.api.Result;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.FutureCallback;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class HttpDestination
implements Destination,
AutoCloseable,
Dumpable {
    private static final Logger LOG = Log.getLogger(HttpDestination.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final HttpClient client;
    private final String scheme;
    private final String host;
    private final int port;
    private final Queue<RequestPair> requests;
    private final BlockingQueue<Connection> idleConnections;
    private final BlockingQueue<Connection> activeConnections;
    private final RequestNotifier requestNotifier;
    private final ResponseNotifier responseNotifier;

    public HttpDestination(HttpClient client, String scheme, String host, int port) {
        this.client = client;
        this.scheme = scheme;
        this.host = host;
        this.port = port;
        this.requests = new ArrayBlockingQueue<RequestPair>(client.getMaxQueueSizePerAddress());
        this.idleConnections = new ArrayBlockingQueue<Connection>(client.getMaxConnectionsPerAddress());
        this.activeConnections = new ArrayBlockingQueue<Connection>(client.getMaxConnectionsPerAddress());
        this.requestNotifier = new RequestNotifier(client);
        this.responseNotifier = new ResponseNotifier(client);
    }

    protected BlockingQueue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    protected BlockingQueue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    @Override
    public String scheme() {
        return this.scheme;
    }

    @Override
    public String host() {
        return this.host;
    }

    @Override
    public int port() {
        return this.port;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void send(Request request, Response.Listener listener) {
        if (!this.scheme.equals(request.scheme())) {
            throw new IllegalArgumentException("Invalid request scheme " + request.scheme() + " for destination " + this);
        }
        if (!this.host.equals(request.host())) {
            throw new IllegalArgumentException("Invalid request host " + request.host() + " for destination " + this);
        }
        int port = request.port();
        if (port >= 0 && this.port != port) {
            throw new IllegalArgumentException("Invalid request port " + port + " for destination " + this);
        }
        RequestPair requestPair = new RequestPair(request, listener);
        if (!this.client.isRunning()) throw new RejectedExecutionException((Object)((Object)this.client) + " is stopped");
        if (!this.requests.offer(requestPair)) throw new RejectedExecutionException("Max requests per address " + this.client.getMaxQueueSizePerAddress() + " exceeded");
        if (!this.client.isRunning() && this.requests.remove(requestPair)) {
            throw new RejectedExecutionException((Object)((Object)this.client) + " is stopping");
        }
        LOG.debug("Queued {}", new Object[]{request});
        this.requestNotifier.notifyQueued(request);
        Connection connection = this.acquire();
        if (connection == null) return;
        this.process(connection, false);
    }

    @Override
    public Future<Connection> newConnection() {
        FutureCallback result = new FutureCallback();
        this.newConnection((Callback<Connection>)result);
        return result;
    }

    protected void newConnection(Callback<Connection> callback) {
        this.client.newConnection(this, callback);
    }

    protected Connection acquire() {
        int next;
        int current;
        Connection result = (Connection)this.idleConnections.poll();
        if (result != null) {
            return result;
        }
        final int maxConnections = this.client.getMaxConnectionsPerAddress();
        do {
            if ((next = (current = this.connectionCount.get()) + 1) <= maxConnections) continue;
            LOG.debug("Max connections {} reached for {}", new Object[]{current, this});
            return (Connection)this.idleConnections.poll();
        } while (!this.connectionCount.compareAndSet(current, next));
        LOG.debug("Creating connection {}/{} for {}", new Object[]{next, maxConnections, this});
        this.newConnection(new Callback<Connection>(){

            public void completed(Connection connection) {
                LOG.debug("Created connection {}/{} {} for {}", new Object[]{next, maxConnections, connection, HttpDestination.this});
                HttpDestination.this.process(connection, true);
            }

            public void failed(Connection connection, final Throwable x) {
                LOG.debug("Connection failed {} for {}", new Object[]{x, HttpDestination.this});
                HttpDestination.this.connectionCount.decrementAndGet();
                HttpDestination.this.client.getExecutor().execute(new Runnable(){

                    @Override
                    public void run() {
                        RequestPair pair = (RequestPair)HttpDestination.this.requests.poll();
                        if (pair != null) {
                            HttpDestination.this.requestNotifier.notifyFailure(pair.request, x);
                            HttpDestination.this.responseNotifier.notifyComplete(pair.listener, new Result(pair.request, x, null));
                        }
                    }
                });
            }
        });
        return (Connection)this.idleConnections.poll();
    }

    protected void process(final Connection connection, boolean dispatch) {
        final RequestPair requestPair = this.requests.poll();
        if (requestPair == null) {
            LOG.debug("{} idle", new Object[]{connection});
            if (!this.idleConnections.offer(connection)) {
                LOG.debug("{} idle overflow", new Object[0]);
                connection.close();
            }
            if (!this.client.isRunning()) {
                LOG.debug("{} is stopping", new Object[]{this.client});
                this.remove(connection);
                connection.close();
            }
        } else {
            LOG.debug("{} active", new Object[]{connection});
            if (!this.activeConnections.offer(connection)) {
                LOG.warn("{} active overflow", new Object[0]);
            }
            if (dispatch) {
                this.client.getExecutor().execute(new Runnable(){

                    @Override
                    public void run() {
                        connection.send(requestPair.request, requestPair.listener);
                    }
                });
            } else {
                connection.send(requestPair.request, requestPair.listener);
            }
        }
    }

    public void release(Connection connection) {
        LOG.debug("{} released", new Object[]{connection});
        if (this.client.isRunning()) {
            boolean removed = this.activeConnections.remove(connection);
            if (removed) {
                this.process(connection, false);
            } else {
                LOG.debug("{} explicit", new Object[]{connection});
            }
        } else {
            LOG.debug("{} is stopped", new Object[]{this.client});
            this.remove(connection);
            connection.close();
        }
    }

    public void remove(Connection connection) {
        LOG.debug("{} removed", new Object[]{connection});
        this.connectionCount.decrementAndGet();
        this.activeConnections.remove(connection);
        this.idleConnections.remove(connection);
        if (!this.requests.isEmpty() && (connection = this.acquire()) != null) {
            this.process(connection, false);
        }
    }

    @Override
    public void close() {
        RequestPair pair;
        for (Connection connection : this.idleConnections) {
            connection.close();
        }
        this.idleConnections.clear();
        for (Connection connection : this.activeConnections) {
            connection.close();
        }
        this.activeConnections.clear();
        AsynchronousCloseException failure = new AsynchronousCloseException();
        while ((pair = this.requests.poll()) != null) {
            this.requestNotifier.notifyFailure(pair.request, failure);
            this.responseNotifier.notifyComplete(pair.listener, new Result(pair.request, failure, null));
        }
        this.connectionCount.set(0);
        LOG.debug("Closed {}", new Object[]{this});
    }

    public String dump() {
        return ContainerLifeCycle.dump((Dumpable)this);
    }

    public void dump(Appendable out, String indent) throws IOException {
        ContainerLifeCycle.dumpObject((Appendable)out, (Object)(this + " - requests queued: " + this.requests.size()));
        ArrayList<String> connections = new ArrayList<String>();
        for (Connection connection : this.idleConnections) {
            connections.add(connection + " - IDLE");
        }
        for (Connection connection : this.activeConnections) {
            connections.add(connection + " - ACTIVE");
        }
        ContainerLifeCycle.dump((Appendable)out, (String)indent, (Collection[])new Collection[]{connections});
    }

    public String toString() {
        return String.format("%s(%s://%s:%d)", HttpDestination.class.getSimpleName(), this.scheme(), this.host(), this.port());
    }

    private static class RequestPair {
        private final Request request;
        private final Response.Listener listener;

        public RequestPair(Request request, Response.Listener listener) {
            this.request = request;
            this.listener = listener;
        }
    }
}

