/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.client;

import java.io.IOException;
import java.nio.channels.AsynchronousCloseException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.client.HttpConnection;
import org.eclipse.jetty.client.HttpRequestException;
import org.eclipse.jetty.client.HttpResponse;
import org.eclipse.jetty.client.HttpResponseException;
import org.eclipse.jetty.client.RequestNotifier;
import org.eclipse.jetty.client.ResponseNotifier;
import org.eclipse.jetty.client.api.Connection;
import org.eclipse.jetty.client.api.Destination;
import org.eclipse.jetty.client.api.Request;
import org.eclipse.jetty.client.api.Response;
import org.eclipse.jetty.client.api.Result;
import org.eclipse.jetty.util.Callback;
import org.eclipse.jetty.util.FutureCallback;
import org.eclipse.jetty.util.component.ContainerLifeCycle;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class HttpDestination
implements Destination,
AutoCloseable,
Dumpable {
    private static final Logger LOG = Log.getLogger(HttpDestination.class);
    private final AtomicInteger connectionCount = new AtomicInteger();
    private final HttpClient client;
    private final String scheme;
    private final String host;
    private final int port;
    private final Queue<RequestContext> requests;
    private final BlockingQueue<Connection> idleConnections;
    private final BlockingQueue<Connection> activeConnections;
    private final RequestNotifier requestNotifier;
    private final ResponseNotifier responseNotifier;

    public HttpDestination(HttpClient client, String scheme, String host, int port) {
        this.client = client;
        this.scheme = scheme;
        this.host = host;
        this.port = port;
        this.requests = new ArrayBlockingQueue<RequestContext>(client.getMaxQueueSizePerAddress());
        this.idleConnections = new ArrayBlockingQueue<Connection>(client.getMaxConnectionsPerAddress());
        this.activeConnections = new ArrayBlockingQueue<Connection>(client.getMaxConnectionsPerAddress());
        this.requestNotifier = new RequestNotifier(client);
        this.responseNotifier = new ResponseNotifier(client);
    }

    protected BlockingQueue<Connection> getIdleConnections() {
        return this.idleConnections;
    }

    protected BlockingQueue<Connection> getActiveConnections() {
        return this.activeConnections;
    }

    @Override
    public String getScheme() {
        return this.scheme;
    }

    @Override
    public String getHost() {
        return this.host;
    }

    @Override
    public int getPort() {
        return this.port;
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void send(Request request, List<Response.ResponseListener> listeners) {
        if (!this.scheme.equals(request.getScheme())) {
            throw new IllegalArgumentException("Invalid request scheme " + request.getScheme() + " for destination " + this);
        }
        if (!this.host.equals(request.getHost())) {
            throw new IllegalArgumentException("Invalid request host " + request.getHost() + " for destination " + this);
        }
        int port = request.getPort();
        if (port >= 0 && this.port != port) {
            throw new IllegalArgumentException("Invalid request port " + port + " for destination " + this);
        }
        RequestContext requestContext = new RequestContext(request, listeners);
        if (!this.client.isRunning()) throw new RejectedExecutionException((Object)((Object)this.client) + " is stopped");
        if (!this.requests.offer(requestContext)) throw new RejectedExecutionException("Max requests per address " + this.client.getMaxQueueSizePerAddress() + " exceeded");
        if (!this.client.isRunning() && this.requests.remove(requestContext)) {
            throw new RejectedExecutionException((Object)((Object)this.client) + " is stopping");
        }
        LOG.debug("Queued {}", new Object[]{request});
        this.requestNotifier.notifyQueued(request);
        Connection connection = this.acquire();
        if (connection == null) return;
        this.process(connection, false);
    }

    @Override
    public Future<Connection> newConnection() {
        FutureCallback result = new FutureCallback();
        this.newConnection((Callback<Connection>)result);
        return result;
    }

    protected void newConnection(Callback<Connection> callback) {
        this.client.newConnection(this, callback);
    }

    protected Connection acquire() {
        int next;
        int current;
        Connection result = (Connection)this.idleConnections.poll();
        if (result != null) {
            return result;
        }
        final int maxConnections = this.client.getMaxConnectionsPerAddress();
        do {
            if ((next = (current = this.connectionCount.get()) + 1) <= maxConnections) continue;
            LOG.debug("Max connections {} reached for {}", new Object[]{current, this});
            return (Connection)this.idleConnections.poll();
        } while (!this.connectionCount.compareAndSet(current, next));
        LOG.debug("Creating connection {}/{} for {}", new Object[]{next, maxConnections, this});
        this.newConnection(new Callback<Connection>(){

            public void completed(Connection connection) {
                LOG.debug("Created connection {}/{} {} for {}", new Object[]{next, maxConnections, connection, HttpDestination.this});
                HttpDestination.this.process(connection, true);
            }

            public void failed(Connection connection, final Throwable x) {
                LOG.debug("Connection failed {} for {}", new Object[]{x, HttpDestination.this});
                HttpDestination.this.connectionCount.decrementAndGet();
                HttpDestination.this.client.getExecutor().execute(new Runnable(){

                    @Override
                    public void run() {
                        HttpDestination.this.drain(x);
                    }
                });
            }
        });
        return (Connection)this.idleConnections.poll();
    }

    private void drain(Throwable x) {
        RequestContext requestContext;
        while ((requestContext = this.requests.poll()) != null) {
            Request request = requestContext.request;
            this.requestNotifier.notifyFailure(request, x);
            List listeners = requestContext.listeners;
            HttpResponse response = new HttpResponse(request, listeners);
            this.responseNotifier.notifyFailure(listeners, (Response)response, x);
            this.responseNotifier.notifyComplete(listeners, new Result(request, x, response, x));
        }
    }

    protected void process(Connection connection, boolean dispatch) {
        final HttpConnection httpConnection = (HttpConnection)connection;
        RequestContext requestContext = this.requests.poll();
        if (requestContext == null) {
            LOG.debug("{} idle", new Object[]{httpConnection});
            if (!this.idleConnections.offer(httpConnection)) {
                LOG.debug("{} idle overflow", new Object[0]);
                httpConnection.close();
            }
            if (!this.client.isRunning()) {
                LOG.debug("{} is stopping", new Object[]{this.client});
                this.remove(httpConnection);
                httpConnection.close();
            }
        } else {
            final Request request = requestContext.request;
            final List listeners = requestContext.listeners;
            if (request.isAborted()) {
                this.abort(request, listeners, "Aborted");
                LOG.debug("Aborted {} before processing", new Object[]{request});
            } else {
                LOG.debug("{} active", new Object[]{httpConnection});
                if (!this.activeConnections.offer(httpConnection)) {
                    LOG.warn("{} active overflow", new Object[0]);
                }
                if (dispatch) {
                    this.client.getExecutor().execute(new Runnable(){

                        @Override
                        public void run() {
                            httpConnection.send(request, listeners);
                        }
                    });
                } else {
                    httpConnection.send(request, listeners);
                }
            }
        }
    }

    public void release(Connection connection) {
        LOG.debug("{} released", new Object[]{connection});
        if (this.client.isRunning()) {
            boolean removed = this.activeConnections.remove(connection);
            if (removed) {
                this.process(connection, false);
            } else {
                LOG.debug("{} explicit", new Object[]{connection});
            }
        } else {
            LOG.debug("{} is stopped", new Object[]{this.client});
            this.remove(connection);
            connection.close();
        }
    }

    public void remove(Connection connection) {
        LOG.debug("{} removed", new Object[]{connection});
        this.connectionCount.decrementAndGet();
        this.activeConnections.remove(connection);
        this.idleConnections.remove(connection);
        if (!this.requests.isEmpty() && (connection = this.acquire()) != null) {
            this.process(connection, false);
        }
    }

    @Override
    public void close() {
        for (Connection connection : this.idleConnections) {
            connection.close();
        }
        this.idleConnections.clear();
        for (Connection connection : this.activeConnections) {
            connection.close();
        }
        this.activeConnections.clear();
        this.drain(new AsynchronousCloseException());
        this.connectionCount.set(0);
        LOG.debug("Closed {}", new Object[]{this});
    }

    public boolean abort(Request request, String reason) {
        for (RequestContext requestContext : this.requests) {
            if (requestContext.request != request || !this.requests.remove(requestContext)) continue;
            this.abort(request, requestContext.listeners, reason);
            LOG.debug("Aborted {} while queued", new Object[]{request});
            return true;
        }
        return false;
    }

    private void abort(Request request, List<Response.ResponseListener> listeners, String reason) {
        HttpResponse response = new HttpResponse(request, listeners);
        HttpResponseException responseFailure = new HttpResponseException(reason, response);
        this.responseNotifier.notifyFailure(listeners, (Response)response, (Throwable)responseFailure);
        HttpRequestException requestFailure = new HttpRequestException(reason, request);
        this.responseNotifier.notifyComplete(listeners, new Result(request, requestFailure, response, responseFailure));
    }

    public String dump() {
        return ContainerLifeCycle.dump((Dumpable)this);
    }

    public void dump(Appendable out, String indent) throws IOException {
        ContainerLifeCycle.dumpObject((Appendable)out, (Object)(this + " - requests queued: " + this.requests.size()));
        ArrayList<String> connections = new ArrayList<String>();
        for (Connection connection : this.idleConnections) {
            connections.add(connection + " - IDLE");
        }
        for (Connection connection : this.activeConnections) {
            connections.add(connection + " - ACTIVE");
        }
        ContainerLifeCycle.dump((Appendable)out, (String)indent, (Collection[])new Collection[]{connections});
    }

    public String toString() {
        return String.format("%s(%s://%s:%d)", HttpDestination.class.getSimpleName(), this.getScheme(), this.getHost(), this.getPort());
    }

    private static class RequestContext {
        private final Request request;
        private final List<Response.ResponseListener> listeners;

        private RequestContext(Request request, List<Response.ResponseListener> listeners) {
            this.request = request;
            this.listeners = listeners;
        }
    }
}

