/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.eclipse.jetty.http.CompressedContentFormat;
import org.eclipse.jetty.http.GZIPContentDecoder;
import org.eclipse.jetty.io.ArrayByteBufferPool;
import org.eclipse.jetty.io.ByteBufferPool;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.ValueSource;

public class GZIPContentDecoderTest {
    private ArrayByteBufferPool pool;
    private AtomicInteger buffers = new AtomicInteger(0);
    final long INT_MAX = Integer.MAX_VALUE;
    final long UINT_MAX = 0xFFFFFFFFL;

    @BeforeEach
    public void before() {
        this.buffers.set(0);
        this.pool = new ArrayByteBufferPool(){

            public ByteBuffer acquire(int size, boolean direct) {
                GZIPContentDecoderTest.this.buffers.incrementAndGet();
                return super.acquire(size, direct);
            }

            public void release(ByteBuffer buffer) {
                GZIPContentDecoderTest.this.buffers.decrementAndGet();
                super.release(buffer);
            }
        };
    }

    @AfterEach
    public void after() {
        Assertions.assertEquals((int)0, (int)this.buffers.get());
    }

    @Test
    public void testCompressedContentFormat() {
        Assertions.assertTrue((boolean)CompressedContentFormat.tagEquals((String)"tag", (String)"tag"));
        Assertions.assertTrue((boolean)CompressedContentFormat.tagEquals((String)"\"tag\"", (String)"\"tag\""));
        Assertions.assertTrue((boolean)CompressedContentFormat.tagEquals((String)"\"tag\"", (String)"\"tag--gzip\""));
        Assertions.assertFalse((boolean)CompressedContentFormat.tagEquals((String)"Zag", (String)"Xag--gzip"));
        Assertions.assertFalse((boolean)CompressedContentFormat.tagEquals((String)"xtag", (String)"tag"));
    }

    @Test
    public void testStreamNoBlocks() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.close();
        byte[] bytes = baos.toByteArray();
        GZIPInputStream input = new GZIPInputStream((InputStream)new ByteArrayInputStream(bytes), 1);
        int read = input.read();
        Assertions.assertEquals((int)-1, (int)read);
    }

    @Test
    public void testStreamBigBlockOneByteAtATime() throws Exception {
        int read;
        String data = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data = data + data;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        baos = new ByteArrayOutputStream();
        GZIPInputStream input = new GZIPInputStream((InputStream)new ByteArrayInputStream(bytes), 1);
        while ((read = input.read()) >= 0) {
            baos.write(read);
        }
        Assertions.assertEquals((Object)data, (Object)new String(baos.toByteArray(), StandardCharsets.UTF_8));
    }

    @Test
    public void testNoBlocks() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.close();
        byte[] bytes = baos.toByteArray();
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes));
        Assertions.assertEquals((int)0, (int)decoded.remaining());
    }

    @Test
    public void testSmallBlock() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes));
        Assertions.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        decoder.release(decoded);
    }

    @Test
    public void testSmallBlockWithGZIPChunkedAtBegin() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        byte[] bytes1 = new byte[11];
        System.arraycopy(bytes, 0, bytes1, 0, bytes1.length);
        byte[] bytes2 = new byte[bytes.length - bytes1.length];
        System.arraycopy(bytes, bytes1.length, bytes2, 0, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes1));
        Assertions.assertEquals((int)0, (int)decoded.capacity());
        decoded = decoder.decode(ByteBuffer.wrap(bytes2));
        Assertions.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        decoder.release(decoded);
    }

    @Test
    public void testSmallBlockWithGZIPChunkedAtEnd() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        byte[] bytes1 = new byte[bytes.length - 9];
        System.arraycopy(bytes, 0, bytes1, 0, bytes1.length);
        byte[] bytes2 = new byte[bytes.length - bytes1.length];
        System.arraycopy(bytes, bytes1.length, bytes2, 0, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes1));
        Assertions.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        Assertions.assertFalse((boolean)decoder.isFinished());
        decoder.release(decoded);
        decoded = decoder.decode(ByteBuffer.wrap(bytes2));
        Assertions.assertEquals((int)0, (int)decoded.remaining());
        Assertions.assertTrue((boolean)decoder.isFinished());
        decoder.release(decoded);
    }

    @Test
    public void testSmallBlockWithGZIPTrailerChunked() throws Exception {
        String data = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        byte[] bytes1 = new byte[bytes.length - 3];
        System.arraycopy(bytes, 0, bytes1, 0, bytes1.length);
        byte[] bytes2 = new byte[bytes.length - bytes1.length];
        System.arraycopy(bytes, bytes1.length, bytes2, 0, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(bytes1));
        Assertions.assertEquals((int)0, (int)decoded.capacity());
        decoder.release(decoded);
        decoded = decoder.decode(ByteBuffer.wrap(bytes2));
        Assertions.assertEquals((Object)data, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        decoder.release(decoded);
    }

    @Test
    public void testTwoSmallBlocks() throws Exception {
        String data1 = "0";
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data1.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes1 = baos.toByteArray();
        String data2 = "1";
        baos = new ByteArrayOutputStream();
        output = new GZIPOutputStream(baos);
        output.write(data2.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes2 = baos.toByteArray();
        byte[] bytes = new byte[bytes1.length + bytes2.length];
        System.arraycopy(bytes1, 0, bytes, 0, bytes1.length);
        System.arraycopy(bytes2, 0, bytes, bytes1.length, bytes2.length);
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        ByteBuffer decoded = decoder.decode(buffer);
        Assertions.assertEquals((Object)data1, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        Assertions.assertTrue((boolean)decoder.isFinished());
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        decoder.release(decoded);
        decoded = decoder.decode(buffer);
        Assertions.assertEquals((Object)data2, (Object)StandardCharsets.UTF_8.decode(decoded).toString());
        Assertions.assertTrue((boolean)decoder.isFinished());
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        decoder.release(decoded);
    }

    @Test
    public void testBigBlock() throws Exception {
        String data = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data = data + data;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        String result = "";
        GZIPContentDecoder decoder = new GZIPContentDecoder((ByteBufferPool)this.pool, 2048);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        while (buffer.hasRemaining()) {
            ByteBuffer decoded = decoder.decode(buffer);
            result = result + StandardCharsets.UTF_8.decode(decoded).toString();
            decoder.release(decoded);
        }
        Assertions.assertEquals((Object)data, (Object)result);
    }

    @Test
    public void testBigBlockOneByteAtATime() throws Exception {
        String data = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data = data + data;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes = baos.toByteArray();
        String result = "";
        GZIPContentDecoder decoder = new GZIPContentDecoder(64);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        while (buffer.hasRemaining()) {
            ByteBuffer decoded = decoder.decode(ByteBuffer.wrap(new byte[]{buffer.get()}));
            if (decoded.hasRemaining()) {
                result = result + StandardCharsets.UTF_8.decode(decoded).toString();
            }
            decoder.release(decoded);
        }
        Assertions.assertEquals((Object)data, (Object)result);
        Assertions.assertTrue((boolean)decoder.isFinished());
    }

    @Test
    public void testBigBlockWithExtraBytes() throws Exception {
        String data1 = "0123456789ABCDEF";
        for (int i = 0; i < 10; ++i) {
            data1 = data1 + data1;
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GZIPOutputStream output = new GZIPOutputStream(baos);
        output.write(data1.getBytes(StandardCharsets.UTF_8));
        output.close();
        byte[] bytes1 = baos.toByteArray();
        String data2 = "HELLO";
        byte[] bytes2 = data2.getBytes(StandardCharsets.UTF_8);
        byte[] bytes = new byte[bytes1.length + bytes2.length];
        System.arraycopy(bytes1, 0, bytes, 0, bytes1.length);
        System.arraycopy(bytes2, 0, bytes, bytes1.length, bytes2.length);
        String result = "";
        GZIPContentDecoder decoder = new GZIPContentDecoder(64);
        ByteBuffer buffer = ByteBuffer.wrap(bytes);
        while (buffer.hasRemaining()) {
            ByteBuffer decoded = decoder.decode(buffer);
            if (decoded.hasRemaining()) {
                result = result + StandardCharsets.UTF_8.decode(decoded).toString();
            }
            decoder.release(decoded);
            if (!decoder.isFinished()) continue;
            break;
        }
        Assertions.assertEquals((Object)data1, (Object)result);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)data2, (Object)StandardCharsets.UTF_8.decode(buffer).toString());
    }

    @ParameterizedTest
    @ValueSource(longs={0x7FFFFFFFL, 0x80000000L, 0xFFFFFFFFL, 0x100000000L})
    public void testLargeGzipStream(long origSize) throws IOException {
        int len;
        int len2;
        int BUFSIZE = 0x100000;
        byte[] cbuf = "0123456789ABCDEFGHIJKLMOPQRSTUVWXYZ".getBytes(StandardCharsets.UTF_8);
        byte[] buf = new byte[0x100000];
        for (int off = 0; off < buf.length; off += len2) {
            len2 = Math.min(cbuf.length, buf.length - off);
            System.arraycopy(cbuf, 0, buf, off, len2);
        }
        GZIPDecoderOutputStream out = new GZIPDecoderOutputStream(new GZIPContentDecoder(0x100000));
        GZIPOutputStream outputStream = new GZIPOutputStream((OutputStream)out, 0x100000);
        for (long bytesLeft = origSize; bytesLeft > 0L; bytesLeft -= (long)len) {
            len = buf.length;
            if (bytesLeft < (long)buf.length) {
                len = (int)bytesLeft;
            }
            outputStream.write(buf, 0, len);
        }
        outputStream.close();
        MatcherAssert.assertThat((String)"Decoded byte count", (Object)out.decodedByteCount, (Matcher)Matchers.is((Object)origSize));
    }

    public static class GZIPDecoderOutputStream
    extends OutputStream {
        private final GZIPContentDecoder decoder;
        public long decodedByteCount = 0L;

        public GZIPDecoderOutputStream(GZIPContentDecoder decoder) {
            this.decoder = decoder;
        }

        @Override
        public void write(byte[] b, int off, int len) throws IOException {
            ByteBuffer buf = ByteBuffer.wrap(b, off, len);
            while (buf.hasRemaining()) {
                ByteBuffer decoded = this.decoder.decode(buf);
                if (decoded.hasRemaining()) {
                    this.decodedByteCount += (long)decoded.remaining();
                }
                this.decoder.release(decoded);
            }
        }

        @Override
        public void write(int b) throws IOException {
            this.write(new byte[]{(byte)b}, 0, 1);
        }
    }
}

