/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import org.eclipse.jetty.http.BadMessageException;
import org.eclipse.jetty.http.HostPortHttpField;
import org.eclipse.jetty.http.HttpCompliance;
import org.eclipse.jetty.http.HttpComplianceSection;
import org.eclipse.jetty.http.HttpField;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.HttpParser;
import org.eclipse.jetty.http.HttpVersion;
import org.eclipse.jetty.toolchain.test.Net;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.StacklessLogging;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class HttpParserTest {
    private String _host;
    private int _port;
    private String _bad;
    private String _content;
    private String _methodOrVersion;
    private String _uriOrStatus;
    private String _versionOrReason;
    private List<HttpField> _fields = new ArrayList<HttpField>();
    private List<HttpField> _trailers = new ArrayList<HttpField>();
    private String[] _hdr;
    private String[] _val;
    private int _headers;
    private boolean _early;
    private boolean _headerCompleted;
    private boolean _contentCompleted;
    private boolean _messageCompleted;
    private final List<HttpComplianceSection> _complianceViolation = new ArrayList<HttpComplianceSection>();

    public static void parseAll(HttpParser parser, ByteBuffer buffer) {
        if (parser.isState(HttpParser.State.END)) {
            parser.reset();
        }
        if (!parser.isState(HttpParser.State.START)) {
            throw new IllegalStateException("!START");
        }
        int remaining = buffer.remaining();
        while (!parser.isState(HttpParser.State.END) && remaining > 0) {
            int wasRemaining = remaining;
            parser.parseNext(buffer);
            remaining = buffer.remaining();
            if (remaining != wasRemaining) continue;
            break;
        }
    }

    @Test
    public void httpMethodTest() {
        Assertions.assertNull((Object)HttpMethod.lookAheadGet((ByteBuffer)BufferUtil.toBuffer((String)"Wibble ")));
        Assertions.assertNull((Object)HttpMethod.lookAheadGet((ByteBuffer)BufferUtil.toBuffer((String)"GET")));
        Assertions.assertNull((Object)HttpMethod.lookAheadGet((ByteBuffer)BufferUtil.toBuffer((String)"MO")));
        Assertions.assertEquals((Object)HttpMethod.GET, (Object)HttpMethod.lookAheadGet((ByteBuffer)BufferUtil.toBuffer((String)"GET ")));
        Assertions.assertEquals((Object)HttpMethod.MOVE, (Object)HttpMethod.lookAheadGet((ByteBuffer)BufferUtil.toBuffer((String)"MOVE ")));
        ByteBuffer b = BufferUtil.allocateDirect((int)128);
        BufferUtil.append((ByteBuffer)b, (ByteBuffer)BufferUtil.toBuffer((String)"GET"));
        Assertions.assertNull((Object)HttpMethod.lookAheadGet((ByteBuffer)b));
        BufferUtil.append((ByteBuffer)b, (ByteBuffer)BufferUtil.toBuffer((String)" "));
        Assertions.assertEquals((Object)HttpMethod.GET, (Object)HttpMethod.lookAheadGet((ByteBuffer)b));
    }

    @Test
    public void testLineParseMockIP() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /mock/127.0.0.1 HTTP/1.1\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/mock/127.0.0.1", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testLineParse0() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /foo HTTP/1.0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/foo", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testLineParse1RFC2616() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /999\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC2616_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/999", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/0.9", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.NO_HTTP_0_9}));
    }

    @Test
    public void testLineParse1() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /999\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"HTTP/0.9 not supported", (Object)this._bad);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.empty());
    }

    @Test
    public void testLineParse2RFC2616() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /222  \r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC2616_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/222", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/0.9", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.NO_HTTP_0_9}));
    }

    @Test
    public void testLineParse2() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /222  \r\n");
        this._versionOrReason = null;
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"HTTP/0.9 not supported", (Object)this._bad);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.empty());
    }

    @Test
    public void testLineParse3() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /fo\u0690 HTTP/1.0\r\n\r\n", (Charset)StandardCharsets.UTF_8);
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/fo\u0690", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testLineParse4() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /foo?param=\u0690 HTTP/1.0\r\n\r\n", (Charset)StandardCharsets.UTF_8);
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/foo?param=\u0690", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testLongURLParse() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/ HTTP/1.0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/123456789abcdef/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testAllowedLinePreamble() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"\r\n\r\nGET / HTTP/1.0\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testDisallowedLinePreamble() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"\r\n \r\nGET / HTTP/1.0\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"Illegal character SPACE=' '", (Object)this._bad);
    }

    @Test
    public void testConnect() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"CONNECT 192.168.1.2:80 HTTP/1.1\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"CONNECT", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"192.168.1.2:80", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
    }

    @Test
    public void testSimple() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Connection", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"close", (Object)this._val[1]);
        Assertions.assertEquals((int)1, (int)this._headers);
    }

    @Test
    public void testFoldedField2616() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName: value\r\n extra\r\nName2: \r\n\tvalue2\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC2616_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.nullValue());
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Name", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value extra", (Object)this._val[1]);
        Assertions.assertEquals((Object)"Name2", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"value2", (Object)this._val[2]);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.NO_FIELD_FOLDING, HttpComplianceSection.NO_FIELD_FOLDING}));
    }

    @Test
    public void testFoldedField7230() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName: value\r\n extra\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, 4096, HttpCompliance.RFC7230_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Header Folding"));
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.empty());
    }

    @Test
    public void testWhiteSpaceInName() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nN ame: value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, 4096, HttpCompliance.RFC7230_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Illegal character"));
    }

    @Test
    public void testWhiteSpaceAfterName() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName : value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, 4096, HttpCompliance.RFC7230_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Illegal character"));
    }

    @Test
    public void testWhiteSpaceBeforeRequest() {
        HttpCompliance[] compliances = new HttpCompliance[]{HttpCompliance.RFC7230, HttpCompliance.RFC2616};
        String[][] whitespaces = new String[][]{{" ", "Illegal character SPACE"}, {"\t", "Illegal character HTAB"}, {"\n", null}, {"\r", "Bad EOL"}, {"\r\n", null}, {"\r\n\r\n", null}, {"\r\n \r\n", "Illegal character SPACE"}, {"\r\n\t\r\n", "Illegal character HTAB"}, {"\r\t\n", "Bad EOL"}, {"\r\r\n", "Bad EOL"}, {"\t\r\t\r\n", "Illegal character HTAB"}, {" \t \r \t \n\n", "Illegal character SPACE"}, {" \r \t \r\n\r\n\r\n", "Illegal character SPACE"}};
        for (HttpCompliance compliance : compliances) {
            for (int j = 0; j < whitespaces.length; ++j) {
                String request = whitespaces[j][0] + "GET / HTTP/1.1\r\nHost: localhost\r\nName: value" + j + "\r\nConnection: close\r\n\r\n";
                ByteBuffer buffer = BufferUtil.toBuffer((String)request);
                Handler handler = new Handler();
                HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, 4096, compliance);
                this._bad = null;
                HttpParserTest.parseAll(parser, buffer);
                String test = "whitespace.[" + compliance + "].[" + j + "]";
                String expected = whitespaces[j][1];
                if (expected == null) {
                    MatcherAssert.assertThat((String)test, (Object)this._bad, (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
                    continue;
                }
                MatcherAssert.assertThat((String)test, (Object)this._bad, (Matcher)Matchers.containsString((String)expected));
            }
        }
    }

    @Test
    public void testNoValue() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName0: \r\nName1:\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Name0", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"", (Object)this._val[1]);
        Assertions.assertEquals((Object)"Name1", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"", (Object)this._val[2]);
        Assertions.assertEquals((int)2, (int)this._headers);
    }

    @Test
    public void testSpaceinNameCustom0() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName with space: value\r\nOther: value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.CUSTOM0);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Illegal character"));
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.NO_WS_AFTER_FIELD_NAME}));
    }

    @Test
    public void testNoColonCustom0() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName \r\nOther: value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.CUSTOM0);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Illegal character"));
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.NO_WS_AFTER_FIELD_NAME}));
    }

    @Test
    public void testTrailingSpacesInHeaderNameInCustom0Mode() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 204 No Content\r\nAccess-Control-Allow-Headers : Origin\r\nOther\t : value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler, -1, HttpCompliance.CUSTOM0);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"204", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"No Content", (Object)this._versionOrReason);
        Assertions.assertNull((Object)this._content);
        Assertions.assertEquals((int)1, (int)this._headers);
        System.out.println(Arrays.asList(this._hdr));
        System.out.println(Arrays.asList(this._val));
        Assertions.assertEquals((Object)"Access-Control-Allow-Headers", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"Origin", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Other", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value", (Object)this._val[1]);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.NO_WS_AFTER_FIELD_NAME, HttpComplianceSection.NO_WS_AFTER_FIELD_NAME}));
    }

    @Test
    public void testTrailingSpacesInHeaderNameNoCustom0() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 204 No Content\r\nAccess-Control-Allow-Headers : Origin\r\nOther: value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"204", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"No Content", (Object)this._versionOrReason);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Illegal character "));
    }

    @Test
    public void testNoColon7230() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nName\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC7230_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Illegal character"));
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.empty());
    }

    @Test
    public void testHeaderParseDirect() {
        ByteBuffer b0 = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nHeader1: value1\r\nHeader2:   value 2a  \r\nHeader3: 3\r\nHeader4:value4\r\nServer5: notServer\r\nHostHeader: notHost\r\nConnection: close\r\nAccept-Encoding: gzip, deflated\r\nAccept: unknown\r\n\r\n");
        ByteBuffer buffer = BufferUtil.allocateDirect((int)b0.capacity());
        int pos = BufferUtil.flipToFill((ByteBuffer)buffer);
        BufferUtil.put((ByteBuffer)b0, (ByteBuffer)buffer);
        BufferUtil.flipToFlush((ByteBuffer)buffer, (int)pos);
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[1]);
        Assertions.assertEquals((Object)"Header2", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"value 2a", (Object)this._val[2]);
        Assertions.assertEquals((Object)"Header3", (Object)this._hdr[3]);
        Assertions.assertEquals((Object)"3", (Object)this._val[3]);
        Assertions.assertEquals((Object)"Header4", (Object)this._hdr[4]);
        Assertions.assertEquals((Object)"value4", (Object)this._val[4]);
        Assertions.assertEquals((Object)"Server5", (Object)this._hdr[5]);
        Assertions.assertEquals((Object)"notServer", (Object)this._val[5]);
        Assertions.assertEquals((Object)"HostHeader", (Object)this._hdr[6]);
        Assertions.assertEquals((Object)"notHost", (Object)this._val[6]);
        Assertions.assertEquals((Object)"Connection", (Object)this._hdr[7]);
        Assertions.assertEquals((Object)"close", (Object)this._val[7]);
        Assertions.assertEquals((Object)"Accept-Encoding", (Object)this._hdr[8]);
        Assertions.assertEquals((Object)"gzip, deflated", (Object)this._val[8]);
        Assertions.assertEquals((Object)"Accept", (Object)this._hdr[9]);
        Assertions.assertEquals((Object)"unknown", (Object)this._val[9]);
        Assertions.assertEquals((int)9, (int)this._headers);
    }

    @Test
    public void testHeaderParseCRLF() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nHost: localhost\r\nHeader1: value1\r\nHeader2:   value 2a  \r\nHeader3: 3\r\nHeader4:value4\r\nServer5: notServer\r\nHostHeader: notHost\r\nConnection: close\r\nAccept-Encoding: gzip, deflated\r\nAccept: unknown\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[1]);
        Assertions.assertEquals((Object)"Header2", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"value 2a", (Object)this._val[2]);
        Assertions.assertEquals((Object)"Header3", (Object)this._hdr[3]);
        Assertions.assertEquals((Object)"3", (Object)this._val[3]);
        Assertions.assertEquals((Object)"Header4", (Object)this._hdr[4]);
        Assertions.assertEquals((Object)"value4", (Object)this._val[4]);
        Assertions.assertEquals((Object)"Server5", (Object)this._hdr[5]);
        Assertions.assertEquals((Object)"notServer", (Object)this._val[5]);
        Assertions.assertEquals((Object)"HostHeader", (Object)this._hdr[6]);
        Assertions.assertEquals((Object)"notHost", (Object)this._val[6]);
        Assertions.assertEquals((Object)"Connection", (Object)this._hdr[7]);
        Assertions.assertEquals((Object)"close", (Object)this._val[7]);
        Assertions.assertEquals((Object)"Accept-Encoding", (Object)this._hdr[8]);
        Assertions.assertEquals((Object)"gzip, deflated", (Object)this._val[8]);
        Assertions.assertEquals((Object)"Accept", (Object)this._hdr[9]);
        Assertions.assertEquals((Object)"unknown", (Object)this._val[9]);
        Assertions.assertEquals((int)9, (int)this._headers);
    }

    @Test
    public void testHeaderParseLF() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\nHost: localhost\nHeader1: value1\nHeader2:   value 2a value 2b  \nHeader3: 3\nHeader4:value4\nServer5: notServer\nHostHeader: notHost\nConnection: close\nAccept-Encoding: gzip, deflated\nAccept: unknown\n\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[1]);
        Assertions.assertEquals((Object)"Header2", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"value 2a value 2b", (Object)this._val[2]);
        Assertions.assertEquals((Object)"Header3", (Object)this._hdr[3]);
        Assertions.assertEquals((Object)"3", (Object)this._val[3]);
        Assertions.assertEquals((Object)"Header4", (Object)this._hdr[4]);
        Assertions.assertEquals((Object)"value4", (Object)this._val[4]);
        Assertions.assertEquals((Object)"Server5", (Object)this._hdr[5]);
        Assertions.assertEquals((Object)"notServer", (Object)this._val[5]);
        Assertions.assertEquals((Object)"HostHeader", (Object)this._hdr[6]);
        Assertions.assertEquals((Object)"notHost", (Object)this._val[6]);
        Assertions.assertEquals((Object)"Connection", (Object)this._hdr[7]);
        Assertions.assertEquals((Object)"close", (Object)this._val[7]);
        Assertions.assertEquals((Object)"Accept-Encoding", (Object)this._hdr[8]);
        Assertions.assertEquals((Object)"gzip, deflated", (Object)this._val[8]);
        Assertions.assertEquals((Object)"Accept", (Object)this._hdr[9]);
        Assertions.assertEquals((Object)"unknown", (Object)this._val[9]);
        Assertions.assertEquals((int)9, (int)this._headers);
    }

    @Test
    public void testQuoted() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\nName0: \"value0\"\t\nName1: \"value\t1\"\nName2: \"value\t2A\",\"value,2B\"\t\n\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Name0", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"\"value0\"", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Name1", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"\"value\t1\"", (Object)this._val[1]);
        Assertions.assertEquals((Object)"Name2", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"\"value\t2A\",\"value,2B\"", (Object)this._val[2]);
        Assertions.assertEquals((int)2, (int)this._headers);
    }

    @Test
    public void testEncodedHeader() {
        ByteBuffer buffer = BufferUtil.allocate((int)4096);
        BufferUtil.flipToFill((ByteBuffer)buffer);
        BufferUtil.put((ByteBuffer)BufferUtil.toBuffer((String)"GET "), (ByteBuffer)buffer);
        buffer.put("/foo/\u0690/".getBytes(StandardCharsets.UTF_8));
        BufferUtil.put((ByteBuffer)BufferUtil.toBuffer((String)" HTTP/1.0\r\n"), (ByteBuffer)buffer);
        BufferUtil.put((ByteBuffer)BufferUtil.toBuffer((String)"Header1: "), (ByteBuffer)buffer);
        buffer.put("\u00e6 \u00e6".getBytes(StandardCharsets.ISO_8859_1));
        BufferUtil.put((ByteBuffer)BufferUtil.toBuffer((String)"  \r\nHeader2: "), (ByteBuffer)buffer);
        buffer.put((byte)-1);
        BufferUtil.put((ByteBuffer)BufferUtil.toBuffer((String)"\r\n\r\n"), (ByteBuffer)buffer);
        BufferUtil.flipToFlush((ByteBuffer)buffer, (int)0);
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/foo/\u0690/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"\u00e6 \u00e6", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Header2", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"\u00ff", (Object)this._val[1]);
        Assertions.assertEquals((int)1, (int)this._headers);
        Assertions.assertNull((Object)this._bad);
    }

    @Test
    public void testResponseBufferUpgradeFrom() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 101 Upgrade\r\nConnection: upgrade\r\nContent-Length: 0\r\nSec-WebSocket-Accept: 4GnyoUP4Sc1JD+2pCbNYAhFYVVA\r\n\r\nFOOGRADE");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        while (!parser.isState(HttpParser.State.END)) {
            parser.parseNext(buffer);
        }
        MatcherAssert.assertThat((Object)BufferUtil.toUTF8String((ByteBuffer)buffer), (Matcher)Matchers.is((Object)"FOOGRADE"));
    }

    @Test
    public void testBadMethodEncoding() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"G\u00e6T / HTTP/1.0\r\nHeader0: value0\r\n\n\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.notNullValue());
    }

    @Test
    public void testBadVersionEncoding() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / H\u00e6P/1.0\r\nHeader0: value0\r\n\n\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.notNullValue());
    }

    @Test
    public void testBadHeaderEncoding() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nH\u00e6der0: value0\r\n\n\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.notNullValue());
    }

    @Test
    public void testBadHeaderNames() {
        String[] bad;
        for (String s : bad = new String[]{"Foo\\Bar: value\r\n", "Foo@Bar: value\r\n", "Foo,Bar: value\r\n", "Foo}Bar: value\r\n", "Foo{Bar: value\r\n", "Foo=Bar: value\r\n", "Foo>Bar: value\r\n", "Foo<Bar: value\r\n", "Foo)Bar: value\r\n", "Foo(Bar: value\r\n", "Foo?Bar: value\r\n", "Foo\"Bar: value\r\n", "Foo/Bar: value\r\n", "Foo]Bar: value\r\n", "Foo[Bar: value\r\n"}) {
            ByteBuffer buffer = BufferUtil.toBuffer((String)("GET / HTTP/1.0\r\n" + s + "\r\n"));
            Handler handler = new Handler();
            HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
            HttpParserTest.parseAll(parser, buffer);
            MatcherAssert.assertThat((String)s, (Object)this._bad, (Matcher)Matchers.notNullValue());
        }
    }

    @Test
    public void testHeaderTab() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: localhost\r\nHeader: value\talternate\r\n\n\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Header", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value\talternate", (Object)this._val[1]);
    }

    @Test
    public void testCaseSensitiveMethod() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"gEt / http/1.0\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, -1, HttpCompliance.RFC7230_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.METHOD_CASE_SENSITIVE}));
    }

    @Test
    public void testCaseSensitiveMethodLegacy() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"gEt / http/1.0\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, -1, HttpCompliance.LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"gEt", (Object)this._methodOrVersion);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.empty());
    }

    @Test
    public void testCaseInsensitiveHeader() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / http/1.0\r\nHOST: localhost\r\ncOnNeCtIoN: ClOsE\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, -1, HttpCompliance.RFC7230_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Connection", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"close", (Object)this._val[1]);
        Assertions.assertEquals((int)1, (int)this._headers);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.empty());
    }

    @Test
    public void testCaseInSensitiveHeaderLegacy() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / http/1.0\r\nHOST: localhost\r\ncOnNeCtIoN: ClOsE\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, -1, HttpCompliance.LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"HOST", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"cOnNeCtIoN", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"ClOsE", (Object)this._val[1]);
        Assertions.assertEquals((int)1, (int)this._headers);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.FIELD_NAME_CASE_INSENSITIVE, HttpComplianceSection.FIELD_NAME_CASE_INSENSITIVE, HttpComplianceSection.CASE_INSENSITIVE_FIELD_VALUE_CACHE}));
    }

    @Test
    public void testSplitHeaderParse() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"XXXXSPLIT / HTTP/1.0\r\nHost: localhost\r\nHeader1: value1\r\nHeader2:   value 2a  \r\nHeader3: 3\r\nHeader4:value4\r\nServer5: notServer\r\n\r\nZZZZ");
        buffer.position(2);
        buffer.limit(buffer.capacity() - 2);
        buffer = buffer.slice();
        for (int i = 0; i < buffer.capacity() - 4; ++i) {
            Handler handler = new Handler();
            HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
            buffer.position(2);
            buffer.limit(2 + i);
            if (!parser.parseNext(buffer)) {
                Assertions.assertEquals((int)0, (int)buffer.remaining());
                buffer.limit(buffer.capacity() - 2);
                parser.parseNext(buffer);
            }
            Assertions.assertEquals((Object)"SPLIT", (Object)this._methodOrVersion);
            Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
            Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
            Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
            Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
            Assertions.assertEquals((Object)"Header1", (Object)this._hdr[1]);
            Assertions.assertEquals((Object)"value1", (Object)this._val[1]);
            Assertions.assertEquals((Object)"Header2", (Object)this._hdr[2]);
            Assertions.assertEquals((Object)"value 2a", (Object)this._val[2]);
            Assertions.assertEquals((Object)"Header3", (Object)this._hdr[3]);
            Assertions.assertEquals((Object)"3", (Object)this._val[3]);
            Assertions.assertEquals((Object)"Header4", (Object)this._hdr[4]);
            Assertions.assertEquals((Object)"value4", (Object)this._val[4]);
            Assertions.assertEquals((Object)"Server5", (Object)this._hdr[5]);
            Assertions.assertEquals((Object)"notServer", (Object)this._val[5]);
            Assertions.assertEquals((int)5, (int)this._headers);
        }
    }

    @Test
    public void testChunkParse() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nHeader1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)1, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[0]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testBadChunkParse() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nHeader1: value1\r\nTransfer-Encoding: chunked, identity\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Bad Transfer-Encoding"));
    }

    @Test
    public void testChunkParseTrailer() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nHeader1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\nTrailer: value\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)1, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[0]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        Assertions.assertEquals((int)1, (int)this._trailers.size());
        HttpField trailer1 = this._trailers.get(0);
        Assertions.assertEquals((Object)"Trailer", (Object)trailer1.getName());
        Assertions.assertEquals((Object)"value", (Object)trailer1.getValue());
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testChunkParseTrailers() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\nTrailer: value\r\nFoo: bar\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)0, (int)this._headers);
        Assertions.assertEquals((Object)"Transfer-Encoding", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"chunked", (Object)this._val[0]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        Assertions.assertEquals((int)2, (int)this._trailers.size());
        HttpField trailer1 = this._trailers.get(0);
        Assertions.assertEquals((Object)"Trailer", (Object)trailer1.getName());
        Assertions.assertEquals((Object)"value", (Object)trailer1.getValue());
        HttpField trailer2 = this._trailers.get(1);
        Assertions.assertEquals((Object)"Foo", (Object)trailer2.getName());
        Assertions.assertEquals((Object)"bar", (Object)trailer2.getValue());
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testChunkParseBadTrailer() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nHeader1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\nTrailer: value");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)1, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[0]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._early);
    }

    @Test
    public void testChunkParseNoTrailer() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nHeader1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)1, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[0]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testStartEOF() {
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertTrue((boolean)this._early);
        Assertions.assertNull((Object)this._bad);
    }

    @Test
    public void testEarlyEOF() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /uri HTTP/1.0\r\nContent-Length: 20\r\n\r\n0123456789");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.atEOF();
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/uri", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"0123456789", (Object)this._content);
        Assertions.assertTrue((boolean)this._early);
    }

    @Test
    public void testChunkEarlyEOF() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /chunk HTTP/1.0\r\nHeader1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.atEOF();
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)1, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[0]);
        Assertions.assertEquals((Object)"0123456789", (Object)this._content);
        Assertions.assertTrue((boolean)this._early);
    }

    @Test
    public void testMultiParse() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /mp HTTP/1.0\r\nConnection: Keep-Alive\r\nHeader1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\n\r\nPOST /foo HTTP/1.0\r\nConnection: Keep-Alive\r\nHeader2: value2\r\nContent-Length: 0\r\n\r\nPUT /doodle HTTP/1.0\r\nConnection: close\r\nHeader3: value3\r\nContent-Length: 10\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/mp", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[1]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        parser.reset();
        this.init();
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/foo", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Header2", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value2", (Object)this._val[1]);
        Assertions.assertNull((Object)this._content);
        parser.reset();
        this.init();
        parser.parseNext(buffer);
        parser.atEOF();
        Assertions.assertEquals((Object)"PUT", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/doodle", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Header3", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value3", (Object)this._val[1]);
        Assertions.assertEquals((Object)"0123456789", (Object)this._content);
    }

    @Test
    public void testMultiParseEarlyEOF() {
        ByteBuffer buffer0 = BufferUtil.toBuffer((String)"GET /mp HTTP/1.0\r\nConnection: Keep-Alive\r\n");
        ByteBuffer buffer1 = BufferUtil.toBuffer((String)"Header1: value1\r\nTransfer-Encoding: chunked\r\n\r\na;\r\n0123456789\r\n1a\r\nABCDEFGHIJKLMNOPQRSTUVWXYZ\r\n0\r\n\r\nPOST /foo HTTP/1.0\r\nConnection: Keep-Alive\r\nHeader2: value2\r\nContent-Length: 0\r\n\r\nPUT /doodle HTTP/1.0\r\nConnection: close\r\nHeader3: value3\r\nContent-Length: 10\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer0);
        parser.atEOF();
        parser.parseNext(buffer1);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/mp", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Header1", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value1", (Object)this._val[1]);
        Assertions.assertEquals((Object)"0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ", (Object)this._content);
        parser.reset();
        this.init();
        parser.parseNext(buffer1);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/foo", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Header2", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value2", (Object)this._val[1]);
        Assertions.assertNull((Object)this._content);
        parser.reset();
        this.init();
        parser.parseNext(buffer1);
        Assertions.assertEquals((Object)"PUT", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/doodle", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)2, (int)this._headers);
        Assertions.assertEquals((Object)"Header3", (Object)this._hdr[1]);
        Assertions.assertEquals((Object)"value3", (Object)this._val[1]);
        Assertions.assertEquals((Object)"0123456789", (Object)this._content);
    }

    @Test
    public void testResponseParse0() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 Correct\r\nContent-Length: 10\r\nContent-Type: text/plain\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"Correct", (Object)this._versionOrReason);
        Assertions.assertEquals((int)10, (int)this._content.length());
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponseParse1() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 Not-Modified\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"304", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"Not-Modified", (Object)this._versionOrReason);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponseParse2() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 204 No-Content\r\nHeader: value\r\n\r\nHTTP/1.1 200 Correct\r\nContent-Length: 10\r\nContent-Type: text/plain\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"204", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"No-Content", (Object)this._versionOrReason);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        parser.parseNext(buffer);
        parser.atEOF();
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"Correct", (Object)this._versionOrReason);
        Assertions.assertEquals((int)this._content.length(), (int)10);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponseParse3() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200\r\nContent-Length: 10\r\nContent-Type: text/plain\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertNull((Object)this._versionOrReason);
        Assertions.assertEquals((int)this._content.length(), (int)10);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponseParse4() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 \r\nContent-Length: 10\r\nContent-Type: text/plain\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertNull((Object)this._versionOrReason);
        Assertions.assertEquals((int)this._content.length(), (int)10);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponseEOFContent() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 \r\nContent-Type: text/plain\r\n\r\n0123456789\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.atEOF();
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertNull((Object)this._versionOrReason);
        Assertions.assertEquals((int)12, (int)this._content.length());
        Assertions.assertEquals((Object)"0123456789\r\n", (Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponse304WithContentLength() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 found\r\nContent-Length: 10\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"304", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"found", (Object)this._versionOrReason);
        Assertions.assertNull((Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponse101WithTransferEncoding() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 101 switching protocols\r\nTransfer-Encoding: chunked\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"101", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"switching protocols", (Object)this._versionOrReason);
        Assertions.assertNull((Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testResponseReasonIso88591() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 302 d\u00e9plac\u00e9 temporairement\r\nContent-Length: 0\r\n\r\n", (Charset)StandardCharsets.ISO_8859_1);
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"302", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"d\u00e9plac\u00e9 temporairement", (Object)this._versionOrReason);
    }

    @Test
    public void testSeekEOF() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nContent-Length: 0\r\nConnection: close\r\n\r\n\r\nHTTP/1.1 400 OK\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"OK", (Object)this._versionOrReason);
        Assertions.assertNull((Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.close();
        parser.reset();
        parser.parseNext(buffer);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testNoURI() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET\r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"No URI", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testNoURI2() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET \r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"No URI", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testUnknownReponseVersion() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HPPT/7.7 200 OK\r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Unknown Version", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testNoStatus() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1\r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"No Status", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testNoStatus2() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 \r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"No Status", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testBadRequestVersion() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HPPT/7.7\r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Unknown Version", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
        buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.01\r\nContent-Length: 0\r\nConnection: close\r\n\r\n");
        handler = new Handler();
        parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Unknown Version", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testBadCR() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nContent-Length: 0\rConnection: close\r\r");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"Bad EOL", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
        buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\rContent-Length: 0\rConnection: close\r\r");
        handler = new Handler();
        parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"Bad EOL", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testBadContentLength0() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nContent-Length: abc\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Invalid Content-Length Value", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testBadContentLength1() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nContent-Length: 9999999999999999999999999999999999999999999999\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Invalid Content-Length Value", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testBadContentLength2() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.0\r\nContent-Length: 1.5\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Invalid Content-Length Value", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testMultipleContentLengthWithLargerThenCorrectValue() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST / HTTP/1.1\r\nContent-Length: 2\r\nContent-Length: 1\r\nConnection: close\r\n\r\nX");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Multiple Content-Lengths", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testMultipleContentLengthWithCorrectThenLargerValue() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST / HTTP/1.1\r\nContent-Length: 1\r\nContent-Length: 2\r\nConnection: close\r\n\r\nX");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"Multiple Content-Lengths", (Object)this._bad);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)HttpParser.State.CLOSE, (Object)parser.getState());
        parser.atEOF();
        parser.parseNext(BufferUtil.EMPTY_BUFFER);
        Assertions.assertEquals((Object)HttpParser.State.CLOSED, (Object)parser.getState());
    }

    @Test
    public void testTransferEncodingChunkedThenContentLength() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /chunk HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Length: 1\r\n\r\n1\r\nX\r\n0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC2616_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"X", (Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.TRANSFER_ENCODING_WITH_CONTENT_LENGTH}));
    }

    @Test
    public void testContentLengthThenTransferEncodingChunked() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"POST /chunk HTTP/1.1\r\nHost: localhost\r\nContent-Length: 1\r\nTransfer-Encoding: chunked\r\n\r\n1\r\nX\r\n0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC2616_LEGACY);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"POST", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/chunk", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"X", (Object)this._content);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        MatcherAssert.assertThat(this._complianceViolation, (Matcher)Matchers.contains((Object[])new HttpComplianceSection[]{HttpComplianceSection.TRANSFER_ENCODING_WITH_CONTENT_LENGTH}));
    }

    @Test
    public void testHost() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: host\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"host", (Object)this._host);
        Assertions.assertEquals((int)0, (int)this._port);
    }

    @Test
    public void testUriHost11() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET http://host/ HTTP/1.1\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"No Host", (Object)this._bad);
        Assertions.assertEquals((Object)"http://host/", (Object)this._uriOrStatus);
        Assertions.assertEquals((int)0, (int)this._port);
    }

    @Test
    public void testUriHost10() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET http://host/ HTTP/1.0\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._bad);
        Assertions.assertEquals((Object)"http://host/", (Object)this._uriOrStatus);
        Assertions.assertEquals((int)0, (int)this._port);
    }

    @Test
    public void testNoHost() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"No Host", (Object)this._bad);
    }

    @Test
    public void testIPHost() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: 192.168.0.1\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"192.168.0.1", (Object)this._host);
        Assertions.assertEquals((int)0, (int)this._port);
    }

    @Test
    public void testIPv6Host() {
        Assumptions.assumeTrue((boolean)Net.isIpv6InterfaceAvailable());
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: [::1]\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"[::1]", (Object)this._host);
        Assertions.assertEquals((int)0, (int)this._port);
    }

    @Test
    public void testBadIPv6Host() {
        try (StacklessLogging s = new StacklessLogging(new Class[]{HttpParser.class});){
            ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: [::1\r\nConnection: close\r\n\r\n");
            Handler handler = new Handler();
            HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
            parser.parseNext(buffer);
            MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Bad"));
        }
    }

    @Test
    public void testHostPort() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: myhost:8888\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"myhost", (Object)this._host);
        Assertions.assertEquals((int)8888, (int)this._port);
    }

    @Test
    public void testHostBadPort() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: myhost:testBadPort\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        MatcherAssert.assertThat((Object)this._bad, (Matcher)Matchers.containsString((String)"Bad Host"));
    }

    @Test
    public void testIPHostPort() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: 192.168.0.1:8888\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"192.168.0.1", (Object)this._host);
        Assertions.assertEquals((int)8888, (int)this._port);
    }

    @Test
    public void testIPv6HostPort() {
        Assumptions.assumeTrue((boolean)Net.isIpv6InterfaceAvailable());
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: [::1]:8888\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"[::1]", (Object)this._host);
        Assertions.assertEquals((int)8888, (int)this._port);
    }

    @Test
    public void testEmptyHostPort() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost:\r\nConnection: close\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertNull((Object)this._host);
        Assertions.assertNull((Object)this._bad);
    }

    @Test
    public void testCachedField() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: www.smh.com.au\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertEquals((Object)"www.smh.com.au", (Object)((HttpField)parser.getFieldCache().get("Host: www.smh.com.au")).getValue());
        HttpField field = this._fields.get(0);
        buffer.position(0);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertSame((Object)field, (Object)this._fields.get(0));
    }

    @Test
    public void testParseRequest() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET / HTTP/1.1\r\nHost: localhost\r\nHeader1: value1\r\nConnection: close\r\nAccept-Encoding: gzip, deflated\r\nAccept: unknown\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        parser.parseNext(buffer);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/1.1", (Object)this._versionOrReason);
        Assertions.assertEquals((Object)"Host", (Object)this._hdr[0]);
        Assertions.assertEquals((Object)"localhost", (Object)this._val[0]);
        Assertions.assertEquals((Object)"Connection", (Object)this._hdr[2]);
        Assertions.assertEquals((Object)"close", (Object)this._val[2]);
        Assertions.assertEquals((Object)"Accept-Encoding", (Object)this._hdr[3]);
        Assertions.assertEquals((Object)"gzip, deflated", (Object)this._val[3]);
        Assertions.assertEquals((Object)"Accept", (Object)this._hdr[4]);
        Assertions.assertEquals((Object)"unknown", (Object)this._val[4]);
    }

    @Test
    public void testHTTP2Preface() {
        ByteBuffer buffer = BufferUtil.toBuffer((String)"PRI * HTTP/2.0\r\n\r\nSM\r\n\r\n");
        Handler handler = new Handler();
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler);
        HttpParserTest.parseAll(parser, buffer);
        Assertions.assertTrue((boolean)this._headerCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        Assertions.assertEquals((Object)"PRI", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"*", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/2.0", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
        Assertions.assertNull((Object)this._bad);
    }

    @Test
    public void testForHTTP09HeaderCompleteTrueDoesNotEmitContentComplete() {
        Handler handler = new Handler(){

            @Override
            public boolean headerComplete() {
                super.headerComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.RequestHandler)handler, HttpCompliance.RFC2616_LEGACY);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"GET /path\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        Assertions.assertEquals((Object)"GET", (Object)this._methodOrVersion);
        Assertions.assertEquals((Object)"/path", (Object)this._uriOrStatus);
        Assertions.assertEquals((Object)"HTTP/0.9", (Object)this._versionOrReason);
        Assertions.assertEquals((int)-1, (int)this._headers);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testForContentLengthZeroHeaderCompleteTrueDoesNotEmitContentComplete() {
        Handler handler = new Handler(){

            @Override
            public boolean headerComplete() {
                super.headerComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testForEmptyChunkedContentHeaderCompleteTrueDoesNotEmitContentComplete() {
        Handler handler = new Handler(){

            @Override
            public boolean headerComplete() {
                super.headerComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nTransfer-Encoding: chunked\r\n\r\n0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testForContentLengthZeroContentCompleteTrueDoesNotEmitMessageComplete() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testForEmptyChunkedContentContentCompleteTrueDoesNotEmitMessageComplete() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nTransfer-Encoding: chunked\r\n\r\n0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testHeaderAfterContentLengthZeroContentCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        String header = "Header: Foobar\r\n";
        ByteBuffer buffer = BufferUtil.toBuffer((String)("HTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n" + header));
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)header, (Object)BufferUtil.toString((ByteBuffer)buffer));
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)header, (Object)BufferUtil.toString((ByteBuffer)buffer));
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testSmallContentLengthContentCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        String header = "Header: Foobar\r\n";
        ByteBuffer buffer = BufferUtil.toBuffer((String)("HTTP/1.1 200 OK\r\nContent-Length: 1\r\n\r\n0" + header));
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)header, (Object)BufferUtil.toString((ByteBuffer)buffer));
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)header, (Object)BufferUtil.toString((ByteBuffer)buffer));
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testHeaderAfterSmallContentLengthContentCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nContent-Length: 1\r\n\r\n0");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testEOFContentContentCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\n\r\n0");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"0", (Object)this._content);
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        parser.atEOF();
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testHEADRequestHeaderCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean headerComplete() {
                super.headerComplete();
                return true;
            }

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        parser.setHeadResponse(true);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testNoContentHeaderCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean headerComplete() {
                super.headerComplete();
                return true;
            }

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 Not Modified\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testCRLFAfterResponseHeaderCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean headerComplete() {
                super.headerComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 Not Modified\r\n\r\n\r\n\r\nHTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n\r\n\r\nHTTP/1.1 303 See Other\r\nContent-Length: 0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"304", (Object)this._uriOrStatus);
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"303", (Object)this._uriOrStatus);
        Assertions.assertFalse((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testCRLFAfterResponseContentCompleteTrue() {
        Handler handler = new Handler(){

            @Override
            public boolean contentComplete() {
                super.contentComplete();
                return true;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 Not Modified\r\n\r\n\r\n\r\nHTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n\r\n\r\nHTTP/1.1 303 See Other\r\nContent-Length: 0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"304", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"303", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertFalse((boolean)this._messageCompleted);
        handle = parser.parseNext(buffer);
        Assertions.assertTrue((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testCRLFAfterResponseMessageCompleteFalse() {
        Handler handler = new Handler(){

            @Override
            public boolean messageComplete() {
                super.messageComplete();
                return false;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 Not Modified\r\n\r\n\r\n\r\nHTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n\r\n\r\nHTTP/1.1 303 See Other\r\nContent-Length: 0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"304", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"303", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
    }

    @Test
    public void testSPAfterResponseMessageCompleteFalse() {
        Handler handler = new Handler(){

            @Override
            public boolean messageComplete() {
                super.messageComplete();
                return false;
            }
        };
        HttpParser parser = new HttpParser((HttpParser.ResponseHandler)handler);
        ByteBuffer buffer = BufferUtil.toBuffer((String)"HTTP/1.1 304 Not Modified\r\n\r\n HTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n");
        boolean handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"304", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertNotNull((Object)this._bad);
        buffer = BufferUtil.toBuffer((String)"HTTP/1.1 200 OK\r\nContent-Length: 0\r\n\r\n HTTP/1.1 303 See Other\r\nContent-Length: 0\r\n\r\n");
        parser = new HttpParser((HttpParser.ResponseHandler)handler);
        handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertTrue((boolean)buffer.hasRemaining());
        Assertions.assertEquals((Object)"200", (Object)this._uriOrStatus);
        Assertions.assertTrue((boolean)this._contentCompleted);
        Assertions.assertTrue((boolean)this._messageCompleted);
        parser.reset();
        this.init();
        handle = parser.parseNext(buffer);
        Assertions.assertFalse((boolean)handle);
        Assertions.assertFalse((boolean)buffer.hasRemaining());
        Assertions.assertNotNull((Object)this._bad);
    }

    @BeforeEach
    public void init() {
        this._bad = null;
        this._content = null;
        this._methodOrVersion = null;
        this._uriOrStatus = null;
        this._versionOrReason = null;
        this._hdr = null;
        this._val = null;
        this._headers = 0;
        this._headerCompleted = false;
        this._contentCompleted = false;
        this._messageCompleted = false;
        this._complianceViolation.clear();
    }

    static /* synthetic */ String[] access$502(HttpParserTest x0, String[] x1) {
        x0._hdr = x1;
        return x1;
    }

    static /* synthetic */ String[] access$602(HttpParserTest x0, String[] x1) {
        x0._val = x1;
        return x1;
    }

    static {
        HttpCompliance.CUSTOM0.sections().remove(HttpComplianceSection.NO_WS_AFTER_FIELD_NAME);
    }

    private class Handler
    implements HttpParser.RequestHandler,
    HttpParser.ResponseHandler,
    HttpParser.ComplianceHandler {
        private Handler() {
        }

        public boolean content(ByteBuffer ref) {
            if (HttpParserTest.this._content == null) {
                HttpParserTest.this._content = "";
            }
            String c = BufferUtil.toString((ByteBuffer)ref, (Charset)StandardCharsets.UTF_8);
            HttpParserTest.this._content = HttpParserTest.this._content + c;
            ref.position(ref.limit());
            return false;
        }

        public boolean startRequest(String method, String uri, HttpVersion version) {
            HttpParserTest.this._fields.clear();
            HttpParserTest.this._trailers.clear();
            HttpParserTest.this._headers = -1;
            HttpParserTest.access$502(HttpParserTest.this, new String[10]);
            HttpParserTest.access$602(HttpParserTest.this, new String[10]);
            HttpParserTest.this._methodOrVersion = method;
            HttpParserTest.this._uriOrStatus = uri;
            HttpParserTest.this._versionOrReason = version == null ? null : version.asString();
            HttpParserTest.this._messageCompleted = false;
            HttpParserTest.this._headerCompleted = false;
            HttpParserTest.this._early = false;
            return false;
        }

        public void parsedHeader(HttpField field) {
            HttpParserTest.this._fields.add(field);
            ((HttpParserTest)HttpParserTest.this)._hdr[++((HttpParserTest)HttpParserTest.this)._headers] = field.getName();
            ((HttpParserTest)HttpParserTest.this)._val[((HttpParserTest)HttpParserTest.this)._headers] = field.getValue();
            if (field instanceof HostPortHttpField) {
                HostPortHttpField hpfield = (HostPortHttpField)field;
                HttpParserTest.this._host = hpfield.getHost();
                HttpParserTest.this._port = hpfield.getPort();
            }
        }

        public boolean headerComplete() {
            HttpParserTest.this._content = null;
            HttpParserTest.this._headerCompleted = true;
            return false;
        }

        public void parsedTrailer(HttpField field) {
            HttpParserTest.this._trailers.add(field);
        }

        public boolean contentComplete() {
            HttpParserTest.this._contentCompleted = true;
            return false;
        }

        public boolean messageComplete() {
            HttpParserTest.this._messageCompleted = true;
            return true;
        }

        public void badMessage(BadMessageException failure) {
            String reason = failure.getReason();
            HttpParserTest.this._bad = reason == null ? String.valueOf(failure.getCode()) : reason;
        }

        public boolean startResponse(HttpVersion version, int status, String reason) {
            HttpParserTest.this._fields.clear();
            HttpParserTest.this._trailers.clear();
            HttpParserTest.this._methodOrVersion = version.asString();
            HttpParserTest.this._uriOrStatus = Integer.toString(status);
            HttpParserTest.this._versionOrReason = reason;
            HttpParserTest.this._headers = -1;
            HttpParserTest.access$502(HttpParserTest.this, new String[10]);
            HttpParserTest.access$602(HttpParserTest.this, new String[10]);
            HttpParserTest.this._messageCompleted = false;
            HttpParserTest.this._headerCompleted = false;
            return false;
        }

        public void earlyEOF() {
            HttpParserTest.this._early = true;
        }

        public int getHeaderCacheSize() {
            return 1024;
        }

        public void onComplianceViolation(HttpCompliance compliance, HttpComplianceSection violation, String reason) {
            HttpParserTest.this._complianceViolation.add(violation);
        }
    }
}

