/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.http;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.stream.Stream;
import org.eclipse.jetty.http.HttpMethod;
import org.eclipse.jetty.http.HttpURI;
import org.eclipse.jetty.util.MultiMap;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

public class HttpURITest {
    @Test
    public void testInvalidAddress() throws Exception {
        this.assertInvalidURI("http://[ffff::1:8080/", "Invalid URL; no closing ']' -- should throw exception");
        this.assertInvalidURI("**", "only '*', not '**'");
        this.assertInvalidURI("*/", "only '*', not '*/'");
    }

    private void assertInvalidURI(String invalidURI, String message) {
        HttpURI uri = new HttpURI();
        try {
            uri.parse(invalidURI);
            Assertions.fail((String)message);
        }
        catch (IllegalArgumentException e) {
            Assertions.assertTrue((boolean)true);
        }
    }

    @Test
    public void testParse() {
        HttpURI uri = new HttpURI();
        uri.parse("*");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.nullValue());
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"*"));
        uri.parse("/foo/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.nullValue());
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/foo/bar"));
        uri.parse("//foo/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.is((Object)"foo"));
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/bar"));
        uri.parse("http://foo/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.is((Object)"foo"));
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/bar"));
        uri.parse("http://fo\u0000/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.is((Object)"fo\u0000"));
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/bar"));
    }

    @Test
    public void testParseRequestTarget() {
        HttpURI uri = new HttpURI();
        uri.parseRequestTarget("GET", "*");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.nullValue());
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"*"));
        uri.parseRequestTarget("GET", "/foo/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.nullValue());
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/foo/bar"));
        uri.parseRequestTarget("GET", "//foo/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.nullValue());
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"//foo/bar"));
        uri.parseRequestTarget("GET", "http://foo/bar");
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.is((Object)"foo"));
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/bar"));
    }

    @Test
    public void testExtB() throws Exception {
        for (String value : new String[]{"a", "abcdABCD", "\u00c0", "\u697c", "\ud869\uded5", "\ud840\udc08"}) {
            HttpURI uri = new HttpURI("/path?value=" + URLEncoder.encode(value, "UTF-8"));
            MultiMap parameters = new MultiMap();
            uri.decodeQueryTo(parameters, StandardCharsets.UTF_8);
            Assertions.assertEquals((Object)value, (Object)parameters.getString("value"));
        }
    }

    @Test
    public void testAt() throws Exception {
        HttpURI uri = new HttpURI("/@foo/bar");
        Assertions.assertEquals((Object)"/@foo/bar", (Object)uri.getPath());
    }

    @Test
    public void testParams() throws Exception {
        HttpURI uri = new HttpURI("/foo/bar");
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals(null, (Object)uri.getParam());
        uri = new HttpURI("/foo/bar;jsessionid=12345");
        Assertions.assertEquals((Object)"/foo/bar;jsessionid=12345", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"jsessionid=12345", (Object)uri.getParam());
        uri = new HttpURI("/foo;abc=123/bar;jsessionid=12345");
        Assertions.assertEquals((Object)"/foo;abc=123/bar;jsessionid=12345", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"jsessionid=12345", (Object)uri.getParam());
        uri = new HttpURI("/foo;abc=123/bar;jsessionid=12345?name=value");
        Assertions.assertEquals((Object)"/foo;abc=123/bar;jsessionid=12345", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"jsessionid=12345", (Object)uri.getParam());
        uri = new HttpURI("/foo;abc=123/bar;jsessionid=12345#target");
        Assertions.assertEquals((Object)"/foo;abc=123/bar;jsessionid=12345", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"jsessionid=12345", (Object)uri.getParam());
    }

    @Test
    public void testMutableURI() {
        HttpURI uri = new HttpURI("/foo/bar");
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.toString());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        uri.setScheme("http");
        Assertions.assertEquals((Object)"http:/foo/bar", (Object)uri.toString());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        uri.setAuthority("host", 0);
        Assertions.assertEquals((Object)"http://host/foo/bar", (Object)uri.toString());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        uri.setAuthority("host", 8888);
        Assertions.assertEquals((Object)"http://host:8888/foo/bar", (Object)uri.toString());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/foo/bar", (Object)uri.getDecodedPath());
        uri.setPathQuery("/f%30%30;p0/bar;p1;p2");
        Assertions.assertEquals((Object)"http://host:8888/f%30%30;p0/bar;p1;p2", (Object)uri.toString());
        Assertions.assertEquals((Object)"/f%30%30;p0/bar;p1;p2", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/f00/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"p2", (Object)uri.getParam());
        Assertions.assertEquals(null, (Object)uri.getQuery());
        uri.setPathQuery("/f%30%30;p0/bar;p1;p2?name=value");
        Assertions.assertEquals((Object)"http://host:8888/f%30%30;p0/bar;p1;p2?name=value", (Object)uri.toString());
        Assertions.assertEquals((Object)"/f%30%30;p0/bar;p1;p2", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/f00/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"p2", (Object)uri.getParam());
        Assertions.assertEquals((Object)"name=value", (Object)uri.getQuery());
        uri.setQuery("other=123456");
        Assertions.assertEquals((Object)"http://host:8888/f%30%30;p0/bar;p1;p2?other=123456", (Object)uri.toString());
        Assertions.assertEquals((Object)"/f%30%30;p0/bar;p1;p2", (Object)uri.getPath());
        Assertions.assertEquals((Object)"/f00/bar", (Object)uri.getDecodedPath());
        Assertions.assertEquals((Object)"p2", (Object)uri.getParam());
        Assertions.assertEquals((Object)"other=123456", (Object)uri.getQuery());
    }

    @Test
    public void testSchemeAndOrAuthority() throws Exception {
        HttpURI uri = new HttpURI("/path/info");
        Assertions.assertEquals((Object)"/path/info", (Object)uri.toString());
        uri.setAuthority("host", 0);
        Assertions.assertEquals((Object)"//host/path/info", (Object)uri.toString());
        uri.setAuthority("host", 8888);
        Assertions.assertEquals((Object)"//host:8888/path/info", (Object)uri.toString());
        uri.setScheme("http");
        Assertions.assertEquals((Object)"http://host:8888/path/info", (Object)uri.toString());
        uri.setAuthority(null, 0);
        Assertions.assertEquals((Object)"http:/path/info", (Object)uri.toString());
    }

    @Test
    public void testSetters() throws Exception {
        HttpURI uri = new HttpURI();
        Assertions.assertEquals((Object)"", (Object)uri.toString());
        uri = new HttpURI(null, null, 0, null, null, null, null);
        Assertions.assertEquals((Object)"", (Object)uri.toString());
        uri.setPath("/path/info");
        Assertions.assertEquals((Object)"/path/info", (Object)uri.toString());
        uri.setAuthority("host", 8080);
        Assertions.assertEquals((Object)"//host:8080/path/info", (Object)uri.toString());
        uri.setParam("param");
        Assertions.assertEquals((Object)"//host:8080/path/info;param", (Object)uri.toString());
        uri.setQuery("a=b");
        Assertions.assertEquals((Object)"//host:8080/path/info;param?a=b", (Object)uri.toString());
        uri.setScheme("http");
        Assertions.assertEquals((Object)"http://host:8080/path/info;param?a=b", (Object)uri.toString());
        uri.setPathQuery("/other;xxx/path;ppp?query");
        Assertions.assertEquals((Object)"http://host:8080/other;xxx/path;ppp?query", (Object)uri.toString());
        MatcherAssert.assertThat((Object)uri.getScheme(), (Matcher)Matchers.is((Object)"http"));
        MatcherAssert.assertThat((Object)uri.getAuthority(), (Matcher)Matchers.is((Object)"host:8080"));
        MatcherAssert.assertThat((Object)uri.getHost(), (Matcher)Matchers.is((Object)"host"));
        MatcherAssert.assertThat((Object)uri.getPort(), (Matcher)Matchers.is((Object)8080));
        MatcherAssert.assertThat((Object)uri.getPath(), (Matcher)Matchers.is((Object)"/other;xxx/path;ppp"));
        MatcherAssert.assertThat((Object)uri.getDecodedPath(), (Matcher)Matchers.is((Object)"/other/path"));
        MatcherAssert.assertThat((Object)uri.getParam(), (Matcher)Matchers.is((Object)"ppp"));
        MatcherAssert.assertThat((Object)uri.getQuery(), (Matcher)Matchers.is((Object)"query"));
        MatcherAssert.assertThat((Object)uri.getPathQuery(), (Matcher)Matchers.is((Object)"/other;xxx/path;ppp?query"));
        uri.setPathQuery(null);
        Assertions.assertEquals((Object)"http://host:8080?query", (Object)uri.toString());
        uri.setQuery(null);
        Assertions.assertEquals((Object)"http://host:8080", (Object)uri.toString());
        uri.setPathQuery("/other;xxx/path;ppp?query");
        Assertions.assertEquals((Object)"http://host:8080/other;xxx/path;ppp?query", (Object)uri.toString());
        uri.setScheme(null);
        Assertions.assertEquals((Object)"//host:8080/other;xxx/path;ppp?query", (Object)uri.toString());
        uri.setAuthority(null, -1);
        Assertions.assertEquals((Object)"/other;xxx/path;ppp?query", (Object)uri.toString());
        uri.setParam(null);
        Assertions.assertEquals((Object)"/other;xxx/path?query", (Object)uri.toString());
        uri.setQuery(null);
        Assertions.assertEquals((Object)"/other;xxx/path", (Object)uri.toString());
        uri.setPath(null);
        Assertions.assertEquals((Object)"", (Object)uri.toString());
    }

    public static Stream<Arguments> decodePathTests() {
        return Arrays.stream(new Object[][]{{"http://host/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"//host/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"ht..tp://host/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"ht1.2+..-3.4tp://127.0.0.1:8080/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"http://h%2est/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"http://h..est/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"http://host/../path/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"http://host/path/../info", "/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"http://host/path/./info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"//host/path/../info", "/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"//host/path/./info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/../info", "/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/./info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"path/../info", "info", EnumSet.noneOf(HttpURI.Violation.class)}, {"path/./info", "path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/f%6f%6F/bar", "/foo/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {"/f%u006f%u006F/bar", "/foo/bar", EnumSet.of(HttpURI.Violation.UTF16)}, {"/f%u0001%u0001/bar", "/f\u0001\u0001/bar", EnumSet.of(HttpURI.Violation.UTF16)}, {"/foo/%u20AC/bar", "/foo/\u20ac/bar", EnumSet.of(HttpURI.Violation.UTF16)}, {"//host/../path/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/../path/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"../path/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/%XX/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/%2/F/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/%/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/%u000X/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/Fo%u0000/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/Fo%00/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/Foo/info%u0000", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/Foo/info%00", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/%U20AC", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%2e%2e/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%u002e%u002e/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%2e%2e;/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%u002e%u002e;/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%2e.", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%u002e.", null, EnumSet.noneOf(HttpURI.Violation.class)}, {".%2e", null, EnumSet.noneOf(HttpURI.Violation.class)}, {".%u002e", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%2e%2e", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%u002e%u002e", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%2e%u002e", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"%u002e%2e", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"..;/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"..;param/info", null, EnumSet.noneOf(HttpURI.Violation.class)}, {"scheme://host/path/%2e/info", "/path/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"scheme:/path/%2e/info", "/path/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e/info", "/path/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"path/%2e/info/", "path/info/", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e/info", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e;/info", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e;param/info", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e;param;other/info;other", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"%2e/info", "info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"%u002e/info", "info", EnumSet.of(HttpURI.Violation.SEGMENT, HttpURI.Violation.UTF16)}, {"%2e", "", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"%u002e", "", EnumSet.of(HttpURI.Violation.SEGMENT, HttpURI.Violation.UTF16)}, {"/foo//bar", "/foo//bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo//../bar", "/foo/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo///../../../bar", "/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo/./../bar", "/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo//./bar", "/foo//bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"foo/bar", "foo/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {"foo;/bar", "foo/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {";/bar", "/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {";?n=v", "", EnumSet.of(HttpURI.Violation.EMPTY)}, {"?n=v", "", EnumSet.noneOf(HttpURI.Violation.class)}, {"#n=v", "", EnumSet.noneOf(HttpURI.Violation.class)}, {"", "", EnumSet.noneOf(HttpURI.Violation.class)}, {"http:/foo", "/foo", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/.;/info", "/path/info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/.;param/info", "/path/info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/..;/info", "/info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/..;param/info", "/info", EnumSet.of(HttpURI.Violation.PARAM)}, {".;/info", "info", EnumSet.of(HttpURI.Violation.PARAM)}, {".;param/info", "info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/%2f/info", "/path///info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"%2f/info", "//info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"%2F/info", "//info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"/path/%2f../info", "/path/info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"/path/%25/info", "/path/%/info", EnumSet.of(HttpURI.Violation.ENCODING)}, {"/path/%u0025/info", "/path/%/info", EnumSet.of(HttpURI.Violation.ENCODING, HttpURI.Violation.UTF16)}, {"%25/info", "%/info", EnumSet.of(HttpURI.Violation.ENCODING)}, {"/path/%25../info", "/path/%../info", EnumSet.of(HttpURI.Violation.ENCODING)}, {"/path/%u0025../info", "/path/%../info", EnumSet.of(HttpURI.Violation.ENCODING, HttpURI.Violation.UTF16)}, {"/path/%2f/..;/info", "/path//info", EnumSet.of(HttpURI.Violation.SEPARATOR, HttpURI.Violation.PARAM)}, {"/path/%u002f/..;/info", "/path//info", EnumSet.of(HttpURI.Violation.SEPARATOR, HttpURI.Violation.PARAM, HttpURI.Violation.UTF16)}, {"/path/%2f/..;/%2e/info", "/path//info", EnumSet.of(HttpURI.Violation.SEPARATOR, HttpURI.Violation.PARAM, HttpURI.Violation.SEGMENT)}, {"http://localhost:9000/x\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32", "/x\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32", EnumSet.noneOf(HttpURI.Violation.class)}, {"http://localhost:9000/\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32", "/\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32\ud83c\udf32", EnumSet.noneOf(HttpURI.Violation.class)}}).map(Arguments::of);
    }

    @ParameterizedTest
    @MethodSource(value={"decodePathTests"})
    public void testDecodedPath(String input, String decodedPath, EnumSet<HttpURI.Violation> expected) {
        try {
            HttpURI uri = new HttpURI(input);
            MatcherAssert.assertThat((Object)uri.getDecodedPath(), (Matcher)Matchers.is((Object)decodedPath));
            EnumSet<HttpURI.Violation> ambiguous = EnumSet.copyOf(expected);
            ambiguous.retainAll(EnumSet.complementOf(EnumSet.of(HttpURI.Violation.UTF16)));
            MatcherAssert.assertThat((Object)uri.isAmbiguous(), (Matcher)Matchers.is((Object)(!ambiguous.isEmpty() ? 1 : 0)));
            MatcherAssert.assertThat((Object)uri.hasAmbiguousSegment(), (Matcher)Matchers.is((Object)ambiguous.contains(HttpURI.Violation.SEGMENT)));
            MatcherAssert.assertThat((Object)uri.hasAmbiguousSeparator(), (Matcher)Matchers.is((Object)ambiguous.contains(HttpURI.Violation.SEPARATOR)));
            MatcherAssert.assertThat((Object)uri.hasAmbiguousParameter(), (Matcher)Matchers.is((Object)ambiguous.contains(HttpURI.Violation.PARAM)));
            MatcherAssert.assertThat((Object)uri.hasAmbiguousEncoding(), (Matcher)Matchers.is((Object)ambiguous.contains(HttpURI.Violation.ENCODING)));
            MatcherAssert.assertThat((Object)uri.hasUtf16Encoding(), (Matcher)Matchers.is((Object)expected.contains(HttpURI.Violation.UTF16)));
        }
        catch (Exception e) {
            if (decodedPath != null) {
                e.printStackTrace();
            }
            MatcherAssert.assertThat((Object)decodedPath, (Matcher)Matchers.nullValue());
        }
    }

    public static Stream<Arguments> testPathQueryTests() {
        return Arrays.stream(new Object[][]{{"/path/info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/../info", "/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/./info", "/path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"path/../info", "info", EnumSet.noneOf(HttpURI.Violation.class)}, {"path/./info", "path/info", EnumSet.noneOf(HttpURI.Violation.class)}, {"/../path/info", null, null}, {"../path/info", null, null}, {"/path/%XX/info", null, null}, {"/path/%2/F/info", null, null}, {"%2e%2e/info", null, null}, {"%2e%2e;/info", null, null}, {"%2e.", null, null}, {".%2e", null, null}, {"%2e%2e", null, null}, {"..;/info", null, null}, {"..;param/info", null, null}, {"/path/%2e/info", "/path/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"path/%2e/info/", "path/info/", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e/info", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e;/info", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e;param/info", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/path/%2e%2e;param;other/info;other", "/info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"%2e/info", "info", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"%2e", "", EnumSet.of(HttpURI.Violation.SEGMENT)}, {"/", "/", EnumSet.noneOf(HttpURI.Violation.class)}, {"/#", "/", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path", "/path", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/", "/path/", EnumSet.noneOf(HttpURI.Violation.class)}, {"//", "//", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo//", "/foo//", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo//bar", "/foo//bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"//foo/bar", "//foo/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo?bar", "/foo", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo#bar", "/foo", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo;bar", "/foo", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo/?bar", "/foo/", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo/#bar", "/foo/", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo/;param", "/foo/", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo/;param/bar", "/foo//bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo//bar", "/foo//bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo//bar//", "/foo//bar//", EnumSet.of(HttpURI.Violation.EMPTY)}, {"//foo//bar//", "//foo//bar//", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo//../bar", "/foo/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo///../../../bar", "/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"/foo/./../bar", "/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {"/foo//./bar", "/foo//bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {"foo/bar", "foo/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {"foo;/bar", "foo/bar", EnumSet.noneOf(HttpURI.Violation.class)}, {";/bar", "/bar", EnumSet.of(HttpURI.Violation.EMPTY)}, {";?n=v", "", EnumSet.of(HttpURI.Violation.EMPTY)}, {"?n=v", "", EnumSet.noneOf(HttpURI.Violation.class)}, {"#n=v", "", EnumSet.noneOf(HttpURI.Violation.class)}, {"", "", EnumSet.noneOf(HttpURI.Violation.class)}, {"/path/.;/info", "/path/info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/.;param/info", "/path/info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/..;/info", "/info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/..;param/info", "/info", EnumSet.of(HttpURI.Violation.PARAM)}, {".;/info", "info", EnumSet.of(HttpURI.Violation.PARAM)}, {".;param/info", "info", EnumSet.of(HttpURI.Violation.PARAM)}, {"/path/%2f/info", "/path///info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"%2f/info", "//info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"%2F/info", "//info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"/path/%2f../info", "/path/info", EnumSet.of(HttpURI.Violation.SEPARATOR)}, {"/path/%25/info", "/path/%/info", EnumSet.of(HttpURI.Violation.ENCODING)}, {"%25/info", "%/info", EnumSet.of(HttpURI.Violation.ENCODING)}, {"/path/%25../info", "/path/%../info", EnumSet.of(HttpURI.Violation.ENCODING)}, {"/path/%2f/..;/info", "/path//info", EnumSet.of(HttpURI.Violation.SEPARATOR, HttpURI.Violation.PARAM)}, {"/path/%2f/..;/%2e/info", "/path//info", EnumSet.of(HttpURI.Violation.SEPARATOR, HttpURI.Violation.PARAM, HttpURI.Violation.SEGMENT)}, {"/path/%2f/%25/..;/%2e//info", "/path////info", EnumSet.of(HttpURI.Violation.SEPARATOR, HttpURI.Violation.PARAM, HttpURI.Violation.SEGMENT, HttpURI.Violation.ENCODING, HttpURI.Violation.EMPTY)}}).map(Arguments::of);
    }

    @ParameterizedTest
    @MethodSource(value={"testPathQueryTests"})
    public void testPathQuery(String input, String decodedPath, EnumSet<HttpURI.Violation> expected) {
        HttpURI uri = new HttpURI();
        if (expected == null) {
            Assertions.assertThrows(Throwable.class, () -> uri.parseRequestTarget(HttpMethod.GET.asString(), input));
            return;
        }
        uri.parseRequestTarget(HttpMethod.GET.asString(), input);
        MatcherAssert.assertThat((Object)uri.getDecodedPath(), (Matcher)Matchers.is((Object)decodedPath));
        MatcherAssert.assertThat((Object)uri.isAmbiguous(), (Matcher)Matchers.is((Object)(!expected.isEmpty() ? 1 : 0)));
        MatcherAssert.assertThat((Object)uri.hasAmbiguousEmptySegment(), (Matcher)Matchers.is((Object)expected.contains(HttpURI.Violation.EMPTY)));
        MatcherAssert.assertThat((Object)uri.hasAmbiguousSegment(), (Matcher)Matchers.is((Object)expected.contains(HttpURI.Violation.SEGMENT)));
        MatcherAssert.assertThat((Object)uri.hasAmbiguousSeparator(), (Matcher)Matchers.is((Object)expected.contains(HttpURI.Violation.SEPARATOR)));
        MatcherAssert.assertThat((Object)uri.hasAmbiguousParameter(), (Matcher)Matchers.is((Object)expected.contains(HttpURI.Violation.PARAM)));
        MatcherAssert.assertThat((Object)uri.hasAmbiguousEncoding(), (Matcher)Matchers.is((Object)expected.contains(HttpURI.Violation.ENCODING)));
    }

    public static Stream<Arguments> parseData() {
        return Stream.of(Arguments.of((Object[])new Object[]{"path", null, null, "-1", "path", null, null, null}), Arguments.of((Object[])new Object[]{"path/path", null, null, "-1", "path/path", null, null, null}), Arguments.of((Object[])new Object[]{"%65ncoded/path", null, null, "-1", "%65ncoded/path", null, null, null}), Arguments.of((Object[])new Object[]{"/path/to/context", null, null, "-1", "/path/to/context", null, null, null}), Arguments.of((Object[])new Object[]{"http://example.com/path/to/context;param?query=%22value%22#fragment", "http", "example.com", "-1", "/path/to/context;param", "param", "query=%22value%22", "fragment"}), Arguments.of((Object[])new Object[]{"http://[::1]/path/to/context;param?query=%22value%22#fragment", "http", "[::1]", "-1", "/path/to/context;param", "param", "query=%22value%22", "fragment"}), Arguments.of((Object[])new Object[]{"http://example.com:8080/path/to/context;param?query=%22value%22#fragment", "http", "example.com", "8080", "/path/to/context;param", "param", "query=%22value%22", "fragment"}), Arguments.of((Object[])new Object[]{"http://[::1]:8080/path/to/context;param?query=%22value%22#fragment", "http", "[::1]", "8080", "/path/to/context;param", "param", "query=%22value%22", "fragment"}), Arguments.of((Object[])new Object[]{"/path/info", null, null, null, "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"/path/info#fragment", null, null, null, "/path/info", null, null, "fragment"}), Arguments.of((Object[])new Object[]{"/path/info?query", null, null, null, "/path/info", null, "query", null}), Arguments.of((Object[])new Object[]{"/path/info?query#fragment", null, null, null, "/path/info", null, "query", "fragment"}), Arguments.of((Object[])new Object[]{"/path/info;param", null, null, null, "/path/info;param", "param", null, null}), Arguments.of((Object[])new Object[]{"/path/info;param#fragment", null, null, null, "/path/info;param", "param", null, "fragment"}), Arguments.of((Object[])new Object[]{"/path/info;param?query", null, null, null, "/path/info;param", "param", "query", null}), Arguments.of((Object[])new Object[]{"/path/info;param?query#fragment", null, null, null, "/path/info;param", "param", "query", "fragment"}), Arguments.of((Object[])new Object[]{"/path/info;a=b/foo;c=d", null, null, null, "/path/info;a=b/foo;c=d", "c=d", null, null}), Arguments.of((Object[])new Object[]{"//host/path/info", null, "host", null, "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"//user@host/path/info", null, "host", null, "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"//user@host:8080/path/info", null, "host", "8080", "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"//host:8080/path/info", null, "host", "8080", "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"http:/path/info", "http", null, null, "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"http:/path/info#fragment", "http", null, null, "/path/info", null, null, "fragment"}), Arguments.of((Object[])new Object[]{"http:/path/info?query", "http", null, null, "/path/info", null, "query", null}), Arguments.of((Object[])new Object[]{"http:/path/info?query#fragment", "http", null, null, "/path/info", null, "query", "fragment"}), Arguments.of((Object[])new Object[]{"http:/path/info;param", "http", null, null, "/path/info;param", "param", null, null}), Arguments.of((Object[])new Object[]{"http:/path/info;param#fragment", "http", null, null, "/path/info;param", "param", null, "fragment"}), Arguments.of((Object[])new Object[]{"http:/path/info;param?query", "http", null, null, "/path/info;param", "param", "query", null}), Arguments.of((Object[])new Object[]{"http:/path/info;param?query#fragment", "http", null, null, "/path/info;param", "param", "query", "fragment"}), Arguments.of((Object[])new Object[]{"http://user@host:8080/path/info;param?query#fragment", "http", "host", "8080", "/path/info;param", "param", "query", "fragment"}), Arguments.of((Object[])new Object[]{"xxxxx://user@host:8080/path/info;param?query#fragment", "xxxxx", "host", "8080", "/path/info;param", "param", "query", "fragment"}), Arguments.of((Object[])new Object[]{"http:///;?#", "http", null, null, "/;", "", "", ""}), Arguments.of((Object[])new Object[]{"/path/info?a=?query", null, null, null, "/path/info", null, "a=?query", null}), Arguments.of((Object[])new Object[]{"/path/info?a=;query", null, null, null, "/path/info", null, "a=;query", null}), Arguments.of((Object[])new Object[]{"/@path/info", null, null, null, "/@path/info", null, null, null}), Arguments.of((Object[])new Object[]{"/user@path/info", null, null, null, "/user@path/info", null, null, null}), Arguments.of((Object[])new Object[]{"//user@host/info", null, "host", null, "/info", null, null, null}), Arguments.of((Object[])new Object[]{"//@host/info", null, "host", null, "/info", null, null, null}), Arguments.of((Object[])new Object[]{"@host/info", null, null, null, "@host/info", null, null, null}), Arguments.of((Object[])new Object[]{"//host:8080//", null, "host", "8080", "//", null, null, null}), Arguments.of((Object[])new Object[]{"file:///path/info", "file", null, null, "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"file:/path/info", "file", null, null, "/path/info", null, null, null}), Arguments.of((Object[])new Object[]{"//", null, null, null, null, null, null, null}), Arguments.of((Object[])new Object[]{"http://localhost/", "http", "localhost", null, "/", null, null, null}), Arguments.of((Object[])new Object[]{"http://localhost:8080/", "http", "localhost", "8080", "/", null, null, null}), Arguments.of((Object[])new Object[]{"http://localhost/?x=y", "http", "localhost", null, "/", null, "x=y", null}), Arguments.of((Object[])new Object[]{"/;param", null, null, null, "/;param", "param", null, null}), Arguments.of((Object[])new Object[]{";param", null, null, null, ";param", "param", null, null}), Arguments.of((Object[])new Object[]{"/?x=y", null, null, null, "/", null, "x=y", null}), Arguments.of((Object[])new Object[]{"/?abc=test", null, null, null, "/", null, "abc=test", null}), Arguments.of((Object[])new Object[]{"/#fragment", null, null, null, "/", null, null, "fragment"}), Arguments.of((Object[])new Object[]{"http://192.0.0.1:8080/", "http", "192.0.0.1", "8080", "/", null, null, null}), Arguments.of((Object[])new Object[]{"http://[2001:db8::1]:8080/", "http", "[2001:db8::1]", "8080", "/", null, null, null}), Arguments.of((Object[])new Object[]{"http://user@[2001:db8::1]:8080/", "http", "[2001:db8::1]", "8080", "/", null, null, null}), Arguments.of((Object[])new Object[]{"http://[2001:db8::1]/", "http", "[2001:db8::1]", null, "/", null, null, null}), Arguments.of((Object[])new Object[]{"//[2001:db8::1]:8080/", null, "[2001:db8::1]", "8080", "/", null, null, null}), Arguments.of((Object[])new Object[]{"*", null, null, null, "*", null, null, null}), Arguments.of((Object[])new Object[]{"http://host:8080/path/info?q1=v1&q2=v2", "http", "host", "8080", "/path/info", null, "q1=v1&q2=v2", null}), Arguments.of((Object[])new Object[]{"/path/info?q1=v1&q2=v2", null, null, null, "/path/info", null, "q1=v1&q2=v2", null}), Arguments.of((Object[])new Object[]{"/info?q1=v1&q2=v2", null, null, null, "/info", null, "q1=v1&q2=v2", null}), Arguments.of((Object[])new Object[]{"info?q1=v1&q2=v2", null, null, null, "info", null, "q1=v1&q2=v2", null}), Arguments.of((Object[])new Object[]{"info;q1=v1?q2=v2", null, null, null, "info;q1=v1", "q1=v1", "q2=v2", null}), Arguments.of((Object[])new Object[]{"?q1=v1&q2=v2", null, null, null, "", null, "q1=v1&q2=v2", null}));
    }

    @ParameterizedTest
    @MethodSource(value={"parseData"})
    public void testParseString(String input, String scheme, String host, Integer port, String path, String param, String query, String fragment) {
        HttpURI httpUri = new HttpURI(input);
        try {
            new URI(input);
            MatcherAssert.assertThat((String)("[" + input + "] expected path (test case) cannot be null"), (Object)path, (Matcher)Matchers.notNullValue());
            MatcherAssert.assertThat((String)("[" + input + "] .scheme"), (Object)httpUri.getScheme(), (Matcher)Matchers.is((Object)scheme));
            MatcherAssert.assertThat((String)("[" + input + "] .host"), (Object)httpUri.getHost(), (Matcher)Matchers.is((Object)host));
            MatcherAssert.assertThat((String)("[" + input + "] .port"), (Object)httpUri.getPort(), (Matcher)Matchers.is((Object)(port == null ? -1 : port)));
            MatcherAssert.assertThat((String)("[" + input + "] .path"), (Object)httpUri.getPath(), (Matcher)Matchers.is((Object)path));
            MatcherAssert.assertThat((String)("[" + input + "] .param"), (Object)httpUri.getParam(), (Matcher)Matchers.is((Object)param));
            MatcherAssert.assertThat((String)("[" + input + "] .query"), (Object)httpUri.getQuery(), (Matcher)Matchers.is((Object)query));
            MatcherAssert.assertThat((String)("[" + input + "] .fragment"), (Object)httpUri.getFragment(), (Matcher)Matchers.is((Object)fragment));
            MatcherAssert.assertThat((String)("[" + input + "] .toString"), (Object)httpUri.toString(), (Matcher)Matchers.is((Object)input));
        }
        catch (URISyntaxException e) {
            MatcherAssert.assertThat((String)("[" + input + "] .scheme"), (Object)httpUri.getScheme(), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
            MatcherAssert.assertThat((String)("[" + input + "] .host"), (Object)httpUri.getHost(), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
            MatcherAssert.assertThat((String)("[" + input + "] .port"), (Object)httpUri.getPort(), (Matcher)Matchers.is((Object)-1));
            MatcherAssert.assertThat((String)("[" + input + "] .path"), (Object)httpUri.getPath(), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
            MatcherAssert.assertThat((String)("[" + input + "] .param"), (Object)httpUri.getParam(), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
            MatcherAssert.assertThat((String)("[" + input + "] .query"), (Object)httpUri.getQuery(), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
            MatcherAssert.assertThat((String)("[" + input + "] .fragment"), (Object)httpUri.getFragment(), (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
        }
    }

    @ParameterizedTest
    @MethodSource(value={"parseData"})
    public void testParseURI(String input, String scheme, String host, Integer port, String path, String param, String query, String fragment) throws Exception {
        URI javaUri = null;
        try {
            javaUri = new URI(input);
        }
        catch (URISyntaxException uRISyntaxException) {
            // empty catch block
        }
        Assumptions.assumeTrue((javaUri != null ? 1 : 0) != 0, (String)("Skipping, not a valid input URI: " + input));
        HttpURI httpUri = new HttpURI(input);
        MatcherAssert.assertThat((String)("[" + input + "] .scheme"), (Object)httpUri.getScheme(), (Matcher)Matchers.is((Object)scheme));
        MatcherAssert.assertThat((String)("[" + input + "] .host"), (Object)httpUri.getHost(), (Matcher)Matchers.is((Object)host));
        MatcherAssert.assertThat((String)("[" + input + "] .port"), (Object)httpUri.getPort(), (Matcher)Matchers.is((Object)(port == null ? -1 : port)));
        MatcherAssert.assertThat((String)("[" + input + "] .path"), (Object)httpUri.getPath(), (Matcher)Matchers.is((Object)path));
        MatcherAssert.assertThat((String)("[" + input + "] .param"), (Object)httpUri.getParam(), (Matcher)Matchers.is((Object)param));
        MatcherAssert.assertThat((String)("[" + input + "] .query"), (Object)httpUri.getQuery(), (Matcher)Matchers.is((Object)query));
        MatcherAssert.assertThat((String)("[" + input + "] .fragment"), (Object)httpUri.getFragment(), (Matcher)Matchers.is((Object)fragment));
        MatcherAssert.assertThat((String)("[" + input + "] .toString"), (Object)httpUri.toString(), (Matcher)Matchers.is((Object)input));
    }

    @ParameterizedTest
    @MethodSource(value={"parseData"})
    public void testCompareToJavaNetURI(String input, String scheme, String host, Integer port, String path, String param, String query, String fragment) throws Exception {
        URI javaUri = null;
        try {
            javaUri = new URI(input);
        }
        catch (URISyntaxException uRISyntaxException) {
            // empty catch block
        }
        Assumptions.assumeTrue((javaUri != null ? 1 : 0) != 0, (String)"Skipping, not a valid input URI");
        HttpURI httpUri = new HttpURI(input);
        MatcherAssert.assertThat((String)("[" + input + "] .scheme"), (Object)httpUri.getScheme(), (Matcher)Matchers.is((Object)javaUri.getScheme()));
        MatcherAssert.assertThat((String)("[" + input + "] .host"), (Object)httpUri.getHost(), (Matcher)Matchers.is((Object)javaUri.getHost()));
        MatcherAssert.assertThat((String)("[" + input + "] .port"), (Object)httpUri.getPort(), (Matcher)Matchers.is((Object)javaUri.getPort()));
        MatcherAssert.assertThat((String)("[" + input + "] .path"), (Object)httpUri.getPath(), (Matcher)Matchers.is((Object)javaUri.getRawPath()));
        MatcherAssert.assertThat((String)("[" + input + "] .query"), (Object)httpUri.getQuery(), (Matcher)Matchers.is((Object)javaUri.getRawQuery()));
        MatcherAssert.assertThat((String)("[" + input + "] .fragment"), (Object)httpUri.getFragment(), (Matcher)Matchers.is((Object)javaUri.getFragment()));
        MatcherAssert.assertThat((String)("[" + input + "] .toString"), (Object)httpUri.toString(), (Matcher)Matchers.is((Object)javaUri.toASCIIString()));
    }

    public static Stream<Arguments> queryData() {
        return Stream.of({"/path?p=%U20AC", "p=%U20AC"}, {"/path?p=%u20AC", "p=%u20AC"}).map(Arguments::of);
    }

    @ParameterizedTest
    @MethodSource(value={"queryData"})
    public void testEncodedQuery(String input, String expectedQuery) {
        HttpURI httpURI = new HttpURI(input);
        MatcherAssert.assertThat((String)("[" + input + "] .query"), (Object)httpURI.getQuery(), (Matcher)Matchers.is((Object)expectedQuery));
    }
}

