/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import jakarta.servlet.ServletException;
import jakarta.servlet.ServletOutputStream;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.net.Socket;
import java.nio.ByteBuffer;
import java.nio.channels.Channel;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.Arrays;
import java.util.concurrent.Exchanger;
import java.util.concurrent.TimeUnit;
import org.eclipse.jetty.io.AbstractConnection;
import org.eclipse.jetty.io.EndPoint;
import org.eclipse.jetty.io.ssl.SslConnection;
import org.eclipse.jetty.logging.StacklessLogging;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpChannel;
import org.eclipse.jetty.server.HttpOutput;
import org.eclipse.jetty.server.HttpServerTestFixture;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.eclipse.jetty.util.IO;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Tag;
import org.junit.jupiter.api.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class ConnectorTimeoutTest
extends HttpServerTestFixture {
    protected static final Logger LOG = LoggerFactory.getLogger(ConnectorTimeoutTest.class);
    protected static final int MAX_IDLE_TIME = 2000;
    private int sleepTime = 2400;
    private int minimumTestRuntime = 1600;
    private int maximumTestRuntime = 20000;

    @Override
    @BeforeEach
    public void before() {
        super.before();
        if (this._httpConfiguration != null) {
            this._httpConfiguration.setMinRequestDataRate(-1L);
            this._httpConfiguration.setIdleTimeout(-1L);
        }
    }

    @Test
    public void testMaxIdleWithRequest10() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.HelloWorldHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\n\r\n").getBytes("utf-8"));
            os.flush();
            IO.toString((InputStream)is);
            Thread.sleep(this.sleepTime);
            Assertions.assertEquals((int)-1, (int)is.read());
        });
        Assertions.assertTrue((TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start > (long)this.minimumTestRuntime ? 1 : 0) != 0);
        Assertions.assertTrue((TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start < (long)this.maximumTestRuntime ? 1 : 0) != 0);
    }

    @Test
    public void testMaxIdleWithRequest11() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            String content = "Wibble";
            byte[] contentB = content.getBytes("utf-8");
            os.write(("POST /echo HTTP/1.1\r\nhost: localhost:" + this._serverURI.getPort() + "\r\ncontent-type: text/plain; charset=utf-8\r\ncontent-length: " + contentB.length + "\r\n\r\n").getBytes("utf-8"));
            os.write(contentB);
            os.flush();
            IO.toString((InputStream)is);
            Thread.sleep(this.sleepTime);
            Assertions.assertEquals((int)-1, (int)is.read());
        });
        Assertions.assertTrue((TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start > (long)this.minimumTestRuntime ? 1 : 0) != 0);
        Assertions.assertTrue((TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start < (long)this.maximumTestRuntime ? 1 : 0) != 0);
    }

    @Test
    public void testMaxIdleWithRequest10NoClientClose() throws Exception {
        final Exchanger<Object> exchanger = new Exchanger<Object>();
        this.configureServer((Handler)new HttpServerTestFixture.HelloWorldHandler(){

            @Override
            public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
                try {
                    exchanger.exchange(baseRequest.getHttpChannel().getEndPoint());
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
                super.handle(target, baseRequest, request, response);
            }
        });
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        EndPoint endPoint = exchanger.exchange(null, 10L, TimeUnit.SECONDS);
        if (endPoint instanceof SslConnection.DecryptedEndPoint) {
            endPoint = ((SslConnection.DecryptedEndPoint)endPoint).getSslConnection().getEndPoint();
        }
        String result = IO.toString((InputStream)is);
        MatcherAssert.assertThat((String)"OK", (Object)result, (Matcher)Matchers.containsString((String)"200 OK"));
        Assertions.assertEquals((int)-1, (int)is.read());
        Assertions.assertTrue((boolean)endPoint.isOutputShutdown());
        TimeUnit.MILLISECONDS.sleep(4000L);
        Assertions.assertFalse((boolean)endPoint.isOpen());
        Object transport = endPoint.getTransport();
        if (transport instanceof Channel) {
            Assertions.assertFalse((boolean)((Channel)transport).isOpen());
        }
    }

    @Test
    public void testMaxIdleWithRequest11NoClientClose() throws Exception {
        final Exchanger<Object> exchanger = new Exchanger<Object>();
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler(){

            @Override
            public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
                try {
                    exchanger.exchange(baseRequest.getHttpChannel().getEndPoint());
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
                super.handle(target, baseRequest, request, response);
            }
        });
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        String content = "Wibble";
        byte[] contentB = content.getBytes("utf-8");
        os.write(("POST /echo HTTP/1.1\r\nhost: localhost:" + this._serverURI.getPort() + "\r\ncontent-type: text/plain; charset=utf-8\r\ncontent-length: " + contentB.length + "\r\nconnection: close\r\n\r\n").getBytes("utf-8"));
        os.write(contentB);
        os.flush();
        EndPoint endPoint = exchanger.exchange(null, 10L, TimeUnit.SECONDS);
        if (endPoint instanceof SslConnection.DecryptedEndPoint) {
            endPoint = ((SslConnection.DecryptedEndPoint)endPoint).getSslConnection().getEndPoint();
        }
        IO.toString((InputStream)is);
        Assertions.assertEquals((int)-1, (int)is.read());
        Assertions.assertTrue((boolean)endPoint.isOutputShutdown());
        TimeUnit.MILLISECONDS.sleep(4000L);
        Assertions.assertFalse((boolean)endPoint.isOpen());
        Object transport = endPoint.getTransport();
        if (transport instanceof Channel) {
            Assertions.assertFalse((boolean)((Channel)transport).isOpen());
        }
    }

    @Test
    @Tag(value="Unstable")
    @Disabled
    public void testNoBlockingTimeoutRead() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        InputStream is = client.getInputStream();
        Assertions.assertFalse((boolean)client.isClosed());
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        OutputStream os = client.getOutputStream();
        os.write(("GET / HTTP/1.1\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n5\r\nLMNOP\r\n").getBytes("utf-8"));
        os.flush();
        try {
            Thread.sleep(250L);
            os.write("1".getBytes("utf-8"));
            os.flush();
            Thread.sleep(250L);
            os.write("0".getBytes("utf-8"));
            os.flush();
            Thread.sleep(250L);
            os.write("\r".getBytes("utf-8"));
            os.flush();
            Thread.sleep(250L);
            os.write("\n".getBytes("utf-8"));
            os.flush();
            Thread.sleep(250L);
            os.write("0123456789ABCDEF\r\n".getBytes("utf-8"));
            os.write("0\r\n".getBytes("utf-8"));
            os.write("\r\n".getBytes("utf-8"));
            os.flush();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        long duration = TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start;
        MatcherAssert.assertThat((Object)duration, (Matcher)Matchers.greaterThan((Comparable)Long.valueOf(500L)));
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            String response = IO.toString((InputStream)is);
            MatcherAssert.assertThat((Object)response, (Matcher)Matchers.startsWith((String)"HTTP/1.1 200 OK"));
            MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"LMNOP0123456789ABCDEF"));
        });
    }

    @Test
    @Tag(value="Unstable")
    @Disabled
    public void testBlockingTimeoutRead() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        InputStream is = client.getInputStream();
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        os.write(("GET / HTTP/1.1\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n5\r\nLMNOP\r\n").getBytes("utf-8"));
        os.flush();
        try (StacklessLogging stackless = new StacklessLogging(new Class[]{HttpChannel.class});){
            Thread.sleep(300L);
            os.write("1".getBytes("utf-8"));
            os.flush();
            Thread.sleep(300L);
            os.write("0".getBytes("utf-8"));
            os.flush();
            Thread.sleep(300L);
            os.write("\r".getBytes("utf-8"));
            os.flush();
            Thread.sleep(300L);
            os.write("\n".getBytes("utf-8"));
            os.flush();
            Thread.sleep(300L);
            os.write("0123456789ABCDEF\r\n".getBytes("utf-8"));
            os.write("0\r\n".getBytes("utf-8"));
            os.write("\r\n".getBytes("utf-8"));
            os.flush();
        }
        long duration = TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start;
        MatcherAssert.assertThat((Object)duration, (Matcher)Matchers.greaterThan((Comparable)Long.valueOf(500L)));
        String response = IO.toString((InputStream)is);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.startsWith((String)"HTTP/1.1 500 "));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"InterruptedIOException"));
    }

    @Test
    @Tag(value="Unstable")
    @Disabled
    public void testNoBlockingTimeoutWrite() throws Exception {
        this.configureServer((Handler)new HugeResponseHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        BufferedReader is = new BufferedReader(new InputStreamReader(client.getInputStream(), StandardCharsets.ISO_8859_1), 2048);
        os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nConnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        String line = is.readLine();
        MatcherAssert.assertThat((Object)line, (Matcher)Matchers.startsWith((String)"HTTP/1.1 200 OK"));
        while (line.length() != 0) {
            line = is.readLine();
        }
        for (int i = 0; i < 131072; ++i) {
            if (i % 1028 == 0) {
                Thread.sleep(20L);
            }
            line = is.readLine();
            MatcherAssert.assertThat((Object)line, (Matcher)Matchers.notNullValue());
            Assertions.assertEquals((int)1022, (int)line.length());
        }
    }

    @Test
    @Tag(value="Unstable")
    @Disabled
    public void testBlockingTimeoutWrite() throws Exception {
        this.configureServer((Handler)new HugeResponseHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        BufferedReader is = new BufferedReader(new InputStreamReader(client.getInputStream(), StandardCharsets.ISO_8859_1), 2048);
        os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nConnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        String line = is.readLine();
        MatcherAssert.assertThat((Object)line, (Matcher)Matchers.startsWith((String)"HTTP/1.1 200 OK"));
        while (line.length() != 0) {
            line = is.readLine();
        }
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        try (StacklessLogging stackless = new StacklessLogging(new Class[]{HttpChannel.class, AbstractConnection.class});){
            for (int i = 0; i < 131072; ++i) {
                if (i % 1028 == 0) {
                    Thread.sleep(20L);
                }
                if ((line = is.readLine()) != null) continue;
                break;
            }
        }
        long end = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        long duration = end - start;
        MatcherAssert.assertThat((Object)duration, (Matcher)Matchers.lessThan((Comparable)Long.valueOf(2560L)));
    }

    @Test
    public void testMaxIdleNoRequest() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        InputStream is = client.getInputStream();
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        os.write("GET ".getBytes("utf-8"));
        os.flush();
        Thread.sleep(this.sleepTime);
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            try {
                String response = IO.toString((InputStream)is);
                MatcherAssert.assertThat((Object)response, (Matcher)Matchers.is((Object)""));
                Assertions.assertEquals((int)-1, (int)is.read());
            }
            catch (Exception e) {
                LOG.warn(e.getMessage());
            }
        });
        Assertions.assertTrue((TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start < (long)this.maximumTestRuntime ? 1 : 0) != 0);
    }

    @Test
    public void testMaxIdleNothingSent() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        InputStream is = client.getInputStream();
        Assertions.assertFalse((boolean)client.isClosed());
        Thread.sleep(this.sleepTime);
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            try {
                String response = IO.toString((InputStream)is);
                MatcherAssert.assertThat((Object)response, (Matcher)Matchers.is((Object)""));
                Assertions.assertEquals((int)-1, (int)is.read());
            }
            catch (IOException e) {
                LOG.warn("Unable to read stream", (Throwable)e);
            }
        });
        Assertions.assertTrue((TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start < (long)this.maximumTestRuntime ? 1 : 0) != 0);
    }

    @Test
    public void testMaxIdleDelayedDispatch() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        InputStream is = client.getInputStream();
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        os.write(("GET / HTTP/1.1\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nContent-Length: 20\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            try {
                String response = IO.toString((InputStream)is);
                MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"500"));
                Assertions.assertEquals((int)-1, (int)is.read());
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        });
        int duration = (int)(TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start);
        MatcherAssert.assertThat((Object)duration, (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(2000)));
        MatcherAssert.assertThat((Object)duration, (Matcher)Matchers.lessThan((Comparable)Integer.valueOf(this.maximumTestRuntime)));
    }

    @Test
    public void testMaxIdleDispatch() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        InputStream is = client.getInputStream();
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        os.write(("GET / HTTP/1.1\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nContent-Length: 20\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n1234567890").getBytes("utf-8"));
        os.flush();
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            try {
                String response = IO.toString((InputStream)is);
                MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"500"));
                Assertions.assertEquals((int)-1, (int)is.read());
            }
            catch (Exception e) {
                e.printStackTrace();
            }
        });
        int duration = (int)(TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start);
        MatcherAssert.assertThat((Object)(duration + 100), (Matcher)Matchers.greaterThanOrEqualTo((Comparable)Integer.valueOf(2000)));
        MatcherAssert.assertThat((Object)(duration - 100), (Matcher)Matchers.lessThan((Comparable)Integer.valueOf(this.maximumTestRuntime)));
    }

    @Test
    public void testMaxIdleWithSlowRequest() throws Exception {
        this.configureServer((Handler)new HttpServerTestFixture.EchoHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        String content = "Wibble\r\n";
        byte[] contentB = content.getBytes("utf-8");
        os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nContent-Length: " + contentB.length * 20 + "\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            for (int i = 0; i < 20; ++i) {
                Thread.sleep(50L);
                os.write(contentB);
                os.flush();
            }
            String in = IO.toString((InputStream)is);
            int offset = 0;
            for (int i = 0; i < 20; ++i) {
                Assertions.assertTrue(((offset = in.indexOf("Wibble", offset + 1)) > 0 ? 1 : 0) != 0, (String)("" + i));
            }
        });
    }

    @Test
    public void testMaxIdleWithSlowResponse() throws Exception {
        this.configureServer((Handler)new SlowResponseHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nConnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            String in = IO.toString((InputStream)is);
            int offset = 0;
            for (int i = 0; i < 20; ++i) {
                Assertions.assertTrue(((offset = in.indexOf("Hello World", offset + 1)) > 0 ? 1 : 0) != 0, (String)("" + i));
            }
        });
    }

    @Test
    public void testMaxIdleWithWait() throws Exception {
        this.configureServer((Handler)new WaitHandler());
        Socket client = this.newSocket(this._serverURI.getHost(), this._serverURI.getPort());
        client.setSoTimeout(10000);
        Assertions.assertFalse((boolean)client.isClosed());
        OutputStream os = client.getOutputStream();
        InputStream is = client.getInputStream();
        os.write(("GET / HTTP/1.0\r\nhost: localhost:" + this._serverURI.getPort() + "\r\nconnection: keep-alive\r\nConnection: close\r\n\r\n").getBytes("utf-8"));
        os.flush();
        Assertions.assertTimeoutPreemptively((Duration)Duration.ofSeconds(10L), () -> {
            String in = IO.toString((InputStream)is);
            MatcherAssert.assertThat((Object)in, (Matcher)Matchers.containsString((String)"Hello World"));
        });
    }

    static {
        System.setProperty("org.eclipse.jetty.io.nio.IDLE_TICK", "500");
    }

    protected static class WaitHandler
    extends AbstractHandler {
        protected WaitHandler() {
        }

        public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
            baseRequest.setHandled(true);
            response.setStatus(200);
            ServletOutputStream out = response.getOutputStream();
            try {
                Thread.sleep(2000L);
            }
            catch (Exception e) {
                e.printStackTrace();
            }
            out.write("Hello World\r\n".getBytes());
            out.flush();
        }
    }

    protected static class HugeResponseHandler
    extends AbstractHandler {
        protected HugeResponseHandler() {
        }

        public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
            baseRequest.setHandled(true);
            response.setStatus(200);
            ServletOutputStream out = response.getOutputStream();
            byte[] buffer = new byte[0x8000000];
            Arrays.fill(buffer, (byte)120);
            for (int i = 0; i < 131072; ++i) {
                buffer[i * 1024 + 1022] = 13;
                buffer[i * 1024 + 1023] = 10;
            }
            ByteBuffer bb = ByteBuffer.wrap(buffer);
            ((HttpOutput)out).sendContent(bb);
            out.close();
        }
    }

    protected static class SlowResponseHandler
    extends AbstractHandler {
        protected SlowResponseHandler() {
        }

        public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
            baseRequest.setHandled(true);
            response.setStatus(200);
            ServletOutputStream out = response.getOutputStream();
            for (int i = 0; i < 20; ++i) {
                out.write("Hello World\r\n".getBytes());
                out.flush();
                try {
                    Thread.sleep(50L);
                    continue;
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
            out.close();
        }
    }
}

