/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import jakarta.servlet.MultipartConfigElement;
import jakarta.servlet.ReadListener;
import jakarta.servlet.ServletInputStream;
import jakarta.servlet.http.Part;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Base64;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.eclipse.jetty.server.MultiPartFormInputStream;
import org.eclipse.jetty.toolchain.test.MavenTestingUtils;
import org.eclipse.jetty.util.IO;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

public class MultiPartFormInputStreamTest {
    private static final AtomicInteger testCounter = new AtomicInteger();
    private static final String FILENAME = "stuff.txt";
    protected String _contentType = "multipart/form-data, boundary=AaB03x";
    protected String _multi = MultiPartFormInputStreamTest.createMultipartRequestString("stuff.txt");
    protected File _tmpDir = MavenTestingUtils.getTargetTestingDir((String)String.valueOf(testCounter.incrementAndGet()));
    protected String _dirname = this._tmpDir.getAbsolutePath();

    public MultiPartFormInputStreamTest() {
        this._tmpDir.deleteOnExit();
    }

    @Test
    public void testBadMultiPartRequest() {
        String boundary = "X0Y0";
        String str = "--" + boundary + "\r\nContent-Disposition: form-data; name=\"fileup\"; filename=\"test.upload\"\r\nContent-Type: application/octet-stream\r\n\r\nHow now brown cow.\r\n--" + boundary + "-\r\nContent-Disposition: form-data; name=\"fileup\"; filename=\"test.upload\"\r\n\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), "multipart/form-data, boundary=" + boundary, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IOException x = (IOException)Assertions.assertThrows(IOException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"Incomplete Multipart");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.startsWith((String)"Incomplete"));
    }

    @Test
    public void testFinalBoundaryOnly() throws Exception {
        String delimiter = "\r\n";
        String boundary = "MockMultiPartTestBoundary";
        String str = delimiter + "Hello world" + delimiter + delimiter + "--MockMultiPartTestBoundary--" + delimiter;
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), "multipart/form-data, boundary=MockMultiPartTestBoundary", config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Assertions.assertTrue((boolean)mpis.getParts().isEmpty());
    }

    @Test
    public void testEmpty() throws Exception {
        String delimiter = "\r\n";
        String boundary = "MockMultiPartTestBoundary";
        String str = delimiter + "--MockMultiPartTestBoundary--" + delimiter;
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), "multipart/form-data, boundary=MockMultiPartTestBoundary", config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Assertions.assertTrue((boolean)mpis.getParts().isEmpty());
    }

    @Test
    public void testNoBoundaryRequest() throws Exception {
        String str = "--\r\nContent-Disposition: form-data; name=\"fileName\"\r\nContent-Type: text/plain; charset=US-ASCII\r\nContent-Transfer-Encoding: 8bit\r\n\r\nabc\r\n--\r\nContent-Disposition: form-data; name=\"desc\"\r\nContent-Type: text/plain; charset=US-ASCII\r\nContent-Transfer-Encoding: 8bit\r\n\r\n123\r\n--\r\nContent-Disposition: form-data; name=\"title\"\r\nContent-Type: text/plain; charset=US-ASCII\r\nContent-Transfer-Encoding: 8bit\r\n\r\nttt\r\n--\r\nContent-Disposition: form-data; name=\"datafile5239138112980980385.txt\"; filename=\"datafile5239138112980980385.txt\"\r\nContent-Type: application/octet-stream; charset=ISO-8859-1\r\nContent-Transfer-Encoding: binary\r\n\r\n000\r\n----\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), "multipart/form-data", config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)4));
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        Part fileName = mpis.getPart("fileName");
        MatcherAssert.assertThat((Object)fileName, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)fileName.getSize(), (Matcher)Matchers.is((Object)3L));
        IO.copy((InputStream)fileName.getInputStream(), (OutputStream)baos);
        MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.US_ASCII), (Matcher)Matchers.is((Object)"abc"));
        baos = new ByteArrayOutputStream();
        Part desc = mpis.getPart("desc");
        MatcherAssert.assertThat((Object)desc, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)desc.getSize(), (Matcher)Matchers.is((Object)3L));
        IO.copy((InputStream)desc.getInputStream(), (OutputStream)baos);
        MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.US_ASCII), (Matcher)Matchers.is((Object)"123"));
        baos = new ByteArrayOutputStream();
        Part title = mpis.getPart("title");
        MatcherAssert.assertThat((Object)title, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)title.getSize(), (Matcher)Matchers.is((Object)3L));
        IO.copy((InputStream)title.getInputStream(), (OutputStream)baos);
        MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.US_ASCII), (Matcher)Matchers.is((Object)"ttt"));
    }

    @Test
    public void testNonMultiPartRequest() {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        Throwable t = Assertions.assertThrows(IllegalArgumentException.class, () -> new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(this._multi.getBytes()), "Content-type: text/plain", config, this._tmpDir));
        MatcherAssert.assertThat((Object)t.getMessage(), (Matcher)Matchers.is((Object)"content type is not multipart/form-data"));
    }

    @Test
    public void testNoBody() {
        String body = "";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(body.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IOException x = (IOException)Assertions.assertThrows(IOException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Missing initial multi part boundary"));
    }

    @Test
    public void testBodyAlreadyConsumed() throws Exception {
        ServletInputStream is = new ServletInputStream(){

            public boolean isFinished() {
                return true;
            }

            public boolean isReady() {
                return false;
            }

            public void setReadListener(ReadListener readListener) {
            }

            public int read() {
                return 0;
            }
        };
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)is, this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        Assertions.assertEquals((int)0, (int)parts.size());
    }

    @Test
    public void testWhitespaceBodyWithCRLF() {
        String whitespace = "              \n\n\n\r\n\r\n\r\n\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(whitespace.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IOException x = (IOException)Assertions.assertThrows(IOException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Missing initial multi part boundary"));
    }

    @Test
    public void testWhitespaceBody() {
        String whitespace = " ";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(whitespace.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IOException x = (IOException)Assertions.assertThrows(IOException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Missing initial"));
    }

    @Test
    public void testLeadingWhitespaceBodyWithCRLF() throws Exception {
        String body = "              \n\n\n\r\n\r\n\r\n\r\n--AaB03x\r\ncontent-disposition: form-data; name=\"field1\"\r\n\r\nJoe Blow\r\n--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"foo.txt\"\r\nContent-Type: text/plain\r\n\r\naaaabbbbb\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(body.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)2));
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream();){
            Part field1 = mpis.getPart("field1");
            MatcherAssert.assertThat((Object)field1, (Matcher)Matchers.notNullValue());
            IO.copy((InputStream)field1.getInputStream(), (OutputStream)baos);
            MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.US_ASCII), (Matcher)Matchers.is((Object)"Joe Blow"));
        }
        baos = new ByteArrayOutputStream();
        try {
            Part stuff = mpis.getPart("stuff");
            MatcherAssert.assertThat((Object)stuff, (Matcher)Matchers.notNullValue());
            IO.copy((InputStream)stuff.getInputStream(), (OutputStream)baos);
            MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.US_ASCII), (Matcher)Matchers.containsString((String)"aaaa"));
        }
        finally {
            baos.close();
        }
    }

    @Test
    public void testLeadingWhitespaceBodyWithoutCRLF() throws Exception {
        String body = "            --AaB03x\r\ncontent-disposition: form-data; name=\"field1\"\r\n\r\nJoe Blow\r\n--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"foo.txt\"\r\nContent-Type: text/plain\r\n\r\naaaabbbbb\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(body.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)1));
        Part stuff = mpis.getPart("stuff");
        MatcherAssert.assertThat((Object)stuff, (Matcher)Matchers.notNullValue());
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IO.copy((InputStream)stuff.getInputStream(), (OutputStream)baos);
        MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.US_ASCII), (Matcher)Matchers.containsString((String)"bbbbb"));
    }

    @Test
    public void testNoLimits() throws Exception {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(this._multi.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        Assertions.assertFalse((boolean)parts.isEmpty());
    }

    @Test
    public void testRequestTooBig() {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 60L, 100L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(this._multi.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Request exceeds maxRequestSize"));
    }

    @Test
    public void testRequestTooBigThrowsErrorOnGetParts() {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 60L, 100L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(this._multi.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Request exceeds maxRequestSize"));
        x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Request exceeds maxRequestSize"));
    }

    @Test
    public void testFileTooBig() {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 40L, 1024L, 30);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(this._multi.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"stuff.txt should have been larger than maxFileSize");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.startsWith((String)"Multipart Mime part"));
    }

    @Test
    public void testFileTooBigThrowsErrorOnGetParts() {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 40L, 1024L, 30);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(this._multi.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"stuff.txt should have been larger than maxFileSize");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.startsWith((String)"Multipart Mime part"));
        x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"stuff.txt should have been larger than maxFileSize");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.startsWith((String)"Multipart Mime part"));
    }

    @Test
    public void testPartFileRelative() throws Exception {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("tptfd").getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        mpis.getParts();
        MultiPartFormInputStream.MultiPart part = (MultiPartFormInputStream.MultiPart)mpis.getPart("stuff");
        File stuff = part.getFile();
        MatcherAssert.assertThat((Object)stuff, (Matcher)Matchers.notNullValue());
        part.write("tptfd.txt");
        File tptfd = new File(this._dirname + File.separator + "tptfd.txt");
        MatcherAssert.assertThat((Object)tptfd.exists(), (Matcher)Matchers.is((Object)true));
        MatcherAssert.assertThat((Object)stuff.exists(), (Matcher)Matchers.is((Object)false));
        part.cleanUp();
        MatcherAssert.assertThat((Object)tptfd.exists(), (Matcher)Matchers.is((Object)true));
        tptfd.deleteOnExit();
    }

    @Test
    public void testPartFileAbsolute() throws Exception {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("tpfa").getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        mpis.getParts();
        MultiPartFormInputStream.MultiPart part = (MultiPartFormInputStream.MultiPart)mpis.getPart("stuff");
        File stuff = part.getFile();
        MatcherAssert.assertThat((Object)stuff, (Matcher)Matchers.notNullValue());
        Path path = MavenTestingUtils.getTargetTestingPath().resolve("tpfa.txt");
        part.write(path.toFile().getAbsolutePath());
        File tpfa = path.toFile();
        MatcherAssert.assertThat((Object)tpfa.exists(), (Matcher)Matchers.is((Object)true));
        MatcherAssert.assertThat((Object)stuff.exists(), (Matcher)Matchers.is((Object)false));
        part.cleanUp();
        MatcherAssert.assertThat((Object)tpfa.exists(), (Matcher)Matchers.is((Object)true));
        tpfa.deleteOnExit();
    }

    @Test
    public void testPartFileAbsoluteFromBuffer() throws Exception {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 5000);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("tpfafb").getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        mpis.getParts();
        MultiPartFormInputStream.MultiPart part = (MultiPartFormInputStream.MultiPart)mpis.getPart("stuff");
        Assertions.assertNull((Object)part.getFile());
        Path path = MavenTestingUtils.getTargetTestingPath().resolve("tpfafb.txt");
        part.write(path.toFile().getAbsolutePath());
        File tpfafb = path.toFile();
        MatcherAssert.assertThat((Object)tpfafb.exists(), (Matcher)Matchers.is((Object)true));
        part.cleanUp();
        MatcherAssert.assertThat((Object)tpfafb.exists(), (Matcher)Matchers.is((Object)true));
        tpfafb.deleteOnExit();
    }

    @Test
    public void testPartFileRelativeFromBuffer() throws Exception {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 5000);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("tpfrfb").getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        mpis.getParts();
        MultiPartFormInputStream.MultiPart part = (MultiPartFormInputStream.MultiPart)mpis.getPart("stuff");
        Assertions.assertNull((Object)part.getFile());
        part.write("tpfrfb.txt");
        File tpfrfb = new File(this._tmpDir, "tpfrfb.txt");
        MatcherAssert.assertThat((Object)tpfrfb.exists(), (Matcher)Matchers.is((Object)true));
        part.cleanUp();
        MatcherAssert.assertThat((Object)tpfrfb.exists(), (Matcher)Matchers.is((Object)true));
        tpfrfb.deleteOnExit();
    }

    @Test
    public void testPartTmpFileDeletion() throws Exception {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("tptfd").getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        mpis.getParts();
        MultiPartFormInputStream.MultiPart part = (MultiPartFormInputStream.MultiPart)mpis.getPart("stuff");
        File stuff = part.getFile();
        MatcherAssert.assertThat((Object)stuff, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)stuff.exists(), (Matcher)Matchers.is((Object)true));
        part.cleanUp();
        MatcherAssert.assertThat((Object)stuff.exists(), (Matcher)Matchers.is((Object)false));
    }

    @Test
    public void testDeleteNPE() {
        ByteArrayInputStream input = new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("myFile").getBytes());
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 1024L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)input, this._contentType, config, this._tmpDir);
        mpis.deleteParts();
    }

    @Test
    public void testAsyncCleanUp() throws Exception {
        final CountDownLatch reading = new CountDownLatch(1);
        final ByteArrayInputStream wrappedStream = new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("myFile").getBytes());
        InputStream slowStream = new InputStream(){

            @Override
            public int read(byte[] b, int off, int len) throws IOException {
                return Math.max(0, super.read(b, off, len));
            }

            @Override
            public int read() throws IOException {
                reading.countDown();
                return wrappedStream.read();
            }
        };
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 1024L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream(slowStream, this._contentType, config, this._tmpDir);
        CompletableFuture cleanupError = new CompletableFuture();
        new Thread(() -> {
            try {
                Assertions.assertTrue((boolean)reading.await(5L, TimeUnit.SECONDS));
                mpis.deleteParts();
                cleanupError.complete(null);
            }
            catch (Throwable t) {
                cleanupError.complete(t);
            }
        }).start();
        Throwable error = Assertions.assertThrows(IOException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)Matchers.is((Object)"DELETING"));
        Assertions.assertNull(cleanupError.get());
        String[] fileList = this._tmpDir.list();
        Assertions.assertNotNull((Object)fileList);
        MatcherAssert.assertThat((Object)fileList.length, (Matcher)Matchers.is((Object)0));
    }

    @Test
    public void testParseAfterCleanUp() {
        ByteArrayInputStream input = new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString("myFile").getBytes());
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 1024L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)input, this._contentType, config, this._tmpDir);
        mpis.deleteParts();
        Throwable error = Assertions.assertThrows(IOException.class, () -> ((MultiPartFormInputStream)mpis).getParts());
        MatcherAssert.assertThat((Object)error.getMessage(), (Matcher)Matchers.is((Object)"DELETED"));
        Assertions.assertFalse((boolean)this._tmpDir.exists());
    }

    @Test
    public void testLFOnlyRequest() throws Exception {
        String str = "--AaB03x\ncontent-disposition: form-data; name=\"field1\"\n\nJoe Blow\r\n--AaB03x\ncontent-disposition: form-data; name=\"field2\"\n\nOther\r\n--AaB03x--\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)2));
        Part p1 = mpis.getPart("field1");
        MatcherAssert.assertThat((Object)p1, (Matcher)Matchers.notNullValue());
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p1.getInputStream(), (OutputStream)baos);
        MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.UTF_8), (Matcher)Matchers.is((Object)"Joe Blow"));
        Part p2 = mpis.getPart("field2");
        MatcherAssert.assertThat((Object)p2, (Matcher)Matchers.notNullValue());
        baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p2.getInputStream(), (OutputStream)baos);
        MatcherAssert.assertThat((Object)baos.toString(StandardCharsets.UTF_8), (Matcher)Matchers.is((Object)"Other"));
    }

    @Test
    public void testCROnlyRequest() {
        String str = "--AaB03x\rcontent-disposition: form-data; name=\"field1\"\r\rJoe Blow\r--AaB03x\rcontent-disposition: form-data; name=\"field2\"\r\rOther\r--AaB03x--\r";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"Improper EOL");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Bad EOL"));
    }

    @Test
    public void testCRandLFMixRequest() {
        String str = "--AaB03x\rcontent-disposition: form-data; name=\"field1\"\r\r\nJoe Blow\n\r--AaB03x\rcontent-disposition: form-data; name=\"field2\"\r\rOther\r--AaB03x--\r";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"Improper EOL");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Bad EOL"));
    }

    @Test
    public void testBufferOverflowNoCRLF() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        baos.write("--AaB03x\r\n".getBytes());
        for (int i = 0; i < 3000; ++i) {
            baos.write(97);
        }
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(baos.toByteArray()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        IllegalStateException x = (IllegalStateException)Assertions.assertThrows(IllegalStateException.class, () -> ((MultiPartFormInputStream)mpis).getParts(), (String)"Header Line Exceeded Max Length");
        MatcherAssert.assertThat((Object)x.getMessage(), (Matcher)Matchers.containsString((String)"Header Line Exceeded Max Length"));
    }

    @Test
    public void testCharsetEncoding() throws Exception {
        String contentType = "multipart/form-data; boundary=TheBoundary; charset=ISO-8859-1";
        String str = "--TheBoundary\r\ncontent-disposition: form-data; name=\"field1\"\r\n\r\n\nJoe Blow\n\r\n--TheBoundary--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(str.getBytes()), contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)1));
    }

    @Test
    public void testBadlyEncodedFilename() throws Exception {
        String contents = "--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"Taken on Aug 22 \\ 2012.jpg\"\r\nContent-Type: text/plain\r\n\r\nstuffaaa\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(contents.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)1));
        MatcherAssert.assertThat((Object)((Part)parts.iterator().next()).getSubmittedFileName(), (Matcher)Matchers.is((Object)"Taken on Aug 22 \\ 2012.jpg"));
    }

    @Test
    public void testBadlyEncodedMSFilename() throws Exception {
        String contents = "--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"c:\\this\\really\\is\\some\\path\\to\\a\\file.txt\"\r\nContent-Type: text/plain\r\n\r\nstuffaaa\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(contents.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)1));
        MatcherAssert.assertThat((Object)((Part)parts.iterator().next()).getSubmittedFileName(), (Matcher)Matchers.is((Object)"c:\\this\\really\\is\\some\\path\\to\\a\\file.txt"));
    }

    @Test
    public void testCorrectlyEncodedMSFilename() throws Exception {
        String contents = "--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"c:\\\\this\\\\really\\\\is\\\\some\\\\path\\\\to\\\\a\\\\file.txt\"\r\nContent-Type: text/plain\r\n\r\nstuffaaa\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(contents.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)1));
        MatcherAssert.assertThat((Object)((Part)parts.iterator().next()).getSubmittedFileName(), (Matcher)Matchers.is((Object)"c:\\this\\really\\is\\some\\path\\to\\a\\file.txt"));
    }

    @Test
    public void testMultiWithSpaceInFilename() throws Exception {
        this.testMulti("stuff with spaces.txt");
    }

    @Test
    public void testWriteFilesIfContentDispositionFilename() throws Exception {
        String s = "--AaB03x\r\ncontent-disposition: form-data; name=\"field1\"; filename=\"frooble.txt\"\r\n\r\nJoe Blow\r\n--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"\r\nContent-Type: text/plain\r\n\r\nsssaaa\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(s.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        mpis.setWriteFilesWithFilenames(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)2));
        Part field1 = mpis.getPart("field1");
        File f = ((MultiPartFormInputStream.MultiPart)field1).getFile();
        MatcherAssert.assertThat((Object)f, (Matcher)Matchers.notNullValue());
        Part stuff = mpis.getPart("stuff");
        f = ((MultiPartFormInputStream.MultiPart)stuff).getFile();
        MatcherAssert.assertThat((Object)f, (Matcher)Matchers.nullValue());
    }

    private void testMulti(String filename) throws IOException {
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(MultiPartFormInputStreamTest.createMultipartRequestString(filename).getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)2));
        Part field1 = mpis.getPart("field1");
        MatcherAssert.assertThat((Object)field1, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)field1.getName(), (Matcher)Matchers.is((Object)"field1"));
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        try (InputStream is = field1.getInputStream();){
            IO.copy((InputStream)is, (OutputStream)os);
        }
        Assertions.assertEquals((Object)"Joe Blow", (Object)new String(os.toByteArray()));
        Assertions.assertEquals((long)8L, (long)field1.getSize());
        Assertions.assertNotNull((Object)((MultiPartFormInputStream.MultiPart)field1).getBytes());
        field1.write("field1.txt");
        Assertions.assertNull((Object)((MultiPartFormInputStream.MultiPart)field1).getBytes());
        File f = new File(this._dirname + File.separator + "field1.txt");
        Assertions.assertTrue((boolean)f.exists());
        field1.write("another_field1.txt");
        File f2 = new File(this._dirname + File.separator + "another_field1.txt");
        Assertions.assertTrue((boolean)f2.exists());
        Assertions.assertFalse((boolean)f.exists());
        field1.delete();
        Assertions.assertFalse((boolean)f.exists());
        Assertions.assertFalse((boolean)f2.exists());
        MultiPartFormInputStream.MultiPart stuff = (MultiPartFormInputStream.MultiPart)mpis.getPart("stuff");
        MatcherAssert.assertThat((Object)stuff.getSubmittedFileName(), (Matcher)Matchers.is((Object)filename));
        MatcherAssert.assertThat((Object)stuff.getContentType(), (Matcher)Matchers.is((Object)"text/plain"));
        MatcherAssert.assertThat((Object)stuff.getHeader("Content-Type"), (Matcher)Matchers.is((Object)"text/plain"));
        MatcherAssert.assertThat((Object)stuff.getHeaders("content-type").size(), (Matcher)Matchers.is((Object)1));
        Assertions.assertNotNull((Object)stuff.getHeaders("non existing part"));
        MatcherAssert.assertThat((Object)stuff.getHeaders("non existing part").size(), (Matcher)Matchers.is((Object)0));
        MatcherAssert.assertThat((Object)stuff.getHeader("content-disposition"), (Matcher)Matchers.is((Object)("form-data; name=\"stuff\"; filename=\"" + filename + "\"")));
        MatcherAssert.assertThat((Object)stuff.getHeaderNames().size(), (Matcher)Matchers.is((Object)2));
        MatcherAssert.assertThat((Object)stuff.getSize(), (Matcher)Matchers.is((Object)51L));
        File tmpfile = stuff.getFile();
        MatcherAssert.assertThat((Object)tmpfile, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)stuff.getBytes(), (Matcher)Matchers.nullValue());
        MatcherAssert.assertThat((Object)tmpfile.exists(), (Matcher)Matchers.is((Object)true));
        MatcherAssert.assertThat((Object)tmpfile.getName(), (Matcher)Matchers.is((Matcher)Matchers.not((Object)"stuff with space.txt")));
        stuff.write(filename);
        f = new File(this._dirname + File.separator + filename);
        MatcherAssert.assertThat((Object)f.exists(), (Matcher)Matchers.is((Object)true));
        MatcherAssert.assertThat((Object)tmpfile.exists(), (Matcher)Matchers.is((Object)false));
        try {
            stuff.getInputStream();
        }
        catch (Exception e) {
            Assertions.fail((String)"Part.getInputStream() after file rename operation", (Throwable)e);
        }
        f.deleteOnExit();
    }

    @Test
    public void testMultiSameNames() throws Exception {
        String sameNames = "--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"stuff1.txt\"\r\nContent-Type: text/plain\r\n\r\n00000\r\n--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"stuff2.txt\"\r\nContent-Type: text/plain\r\n\r\n110000000000000000000000000000000000000000000000000\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(sameNames.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        Assertions.assertEquals((int)2, (int)parts.size());
        for (Part p : parts) {
            Assertions.assertEquals((Object)"stuff", (Object)p.getName());
        }
        Part p = mpis.getPart("stuff");
        Assertions.assertNotNull((Object)p);
        Assertions.assertEquals((long)5L, (long)p.getSize());
    }

    @Test
    public void testBase64EncodedContent() throws Exception {
        String contentWithEncodedPart = "--AaB03x\r\nContent-disposition: form-data; name=\"other\"\r\nContent-Type: text/plain\r\n\r\nother\r\n--AaB03x\r\nContent-disposition: form-data; name=\"stuff\"; filename=\"stuff.txt\"\r\nContent-Transfer-Encoding: base64\r\nContent-Type: application/octet-stream\r\n\r\n" + Base64.getEncoder().encodeToString("hello jetty".getBytes(StandardCharsets.ISO_8859_1)) + "\r\n--AaB03x\r\nContent-disposition: form-data; name=\"final\"\r\nContent-Type: text/plain\r\n\r\nthe end\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(contentWithEncodedPart.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        Assertions.assertEquals((int)3, (int)parts.size());
        Part p1 = mpis.getPart("other");
        Assertions.assertNotNull((Object)p1);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p1.getInputStream(), (OutputStream)baos);
        Assertions.assertEquals((Object)"other", (Object)baos.toString(StandardCharsets.US_ASCII));
        Part p2 = mpis.getPart("stuff");
        Assertions.assertNotNull((Object)p2);
        baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p2.getInputStream(), (OutputStream)baos);
        Assertions.assertEquals((Object)Base64.getEncoder().encodeToString("hello jetty".getBytes(StandardCharsets.ISO_8859_1)), (Object)baos.toString(StandardCharsets.US_ASCII));
        Part p3 = mpis.getPart("final");
        Assertions.assertNotNull((Object)p3);
        baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p3.getInputStream(), (OutputStream)baos);
        Assertions.assertEquals((Object)"the end", (Object)baos.toString(StandardCharsets.US_ASCII));
    }

    @Test
    public void testQuotedPrintableEncoding() throws Exception {
        String contentWithEncodedPart = "--AaB03x\r\nContent-disposition: form-data; name=\"other\"\r\nContent-Type: text/plain\r\n\r\nother\r\n--AaB03x\r\nContent-disposition: form-data; name=\"stuff\"; filename=\"stuff.txt\"\r\nContent-Transfer-Encoding: quoted-printable\r\nContent-Type: text/plain\r\n\r\ntruth=3Dbeauty\r\n--AaB03x--\r\n";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(contentWithEncodedPart.getBytes()), this._contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        Assertions.assertEquals((int)2, (int)parts.size());
        Part p1 = mpis.getPart("other");
        Assertions.assertNotNull((Object)p1);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p1.getInputStream(), (OutputStream)baos);
        Assertions.assertEquals((Object)"other", (Object)baos.toString(StandardCharsets.US_ASCII));
        Part p2 = mpis.getPart("stuff");
        Assertions.assertNotNull((Object)p2);
        baos = new ByteArrayOutputStream();
        IO.copy((InputStream)p2.getInputStream(), (OutputStream)baos);
        Assertions.assertEquals((Object)"truth=3Dbeauty", (Object)baos.toString(StandardCharsets.US_ASCII));
    }

    @Test
    public void testGeneratedForm() throws Exception {
        String contentType = "multipart/form-data, boundary=WebKitFormBoundary7MA4YWf7OaKlSxkTrZu0gW";
        String body = "Content-Type: multipart/form-data; boundary=WebKitFormBoundary7MA4YWf7OaKlSxkTrZu0gW\r\n\r\n--WebKitFormBoundary7MA4YWf7OaKlSxkTrZu0gW\r\nContent-Disposition: form-data; name=\"part1\"\r\n\nwNf\uff90xVam\uffbft\r\n--WebKitFormBoundary7MA4YWf7OaKlSxkTrZu0gW\nContent-Disposition: form-data; name=\"part2\"\r\n\r\n&\uffb3\u001b\u0014\uffba\uffd9\ufff9\uffd6\uffc3O\r\n--WebKitFormBoundary7MA4YWf7OaKlSxkTrZu0gW--";
        MultipartConfigElement config = new MultipartConfigElement(this._dirname, 1024L, 3072L, 50);
        MultiPartFormInputStream mpis = new MultiPartFormInputStream((InputStream)new ByteArrayInputStream(body.getBytes()), contentType, config, this._tmpDir);
        mpis.setDeleteOnExit(true);
        Collection parts = mpis.getParts();
        MatcherAssert.assertThat((Object)parts, (Matcher)Matchers.notNullValue());
        MatcherAssert.assertThat((Object)parts.size(), (Matcher)Matchers.is((Object)2));
        Part part1 = mpis.getPart("part1");
        MatcherAssert.assertThat((Object)part1, (Matcher)Matchers.notNullValue());
        Part part2 = mpis.getPart("part2");
        MatcherAssert.assertThat((Object)part2, (Matcher)Matchers.notNullValue());
    }

    private static String createMultipartRequestString(String filename) {
        int length = filename.length();
        String name = filename;
        if (length > 10) {
            name = filename.substring(0, 10);
        }
        StringBuilder filler = new StringBuilder();
        for (int i = name.length(); i < 51; ++i) {
            filler.append("0");
        }
        return "--AaB03x\r\ncontent-disposition: form-data; name=\"field1\"; filename=\"frooble.txt\"\r\n\r\nJoe Blow\r\n--AaB03x\r\ncontent-disposition: form-data; name=\"stuff\"; filename=\"" + filename + "\"\r\nContent-Type: text/plain\r\n\r\n" + name + filler.toString() + "\r\n--AaB03x--\r\n";
    }
}

