/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.eclipse.jetty.http.HttpVersion;
import org.eclipse.jetty.logging.StacklessLogging;
import org.eclipse.jetty.server.ConnectionFactory;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.DumpHandler;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.LocalConnector;
import org.eclipse.jetty.server.ProxyConnectionFactory;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.ErrorHandler;
import org.eclipse.jetty.toolchain.test.Net;
import org.eclipse.jetty.util.TypeUtil;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

public class ProxyConnectionTest {
    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testBadCRLF(RequestProcessor p) throws Exception {
        String request = "PROXY TCP 1.2.3.4 5.6.7.8 111 222\r \nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        Assertions.assertNull((Object)response);
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testBadChar(RequestProcessor p) throws Exception {
        String request = "PROXY\tTCP 1.2.3.4 5.6.7.8 111 222\r\nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        Assertions.assertNull((Object)response);
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testBadPort(RequestProcessor p) throws Exception {
        try (StacklessLogging stackless = new StacklessLogging(new Class[]{ProxyConnectionFactory.class});){
            String request = "PROXY TCP 1.2.3.4 5.6.7.8 9999999999999 222\r\nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
            String response = p.sendRequestWaitingForResponse(request);
            Assertions.assertNull((Object)response);
        }
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testHttp(RequestProcessor p) throws Exception {
        String request = "GET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"HTTP/1.1 200"));
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testIPv6(RequestProcessor p) throws Exception {
        Assumptions.assumeTrue((boolean)Net.isIpv6InterfaceAvailable());
        String request = "PROXY TCP6 eeee:eeee:eeee:eeee:eeee:eeee:eeee:eeee ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff 65535 65535\r\nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"HTTP/1.1 200"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"pathInfo=/path"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"remote=[eeee:eeee:eeee:eeee:eeee:eeee:eeee:eeee]:65535"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"local=[ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff]:65535"));
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testIPv6V2(RequestProcessor p) throws Exception {
        Assumptions.assumeTrue((boolean)Net.isIpv6InterfaceAvailable());
        String proxy = "0D0A0D0A000D0A515549540A21210024FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEEE30391F90";
        String http = "GET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(TypeUtil.fromHexString((String)proxy), http.getBytes(StandardCharsets.US_ASCII));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"HTTP/1.1 200"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"pathInfo=/path"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"local=[eeee:eeee:eeee:eeee:eeee:eeee:eeee:eeee]:8080"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"remote=[ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff]:12345"));
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testLocalV2(RequestProcessor p) throws Exception {
        String proxy = "0D0A0D0A000D0A515549540A20110010FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF";
        String http = "GET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(TypeUtil.fromHexString((String)proxy), http.getBytes(StandardCharsets.US_ASCII));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"HTTP/1.1 200"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"pathInfo=/path"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"local=0.0.0.0:0"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"remote=0.0.0.0:0"));
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testMissingField(RequestProcessor p) throws Exception {
        String request = "PROXY TCP 1.2.3.4 5.6.7.8 222\r\nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        Assertions.assertNull((Object)response);
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testNotComplete(RequestProcessor p) throws Exception {
        String response = p.customize(connector -> connector.setIdleTimeout(100L)).sendRequestWaitingForResponse("PROXY TIMEOUT");
        Assertions.assertNull((Object)response);
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testTooLong(RequestProcessor p) throws Exception {
        String request = "PROXY TOOLONG!!! eeee:eeee:eeee:eeee:0000:0000:0000:0000 ffff:ffff:ffff:ffff:0000:0000:0000:0000 65535 65535\r\nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        Assertions.assertNull((Object)response);
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testSimple(RequestProcessor p) throws Exception {
        String request = "PROXY TCP 1.2.3.4 5.6.7.8 111 222\r\nGET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(request);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"HTTP/1.1 200"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"pathInfo=/path"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"local=5.6.7.8:222"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"remote=1.2.3.4:111"));
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testSimpleV2(RequestProcessor p) throws Exception {
        String proxy = "0D0A0D0A000D0A515549540A2111000CC0A800017f00000130391F90";
        String http = "GET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(TypeUtil.fromHexString((String)proxy), http.getBytes(StandardCharsets.US_ASCII));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"HTTP/1.1 200"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"pathInfo=/path"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"local=127.0.0.1:8080"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"remote=192.168.0.1:12345"));
    }

    @ParameterizedTest
    @MethodSource(value={"requestProcessors"})
    public void testMaxHeaderLengthV2(RequestProcessor p) throws Exception {
        p.customize(connector -> {
            ProxyConnectionFactory factory = (ProxyConnectionFactory)connector.getConnectionFactory("[proxy]");
            factory.setMaxProxyHeader(11);
        });
        String proxy = "0D0A0D0A000D0A515549540A2111000CC0A800017f00000130391F90";
        String http = "GET /path HTTP/1.1\nHost: server:80\nConnection: close\n\n";
        String response = p.sendRequestWaitingForResponse(TypeUtil.fromHexString((String)proxy), http.getBytes(StandardCharsets.US_ASCII));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.is((Matcher)Matchers.nullValue()));
    }

    static Stream<Arguments> requestProcessors() {
        return Stream.of(Arguments.of((Object[])new Object[]{new RequestProcessor(){

            @Override
            public String process(byte[] ... requests) throws Exception {
                LocalConnector.LocalEndPoint endPoint = this._connector.connect();
                for (byte[] request : requests) {
                    endPoint.addInput(ByteBuffer.wrap(request));
                }
                return endPoint.getResponse();
            }

            public String toString() {
                return "All bytes at once";
            }
        }}), Arguments.of((Object[])new Object[]{new RequestProcessor(){

            @Override
            public String process(byte[] ... requests) throws Exception {
                LocalConnector.LocalEndPoint endPoint = this._connector.connect();
                byte[][] byArray = requests;
                int n = byArray.length;
                for (int i = 0; i < n; ++i) {
                    byte[] request;
                    for (byte b : request = byArray[i]) {
                        endPoint.addInput(ByteBuffer.wrap(new byte[]{b}));
                    }
                }
                return endPoint.getResponse();
            }

            public String toString() {
                return "Byte by byte";
            }
        }}));
    }

    static abstract class RequestProcessor {
        protected LocalConnector _connector;
        private Server _server = new Server();

        public RequestProcessor() {
            HttpConnectionFactory http = new HttpConnectionFactory();
            http.getHttpConfiguration().setRequestHeaderSize(1024);
            http.getHttpConfiguration().setResponseHeaderSize(1024);
            ProxyConnectionFactory proxy = new ProxyConnectionFactory(HttpVersion.HTTP_1_1.asString());
            this._connector = new LocalConnector(this._server, null, null, null, 1, new ConnectionFactory[]{proxy, http});
            this._connector.setIdleTimeout(1000L);
            this._server.addConnector((Connector)this._connector);
            this._server.setHandler((Handler)new DumpHandler());
            ErrorHandler eh = new ErrorHandler();
            eh.setServer(this._server);
            this._server.addBean((Object)eh);
        }

        public RequestProcessor customize(Consumer<LocalConnector> consumer) {
            consumer.accept(this._connector);
            return this;
        }

        public final String sendRequestWaitingForResponse(String request) throws Exception {
            return this.sendRequestWaitingForResponse(new byte[][]{request.getBytes(StandardCharsets.US_ASCII)});
        }

        public final String sendRequestWaitingForResponse(byte[] ... requests) throws Exception {
            try {
                this._server.start();
                String string = this.process(requests);
                return string;
            }
            finally {
                this.destroy();
            }
        }

        protected abstract String process(byte[] ... var1) throws Exception;

        private void destroy() throws Exception {
            this._server.stop();
            this._server.join();
        }
    }
}

