/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.server;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringReader;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import java.util.stream.Stream;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.eclipse.jetty.http.HttpCompliance;
import org.eclipse.jetty.http.HttpHeader;
import org.eclipse.jetty.http.HttpParser;
import org.eclipse.jetty.http.HttpTester;
import org.eclipse.jetty.http.MimeTypes;
import org.eclipse.jetty.server.ConnectionFactory;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.DumpHandler;
import org.eclipse.jetty.server.Handler;
import org.eclipse.jetty.server.HttpChannel;
import org.eclipse.jetty.server.HttpConfiguration;
import org.eclipse.jetty.server.HttpConnection;
import org.eclipse.jetty.server.HttpConnectionFactory;
import org.eclipse.jetty.server.LocalConnector;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.eclipse.jetty.server.handler.ErrorHandler;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.util.log.StacklessLogging;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;

public class HttpConnectionTest {
    private Server server;
    private LocalConnector connector;
    static final int CHUNKED = -1;
    static final int DQUOTED_CHUNKED = -2;
    static final int BAD_CHUNKED = -3;
    static final int UNKNOWN_TE = -4;

    @BeforeEach
    public void init() throws Exception {
        this.server = new Server();
        HttpConfiguration config = new HttpConfiguration();
        config.setRequestHeaderSize(1024);
        config.setResponseHeaderSize(1024);
        config.setSendDateHeader(true);
        HttpConnectionFactory http = new HttpConnectionFactory(config);
        this.connector = new LocalConnector(this.server, (ConnectionFactory)http, null);
        this.connector.setIdleTimeout(5000L);
        this.server.addConnector((Connector)this.connector);
        this.server.setHandler((Handler)new DumpHandler());
        ErrorHandler eh = new ErrorHandler();
        eh.setServer(this.server);
        this.server.addBean((Object)eh);
        this.server.start();
    }

    @AfterEach
    public void destroy() throws Exception {
        this.server.stop();
        this.server.join();
    }

    @Test
    public void testFragmentedChunk() throws Exception {
        String response = null;
        try {
            int offset = 0;
            response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            offset = this.checkContains(response, offset, "HTTP/1.1 200");
            offset = this.checkContains(response, offset, "/R1");
            this.checkContains(response, offset, "12345");
            offset = 0;
            response = this.connector.getResponse("GET /R2 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n5;\r\nABCDE\r\n0;\r\n\r\n");
            offset = this.checkContains(response, offset, "HTTP/1.1 200");
            offset = this.checkContains(response, offset, "/R2");
            this.checkContains(response, offset, "ABCDE");
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testHttp09NoVersion() throws Exception {
        ((HttpConnectionFactory)this.connector.getConnectionFactory(HttpConnectionFactory.class)).setHttpCompliance(HttpCompliance.RFC2616);
        String request = "GET / HTTP/0.9\r\n\r\n";
        String response = this.connector.getResponse(request);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"505 Unsupported Version"));
        ((HttpConnectionFactory)this.connector.getConnectionFactory(HttpConnectionFactory.class)).setHttpCompliance(HttpCompliance.RFC7230);
        request = "GET / HTTP/0.9\r\n\r\n";
        response = this.connector.getResponse(request);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"505 Unsupported Version"));
    }

    @Test
    public void testHttp09NoHeaders() throws Exception {
        ((HttpConnectionFactory)this.connector.getConnectionFactory(HttpConnectionFactory.class)).setHttpCompliance(HttpCompliance.RFC2616);
        String request = "GET /one\r\nGET :/two\r\n\r\n";
        String response = BufferUtil.toString((ByteBuffer)this.connector.executeRequest(request).waitForOutput(10L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"pathInfo=/"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.not((Matcher)Matchers.containsString((String)"two")));
    }

    @Test
    public void testHttp09MultipleRequests() throws Exception {
        ((HttpConnectionFactory)this.connector.getConnectionFactory(HttpConnectionFactory.class)).setHttpCompliance(HttpCompliance.RFC2616);
        String requests = "GET /?id=123\r\n\r\nGET /?id=456\r\n\r\n";
        LocalConnector.LocalEndPoint endp = this.connector.executeRequest(requests);
        String response = BufferUtil.toString((ByteBuffer)endp.waitForOutput(10L, TimeUnit.SECONDS));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"id=123"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.not((Matcher)Matchers.containsString((String)"id=456")));
    }

    @Test
    public void testHttp11ChunkedBodyTruncation() throws Exception {
        String request = "POST /?id=123 HTTP/1.1\r\nHost: local\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n1ff00000008\r\nabcdefgh\r\n\r\n0\r\n\r\nPOST /?id=bogus HTTP/1.1\r\nContent-Length: 5\r\nHost: dummy-host.example.com\r\n\r\n12345";
        String response = this.connector.getResponse(request);
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)" 200 OK"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"Connection: close"));
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"Early EOF"));
    }

    @Test
    public void testHttp11MultipleContentLength() throws Exception {
        HttpParser.LOG.info("badMessage: 400 Bad messages EXPECTED...", new Object[0]);
        int[][] contentLengths = new int[][]{{0, 8}, {8, 0}, {8, 8}, {0, 8, 0}, {1, 2, 3, 4, 5, 6, 7, 8}, {8, 2, 1}, {0, 0}, {8, 0, 8}, {-1, 8}, {8, -1}, {-1, 8, -1}, {-1, -1}, {8, -1, 8}};
        for (int x = 0; x < contentLengths.length; ++x) {
            StringBuilder request = new StringBuilder();
            request.append("POST /?id=").append(Integer.toString(x)).append(" HTTP/1.1\r\n");
            request.append("Host: local\r\n");
            int[] clen = contentLengths[x];
            for (int n = 0; n < clen.length; ++n) {
                request.append("Content-Length: ").append(Integer.toString(clen[n])).append("\r\n");
            }
            request.append("Content-Type: text/plain\r\n");
            request.append("Connection: close\r\n");
            request.append("\r\n");
            request.append("abcdefgh");
            String rawResponse = this.connector.getResponse(request.toString());
            HttpTester.Response response = HttpTester.parseResponse((String)rawResponse);
            MatcherAssert.assertThat((String)"Response.status", (Object)response.getStatus(), (Matcher)Matchers.is((Object)400));
        }
    }

    public static Stream<Arguments> http11ContentLengthAndChunkedData() {
        return Stream.of(Arguments.of((Object[])new Object[]{new int[]{-1, 8}}), Arguments.of((Object[])new Object[]{new int[]{8, -1}}), Arguments.of((Object[])new Object[]{new int[]{8, -1, 8}}), Arguments.of((Object[])new Object[]{new int[]{-2, 8}}), Arguments.of((Object[])new Object[]{new int[]{8, -2}}), Arguments.of((Object[])new Object[]{new int[]{8, -2, 8}}), Arguments.of((Object[])new Object[]{new int[]{-3, 8}}), Arguments.of((Object[])new Object[]{new int[]{8, -3}}), Arguments.of((Object[])new Object[]{new int[]{8, -3, 8}}), Arguments.of((Object[])new Object[]{new int[]{-4, 8}}), Arguments.of((Object[])new Object[]{new int[]{8, -4}}), Arguments.of((Object[])new Object[]{new int[]{8, -4, 8}}), Arguments.of((Object[])new Object[]{new int[]{8, -4, -1, -2, -3, 8}}));
    }

    @ParameterizedTest
    @MethodSource(value={"http11ContentLengthAndChunkedData"})
    public void testHttp11ContentLengthAndChunk(int[] contentLengths) throws Exception {
        HttpParser.LOG.info("badMessage: 400 Bad messages EXPECTED...", new Object[0]);
        StringBuilder request = new StringBuilder();
        request.append("POST / HTTP/1.1\r\n");
        request.append("Host: local\r\n");
        block6: for (int n = 0; n < contentLengths.length; ++n) {
            switch (contentLengths[n]) {
                case -1: {
                    request.append("Transfer-Encoding: chunked\r\n");
                    continue block6;
                }
                case -2: {
                    request.append("Transfer-Encoding: \"chunked\"\r\n");
                    continue block6;
                }
                case -3: {
                    request.append("Transfer-Encoding: 'chunked'\r\n");
                    continue block6;
                }
                case -4: {
                    request.append("Transfer-Encoding: bogus\r\n");
                    continue block6;
                }
                default: {
                    request.append("Content-Length: ").append(contentLengths[n]).append("\r\n");
                }
            }
        }
        request.append("Content-Type: text/plain\r\n");
        request.append("\r\n");
        request.append("8;\r\n");
        request.append("abcdefgh");
        request.append("\r\n0;\r\n\r\n");
        String rawResponse = this.connector.getResponse(request.toString());
        HttpTester.Response response = HttpTester.parseResponse((String)rawResponse);
        MatcherAssert.assertThat((String)"Response.status", (Object)response.getStatus(), (Matcher)Matchers.is((Object)400));
    }

    public static Stream<Arguments> http11TransferEncodingChunked() {
        return Stream.of(Arguments.of((Object[])new Object[]{Arrays.asList("chunked, ")}), Arguments.of((Object[])new Object[]{Arrays.asList(", chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("bogus, chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("'chunked', chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("identity, chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("identity", "chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("", "chunked")}));
    }

    @ParameterizedTest
    @MethodSource(value={"http11TransferEncodingChunked"})
    public void testHttp11TransferEncodingChunked(List<String> tokens) throws Exception {
        StringBuilder request = new StringBuilder();
        request.append("POST / HTTP/1.1\r\n");
        request.append("Host: local\r\n");
        tokens.forEach(token -> request.append("Transfer-Encoding: ").append((String)token).append("\r\n"));
        request.append("Content-Type: text/plain\r\n");
        request.append("\r\n");
        request.append("8;\r\n");
        request.append("abcdefgh");
        request.append("\r\n0;\r\n\r\n");
        System.out.println(request.toString());
        String rawResponse = this.connector.getResponse(request.toString());
        HttpTester.Response response = HttpTester.parseResponse((String)rawResponse);
        MatcherAssert.assertThat((String)("Response.status (" + response.getReason() + ")"), (Object)response.getStatus(), (Matcher)Matchers.is((Object)200));
    }

    public static Stream<Arguments> http11TransferEncodingInvalidChunked() {
        return Stream.of(Arguments.of((Object[])new Object[]{Arrays.asList("bogus", "identity")}), Arguments.of((Object[])new Object[]{Arrays.asList("bad")}), Arguments.of((Object[])new Object[]{Arrays.asList("identity")}), Arguments.of((Object[])new Object[]{Arrays.asList("'chunked'")}), Arguments.of((Object[])new Object[]{Arrays.asList("`chunked`")}), Arguments.of((Object[])new Object[]{Arrays.asList("[chunked]")}), Arguments.of((Object[])new Object[]{Arrays.asList("{chunked}")}), Arguments.of((Object[])new Object[]{Arrays.asList("\u201cchunked\u201d")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked, bogus")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked, 'chunked'")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked, identity")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked, identity, chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked", "identity")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked", "identity", "chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked", "chunked")}), Arguments.of((Object[])new Object[]{Arrays.asList("chunked, chunked")}));
    }

    @ParameterizedTest
    @MethodSource(value={"http11TransferEncodingInvalidChunked"})
    public void testHttp11TransferEncodingInvalidChunked(List<String> tokens) throws Exception {
        HttpParser.LOG.info("badMessage: 400 Bad messages EXPECTED...", new Object[0]);
        StringBuilder request = new StringBuilder();
        request.append("POST / HTTP/1.1\r\n");
        request.append("Host: local\r\n");
        tokens.forEach(token -> request.append("Transfer-Encoding: ").append((String)token).append("\r\n"));
        request.append("Content-Type: text/plain\r\n");
        request.append("\r\n");
        request.append("8;\r\n");
        request.append("abcdefgh");
        request.append("\r\n0;\r\n\r\n");
        System.out.println(request.toString());
        String rawResponse = this.connector.getResponse(request.toString());
        HttpTester.Response response = HttpTester.parseResponse((String)rawResponse);
        MatcherAssert.assertThat((String)"Response.status", (Object)response.getStatus(), (Matcher)Matchers.is((Object)400));
    }

    @Test
    public void testNoPath() throws Exception {
        String response = this.connector.getResponse("GET http://localhost:80 HTTP/1.1\r\nHost: localhost:80\r\nConnection: close\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkContains(response, offset, "pathInfo=/");
    }

    @Test
    public void testDate() throws Exception {
        String response = this.connector.getResponse("GET / HTTP/1.1\r\nHost: localhost:80\r\nConnection: close\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "Date: ");
        this.checkContains(response, offset, "pathInfo=/");
    }

    @Test
    public void testSetDate() throws Exception {
        String response = this.connector.getResponse("GET /?date=1+Jan+1970 HTTP/1.1\r\nHost: localhost:80\r\nConnection: close\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "Date: 1 Jan 1970");
        this.checkContains(response, offset, "pathInfo=/");
    }

    @Test
    public void testBadNoPath() throws Exception {
        String response = this.connector.getResponse("GET http://localhost:80/../cheat HTTP/1.1\r\nHost: localhost:80\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 400");
    }

    @Test
    public void testOKPathDotDotPath() throws Exception {
        String response = this.connector.getResponse("GET /ooops/../path HTTP/1.0\r\nHost: localhost:80\r\n\n");
        this.checkContains(response, 0, "HTTP/1.1 200 OK");
        this.checkContains(response, 0, "pathInfo=/path");
    }

    @Test
    public void testBadPathDotDotPath() throws Exception {
        String response = this.connector.getResponse("GET /ooops/../../path HTTP/1.0\r\nHost: localhost:80\r\n\n");
        this.checkContains(response, 0, "HTTP/1.1 400 ");
    }

    @Test
    public void testBadDotDotPath() throws Exception {
        String response = this.connector.getResponse("GET ../path HTTP/1.0\r\nHost: localhost:80\r\n\n");
        this.checkContains(response, 0, "HTTP/1.1 400 ");
    }

    @Test
    public void testBadSlashDotDotPath() throws Exception {
        String response = this.connector.getResponse("GET /../path HTTP/1.0\r\nHost: localhost:80\r\n\n");
        this.checkContains(response, 0, "HTTP/1.1 400 ");
    }

    @Test
    public void testEncodedBadDotDotPath() throws Exception {
        String response = this.connector.getResponse("GET %2e%2e/path HTTP/1.0\r\nHost: localhost:80\r\n\n");
        this.checkContains(response, 0, "HTTP/1.1 400 ");
    }

    @Test
    public void test09() throws Exception {
        ((HttpConnectionFactory)this.connector.getConnectionFactory(HttpConnectionFactory.class)).setHttpCompliance(HttpCompliance.RFC2616_LEGACY);
        LocalConnector.LocalEndPoint endp = this.connector.executeRequest("GET /R1\n");
        endp.waitUntilClosed();
        String response = BufferUtil.toString((ByteBuffer)endp.takeOutput());
        int offset = 0;
        this.checkNotContained(response, offset, "HTTP/1.1");
        this.checkNotContained(response, offset, "200");
        this.checkContains(response, offset, "pathInfo=/R1");
    }

    @Test
    public void testSimple() throws Exception {
        String response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkContains(response, offset, "/R1");
    }

    @Test
    public void testEmptyNotPersistent() throws Exception {
        String response = this.connector.getResponse("GET /R1?empty=true HTTP/1.0\r\nHost: localhost\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkNotContained(response, offset, "Content-Length");
        response = this.connector.getResponse("GET /R1?empty=true HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkContains(response, offset, "Connection: close");
        this.checkNotContained(response, offset, "Content-Length");
    }

    @Test
    public void testEmptyPersistent() throws Exception {
        String response = this.connector.getResponse("GET /R1?empty=true HTTP/1.0\r\nHost: localhost\r\nConnection: keep-alive\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkContains(response, offset, "Content-Length: 0");
        this.checkNotContained(response, offset, "Connection: close");
        response = this.connector.getResponse("GET /R1?empty=true HTTP/1.1\r\nHost: localhost\r\n\r\n");
        offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkContains(response, offset, "Content-Length: 0");
        this.checkNotContained(response, offset, "Connection: close");
    }

    @Test
    public void testEmptyChunk() throws Exception {
        String response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n0\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkContains(response, offset, "/R1");
    }

    @Test
    public void testChunk() throws Exception {
        String response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\nA\r\n0123456789\r\n0\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "/R1");
        this.checkContains(response, offset, "0123456789");
    }

    @Test
    public void testChunkTrailer() throws Exception {
        String response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\nA\r\n0123456789\r\n0\r\nTrailer: ignored\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "/R1");
        this.checkContains(response, offset, "0123456789");
    }

    @Test
    public void testChunkNoTrailer() throws Exception {
        String response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\n\r\nA\r\n0123456789\r\n0\r\n\r\n");
        int offset = 0;
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "/R1");
        this.checkContains(response, offset, "0123456789");
    }

    @Test
    public void testHead() throws Exception {
        String headLine;
        String postLine;
        String responsePOST = this.connector.getResponse("POST /R1 HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        String responseHEAD = this.connector.getResponse("HEAD /R1 HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        boolean postDate = false;
        HashSet<String> postHeaders = new HashSet<String>();
        try (BufferedReader in = new BufferedReader(new StringReader(responsePOST));){
            postLine = in.readLine();
            String line = in.readLine();
            while (line != null && line.length() > 0) {
                if (line.startsWith("Date:")) {
                    postDate = true;
                } else {
                    postHeaders.add(line);
                }
                line = in.readLine();
            }
        }
        boolean headDate = false;
        HashSet<String> headHeaders = new HashSet<String>();
        try (BufferedReader in = new BufferedReader(new StringReader(responseHEAD));){
            headLine = in.readLine();
            String line = in.readLine();
            while (line != null && line.length() > 0) {
                if (line.startsWith("Date:")) {
                    headDate = true;
                } else {
                    headHeaders.add(line);
                }
                line = in.readLine();
            }
        }
        MatcherAssert.assertThat((Object)postLine, (Matcher)Matchers.equalTo((Object)headLine));
        MatcherAssert.assertThat((Object)postDate, (Matcher)Matchers.equalTo((Object)headDate));
        Assertions.assertTrue((boolean)postHeaders.equals(headHeaders));
    }

    @Test
    public void testHeadChunked() throws Exception {
        String headLine;
        String postLine;
        String responsePOST = this.connector.getResponse("POST /R1?no-content-length=true HTTP/1.1\r\nHost: localhost\r\n\r\n", false, 1L, TimeUnit.SECONDS);
        String responseHEAD = this.connector.getResponse("HEAD /R1?no-content-length=true HTTP/1.1\r\nHost: localhost\r\n\r\n", true, 1L, TimeUnit.SECONDS);
        boolean postDate = false;
        HashSet<String> postHeaders = new HashSet<String>();
        try (BufferedReader in = new BufferedReader(new StringReader(responsePOST));){
            postLine = in.readLine();
            String line = in.readLine();
            while (line != null && line.length() > 0) {
                if (line.startsWith("Date:")) {
                    postDate = true;
                } else {
                    postHeaders.add(line);
                }
                line = in.readLine();
            }
        }
        boolean headDate = false;
        HashSet<String> headHeaders = new HashSet<String>();
        try (BufferedReader in = new BufferedReader(new StringReader(responseHEAD));){
            headLine = in.readLine();
            String line = in.readLine();
            while (line != null && line.length() > 0) {
                if (line.startsWith("Date:")) {
                    headDate = true;
                } else {
                    headHeaders.add(line);
                }
                line = in.readLine();
            }
        }
        MatcherAssert.assertThat((Object)postLine, (Matcher)Matchers.equalTo((Object)headLine));
        MatcherAssert.assertThat((Object)postDate, (Matcher)Matchers.equalTo((Object)headDate));
        Assertions.assertTrue((boolean)postHeaders.equals(headHeaders));
    }

    @Test
    public void testBadHostPort() throws Exception {
        Log.getLogger(HttpParser.class).info("badMessage: Number formate exception expected ...", new Object[0]);
        String response = this.connector.getResponse("GET http://localhost:EXPECTED_NUMBER_FORMAT_EXCEPTION/ HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 400");
    }

    @Test
    public void testNoHost() throws Exception {
        String response = this.connector.getResponse("GET / HTTP/1.1\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 400");
    }

    @Test
    public void testEmptyHost() throws Exception {
        String response = this.connector.getResponse("GET / HTTP/1.1\r\nHost:\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 200");
    }

    @Test
    public void testBadURIencoding() throws Exception {
        String response = this.connector.getResponse("GET /bad/encoding%x HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 400");
    }

    @Test
    public void testBadUTF8FallsbackTo8859() throws Exception {
        Log.getLogger(HttpParser.class).info("badMessage: bad encoding expected ...", new Object[0]);
        String response = this.connector.getResponse("GET /foo/bar%c0%00 HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 400");
        response = this.connector.getResponse("GET /bad/utf8%c1 HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 200");
    }

    @Test
    public void testAutoFlush() throws Exception {
        int offset = 0;
        String response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        this.checkNotContained(response, offset, "IgnoreMe");
        offset = this.checkContains(response, offset, "/R1");
        this.checkContains(response, offset, "12345");
    }

    @Test
    public void testEmptyFlush() throws Exception {
        this.server.stop();
        this.server.setHandler((Handler)new AbstractHandler(){

            public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
                response.setStatus(200);
                ServletOutputStream out = response.getOutputStream();
                out.flush();
                out.flush();
            }
        });
        this.server.start();
        String response = this.connector.getResponse("GET / HTTP/1.1\r\nHost: localhost\r\nConnection: close\r\n\r\n");
        MatcherAssert.assertThat((Object)response, (Matcher)Matchers.containsString((String)"200 OK"));
    }

    @Test
    public void testCharset() throws Exception {
        String response = null;
        try {
            int offset = 0;
            response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            offset = this.checkContains(response, offset, "HTTP/1.1 200");
            offset = this.checkContains(response, offset, "/R1");
            offset = this.checkContains(response, offset, "encoding=UTF-8");
            this.checkContains(response, offset, "12345");
            offset = 0;
            response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset =  iso-8859-1 ; other=value\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            offset = this.checkContains(response, offset, "HTTP/1.1 200");
            offset = this.checkContains(response, offset, "encoding=iso-8859-1");
            offset = this.checkContains(response, offset, "/R1");
            this.checkContains(response, offset, "12345");
            offset = 0;
            Log.getLogger(DumpHandler.class).info("Expecting java.io.UnsupportedEncodingException", new Object[0]);
            response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=unknown\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            offset = this.checkContains(response, offset, "HTTP/1.1 200");
            offset = this.checkContains(response, offset, "encoding=unknown");
            offset = this.checkContains(response, offset, "/R1");
            this.checkContains(response, offset, "UnsupportedEncodingException");
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testUnconsumed() throws Exception {
        int offset = 0;
        String requests = "GET /R1?read=4 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\n\r\n5;\r\n12345\r\n5;\r\n67890\r\n0;\r\n\r\nGET /R2 HTTP/1.1\r\nHost: localhost\r\nContent-Type: text/plain; charset=utf-8\r\nContent-Length: 10\r\nConnection: close\r\n\r\nabcdefghij\r\n";
        LocalConnector.LocalEndPoint endp = this.connector.executeRequest(requests);
        String response = endp.getResponse() + endp.getResponse();
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "pathInfo=/R1");
        offset = this.checkContains(response, offset, "1234");
        this.checkNotContained(response, offset, "56789");
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "pathInfo=/R2");
        offset = this.checkContains(response, offset, "encoding=UTF-8");
        this.checkContains(response, offset, "abcdefghij");
    }

    @Test
    public void testUnconsumedTimeout() throws Exception {
        this.connector.setIdleTimeout(500L);
        int offset = 0;
        String requests = "GET /R1?read=4 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\n\r\n5;\r\n12345\r\n";
        long start = TimeUnit.NANOSECONDS.toMillis(System.nanoTime());
        String response = this.connector.getResponse(requests, 2000L, TimeUnit.MILLISECONDS);
        MatcherAssert.assertThat((Object)(TimeUnit.NANOSECONDS.toMillis(System.nanoTime()) - start), (Matcher)Matchers.lessThanOrEqualTo((Comparable)Long.valueOf(2000L)));
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "pathInfo=/R1");
        offset = this.checkContains(response, offset, "1234");
        this.checkNotContained(response, offset, "56789");
    }

    @Test
    public void testUnconsumedErrorRead() throws Exception {
        int offset = 0;
        String requests = "GET /R1?read=1&error=499 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\n\r\n5;\r\n12345\r\n5;\r\n67890\r\n0;\r\n\r\nGET /R2 HTTP/1.1\r\nHost: localhost\r\nContent-Type: text/plain; charset=utf-8\r\nContent-Length: 10\r\nConnection: close\r\n\r\nabcdefghij\r\n";
        LocalConnector.LocalEndPoint endp = this.connector.executeRequest(requests);
        String response = endp.getResponse() + endp.getResponse();
        offset = this.checkContains(response, offset, "HTTP/1.1 499");
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "/R2");
        offset = this.checkContains(response, offset, "encoding=UTF-8");
        this.checkContains(response, offset, "abcdefghij");
    }

    @Test
    public void testUnconsumedErrorStream() throws Exception {
        int offset = 0;
        String requests = "GET /R1?error=599 HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: application/data; charset=utf-8\r\n\r\n5;\r\n12345\r\n5;\r\n67890\r\n0;\r\n\r\nGET /R2 HTTP/1.1\r\nHost: localhost\r\nContent-Type: text/plain; charset=utf-8\r\nContent-Length: 10\r\nConnection: close\r\n\r\nabcdefghij\r\n";
        LocalConnector.LocalEndPoint endp = this.connector.executeRequest(requests);
        String response = endp.getResponse() + endp.getResponse();
        offset = this.checkContains(response, offset, "HTTP/1.1 599");
        offset = this.checkContains(response, offset, "HTTP/1.1 200");
        offset = this.checkContains(response, offset, "/R2");
        offset = this.checkContains(response, offset, "encoding=UTF-8");
        this.checkContains(response, offset, "abcdefghij");
    }

    @Test
    public void testUnconsumedException() throws Exception {
        int offset = 0;
        String requests = "GET /R1?read=1&ISE=true HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\n\r\n5;\r\n12345\r\n5;\r\n67890\r\n0;\r\n\r\nGET /R2 HTTP/1.1\r\nHost: localhost\r\nContent-Type: text/plain; charset=utf-8\r\nContent-Length: 10\r\n\r\nabcdefghij\r\n";
        Logger logger = Log.getLogger(HttpChannel.class);
        try (StacklessLogging stackless = new StacklessLogging(new Logger[]{logger});){
            logger.info("EXPECTING: java.lang.IllegalStateException...", new Object[0]);
            String response = this.connector.getResponse(requests);
            offset = this.checkContains(response, offset, "HTTP/1.1 500");
            offset = this.checkContains(response, offset, "Connection: close");
            this.checkNotContained(response, offset, "HTTP/1.1 200");
        }
    }

    @Test
    public void testConnection() throws Exception {
        String response = null;
        try {
            int offset = 0;
            response = this.connector.getResponse("GET /R1 HTTP/1.1\r\nHost: localhost\r\nConnection: TE, close\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            this.checkContains(response, offset, "Connection: close");
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testOversizedBuffer() throws Exception {
        String response = null;
        try {
            int offset = 0;
            String cookie = "thisisastringthatshouldreachover1kbytes";
            for (int i = 0; i < 100; ++i) {
                cookie = cookie + "xxxxxxxxxxxx";
            }
            response = this.connector.getResponse("GET / HTTP/1.1\r\nHost: localhost\r\nCookie: " + cookie + "\r\n\r\n");
            this.checkContains(response, offset, "HTTP/1.1 431");
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testExcessiveHeader() throws Exception {
        int offset = 0;
        StringBuilder request = new StringBuilder();
        request.append("GET / HTTP/1.1\r\n");
        request.append("Host: localhost\r\n");
        request.append("Cookie: thisisastring\r\n");
        for (int i = 0; i < 1000; ++i) {
            request.append(String.format("X-Header-%04d: %08x\r\n", i, i));
        }
        request.append("\r\n");
        String response = this.connector.getResponse(request.toString());
        offset = this.checkContains(response, offset, "HTTP/1.1 431");
        this.checkContains(response, offset, "<h1>Bad Message 431</h1>");
    }

    @Test
    public void testOversizedResponse() throws Exception {
        String str = "thisisastringthatshouldreachover1kbytes-";
        for (int i = 0; i < 500; ++i) {
            str = str + "xxxxxxxxxxxx";
        }
        final String longstr = str;
        final CountDownLatch checkError = new CountDownLatch(1);
        this.server.stop();
        this.server.setHandler((Handler)new AbstractHandler(){

            public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
                baseRequest.setHandled(true);
                response.setHeader(HttpHeader.CONTENT_TYPE.toString(), MimeTypes.Type.TEXT_HTML.toString());
                response.setHeader("LongStr", longstr);
                PrintWriter writer = response.getWriter();
                writer.write("<html><h1>FOO</h1></html>");
                writer.flush();
                if (writer.checkError()) {
                    checkError.countDown();
                }
                response.flushBuffer();
            }
        });
        this.server.start();
        Logger logger = Log.getLogger(HttpChannel.class);
        String response = null;
        try (StacklessLogging stackless = new StacklessLogging(new Logger[]{logger});){
            logger.info("Expect IOException: Response header too large...", new Object[0]);
            response = this.connector.getResponse("GET / HTTP/1.1\r\nHost: localhost\r\n\r\n");
            this.checkContains(response, 0, "HTTP/1.1 500");
            Assertions.assertTrue((boolean)checkError.await(1L, TimeUnit.SECONDS));
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testAllowedLargeResponse() throws Exception {
        ((HttpConnectionFactory)this.connector.getBean(HttpConnectionFactory.class)).getHttpConfiguration().setResponseHeaderSize(16384);
        ((HttpConnectionFactory)this.connector.getBean(HttpConnectionFactory.class)).getHttpConfiguration().setOutputBufferSize(8192);
        byte[] bytes = new byte[12288];
        Arrays.fill(bytes, (byte)88);
        final String longstr = "thisisastringthatshouldreachover12kbytes-" + new String(bytes, StandardCharsets.ISO_8859_1) + "_Z_";
        final CountDownLatch checkError = new CountDownLatch(1);
        this.server.stop();
        this.server.setHandler((Handler)new AbstractHandler(){

            public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response) throws IOException, ServletException {
                baseRequest.setHandled(true);
                response.setHeader(HttpHeader.CONTENT_TYPE.toString(), MimeTypes.Type.TEXT_HTML.toString());
                response.setHeader("LongStr", longstr);
                PrintWriter writer = response.getWriter();
                writer.write("<html><h1>FOO</h1></html>");
                writer.flush();
                if (writer.checkError()) {
                    checkError.countDown();
                }
                response.flushBuffer();
            }
        });
        this.server.start();
        String response = null;
        response = this.connector.getResponse("GET / HTTP/1.1\r\nHost: localhost\r\n\r\n");
        this.checkContains(response, 0, "HTTP/1.1 200");
        this.checkContains(response, 0, "LongStr: thisisastringthatshouldreachover12kbytes");
        this.checkContains(response, 0, "XXX_Z_");
        MatcherAssert.assertThat((Object)checkError.getCount(), (Matcher)Matchers.is((Object)1L));
    }

    @Test
    public void testAsterisk() throws Exception {
        String response = null;
        try (StacklessLogging stackless = new StacklessLogging(new Logger[]{HttpParser.LOG});){
            int offset = 0;
            response = this.connector.getResponse("OPTIONS * HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            this.checkContains(response, offset, "HTTP/1.1 200");
            offset = 0;
            response = this.connector.getResponse("GET * HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            this.checkContains(response, offset, "HTTP/1.1 400");
            offset = 0;
            response = this.connector.getResponse("GET ** HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\nContent-Type: text/plain; charset=utf-8\r\nConnection: close\r\n\r\n5;\r\n12345\r\n0;\r\n\r\n");
            this.checkContains(response, offset, "HTTP/1.1 400 Bad Request");
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testCONNECT() throws Exception {
        String response = null;
        try {
            int offset = 0;
            response = this.connector.getResponse("CONNECT www.webtide.com:8080 HTTP/1.1\r\nHost: myproxy:8888\r\n\r\n", 200L, TimeUnit.MILLISECONDS);
            this.checkContains(response, offset, "HTTP/1.1 200");
        }
        catch (Exception e) {
            if (response != null) {
                System.err.println(response);
            }
            throw e;
        }
    }

    @Test
    public void testBytesIn() throws Exception {
        String chunk1 = "0123456789ABCDEF";
        String chunk2 = IntStream.range(0, 64).mapToObj(i -> chunk1).collect(Collectors.joining());
        final long dataLength = chunk1.length() + chunk2.length();
        this.server.stop();
        this.server.setHandler((Handler)new AbstractHandler(){

            public void handle(String target, Request jettyRequest, HttpServletRequest request, HttpServletResponse response) throws IOException {
                jettyRequest.setHandled(true);
                IO.copy((InputStream)request.getInputStream(), (OutputStream)IO.getNullStream());
                HttpConnection connection = HttpConnection.getCurrentConnection();
                long bytesIn = connection.getBytesIn();
                MatcherAssert.assertThat((Object)bytesIn, (Matcher)Matchers.greaterThan((Comparable)Long.valueOf(dataLength)));
            }
        });
        this.server.start();
        LocalConnector.LocalEndPoint localEndPoint = this.connector.executeRequest("POST / HTTP/1.1\r\nHost: localhost\r\nContent-Length: " + dataLength + "\r\n\r\n" + chunk1);
        Thread.sleep(500L);
        localEndPoint.addInput(chunk2);
        HttpTester.Response response = HttpTester.parseResponse((String)localEndPoint.getResponse());
        Assertions.assertEquals((int)response.getStatus(), (int)200);
        localEndPoint.close();
        localEndPoint = this.connector.executeRequest("POST / HTTP/1.1\r\nHost: localhost\r\nTransfer-Encoding: chunked\r\n\r\n" + Integer.toHexString(chunk1.length()) + "\r\n" + chunk1 + "\r\n");
        Thread.sleep(500L);
        localEndPoint.addInput("" + Integer.toHexString(chunk2.length()) + "\r\n" + chunk2 + "\r\n0\r\n\r\n");
        response = HttpTester.parseResponse((String)localEndPoint.getResponse());
        Assertions.assertEquals((int)response.getStatus(), (int)200);
        localEndPoint.close();
    }

    private int checkContains(String s, int offset, String c) {
        MatcherAssert.assertThat((Object)s.substring(offset), (Matcher)Matchers.containsString((String)c));
        return s.indexOf(c, offset);
    }

    private void checkNotContained(String s, int offset, String c) {
        MatcherAssert.assertThat((Object)s.substring(offset), (Matcher)Matchers.not((Matcher)Matchers.containsString((String)c)));
    }
}

