/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util.resource;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.channels.ReadableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.ProviderNotFoundException;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileTime;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.URIUtil;
import org.eclipse.jetty.util.resource.MountedPathResource;
import org.eclipse.jetty.util.resource.PathResource;
import org.eclipse.jetty.util.resource.ResourceCollection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Resource
implements Iterable<Resource> {
    private static final Logger LOG = LoggerFactory.getLogger(Resource.class);
    private static final LinkOption[] NO_FOLLOW_LINKS = new LinkOption[]{LinkOption.NOFOLLOW_LINKS};
    private static final LinkOption[] FOLLOW_LINKS = new LinkOption[0];

    public static ResourceCollection combine(List<Resource> resources) {
        if (resources == null || resources.isEmpty()) {
            throw new IllegalArgumentException("No resources");
        }
        return new ResourceCollection(resources);
    }

    public static ResourceCollection combine(Resource ... resources) {
        if (resources == null || resources.length == 0) {
            throw new IllegalArgumentException("No resources");
        }
        return new ResourceCollection(List.of(resources));
    }

    public static String dump(Resource resource) {
        if (resource == null) {
            return "null exists=false directory=false lm=-1";
        }
        return "%s exists=%b directory=%b lm=%d".formatted(resource.toString(), resource.exists(), resource.isDirectory(), resource.lastModified());
    }

    static Resource create(URI uri) {
        try {
            if (!uri.isAbsolute()) {
                uri = uri.toString().startsWith("/") ? new URI("file", uri.toString(), null) : Paths.get("", new String[0]).toAbsolutePath().toUri().resolve(uri);
                uri = URIUtil.correctFileURI(uri);
            }
            if (PathResource.ALLOWED_SCHEMES.contains(uri.getScheme())) {
                return new PathResource(uri);
            }
            return new MountedPathResource(uri);
        }
        catch (IOException | URISyntaxException | ProviderNotFoundException ex) {
            throw new IllegalArgumentException(ex);
        }
    }

    public abstract Path getPath();

    public abstract boolean isContainedIn(Resource var1);

    @Override
    public Iterator<Resource> iterator() {
        return List.of(this).iterator();
    }

    public boolean exists() {
        return Files.exists(this.getPath(), NO_FOLLOW_LINKS);
    }

    public boolean isDirectory() {
        return Files.isDirectory(this.getPath(), FOLLOW_LINKS);
    }

    public Instant lastModified() {
        Path path = this.getPath();
        if (path == null) {
            return Instant.EPOCH;
        }
        if (!Files.exists(path, new LinkOption[0])) {
            return Instant.EPOCH;
        }
        try {
            FileTime ft = Files.getLastModifiedTime(path, FOLLOW_LINKS);
            return ft.toInstant();
        }
        catch (IOException e) {
            LOG.trace("IGNORED", (Throwable)e);
            return Instant.EPOCH;
        }
    }

    public long length() {
        try {
            return Files.size(this.getPath());
        }
        catch (IOException e) {
            return 0L;
        }
    }

    public abstract URI getURI();

    public abstract String getName();

    public abstract String getFileName();

    public InputStream newInputStream() throws IOException {
        return Files.newInputStream(this.getPath(), StandardOpenOption.READ);
    }

    public ReadableByteChannel newReadableByteChannel() throws IOException {
        return Files.newByteChannel(this.getPath(), StandardOpenOption.READ);
    }

    public List<Resource> list() {
        return List.of();
    }

    public Resource resolve(String subUriPath) {
        if (URIUtil.isNotNormalWithinSelf(subUriPath)) {
            throw new IllegalArgumentException(subUriPath);
        }
        if ("/".equals(subUriPath)) {
            return this;
        }
        while (subUriPath.startsWith("/")) {
            subUriPath = subUriPath.substring(1);
        }
        URI uri = this.getURI();
        URI resolvedUri = URIUtil.addPath(uri, subUriPath);
        return Resource.create(resolvedUri);
    }

    public boolean isAlias() {
        return this.getAlias() != null;
    }

    public URI getAlias() {
        return null;
    }

    public void copyTo(Path destination) throws IOException {
        if (Files.exists(destination, new LinkOption[0])) {
            throw new IllegalArgumentException(destination + " exists");
        }
        Path src = this.getPath();
        if (src != null) {
            Files.copy(src, destination, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.COPY_ATTRIBUTES, StandardCopyOption.REPLACE_EXISTING);
            return;
        }
        try (InputStream in = this.newInputStream();
             OutputStream out = Files.newOutputStream(destination, new OpenOption[0]);){
            IO.copy(in, out);
        }
    }

    public String getWeakETag() {
        return this.getWeakETag("");
    }

    public String getWeakETag(String suffix) {
        StringBuilder b = new StringBuilder(32);
        b.append("W/\"");
        String name = this.getName();
        int length = name.length();
        long lhash = 0L;
        for (int i = 0; i < length; ++i) {
            lhash = 31L * lhash + (long)name.charAt(i);
        }
        Base64.Encoder encoder = Base64.getEncoder().withoutPadding();
        b.append(encoder.encodeToString(Resource.longToBytes(this.lastModified().toEpochMilli() ^ lhash)));
        b.append(encoder.encodeToString(Resource.longToBytes(this.length() ^ lhash)));
        b.append(suffix);
        b.append('\"');
        return b.toString();
    }

    private static byte[] longToBytes(long value) {
        byte[] result = new byte[8];
        for (int i = 7; i >= 0; --i) {
            result[i] = (byte)(value & 0xFFL);
            value >>= 8;
        }
        return result;
    }

    public Collection<Resource> getAllResources() {
        try {
            ArrayList<Resource> deep = new ArrayList<Resource>();
            for (Resource r : this.list()) {
                if (r.isDirectory()) {
                    deep.addAll(r.getAllResources());
                    continue;
                }
                deep.add(r);
            }
            return deep;
        }
        catch (Exception e) {
            throw new IllegalStateException(e);
        }
    }
}

