/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.function.ToIntFunction;
import org.eclipse.jetty.util.IO;
import org.eclipse.jetty.util.annotation.ManagedAttribute;
import org.eclipse.jetty.util.annotation.ManagedObject;
import org.eclipse.jetty.util.component.Dumpable;
import org.eclipse.jetty.util.component.DumpableCollection;
import org.eclipse.jetty.util.thread.AutoLock;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ManagedObject
public class Pool<T>
implements AutoCloseable,
Dumpable {
    private static final Logger LOGGER = LoggerFactory.getLogger(Pool.class);
    private final List<Entry> entries = new CopyOnWriteArrayList<Entry>();
    private final int maxEntries;
    private final StrategyType strategyType;
    private final AutoLock lock = new AutoLock();
    private final ThreadLocal<Entry> cache;
    private final AtomicInteger nextIndex;
    private final ToIntFunction<T> maxMultiplex;
    private volatile boolean closed;

    public Pool(StrategyType strategyType, int maxEntries) {
        this(strategyType, maxEntries, false);
    }

    public Pool(StrategyType strategyType, int maxEntries, boolean cache) {
        this(strategyType, maxEntries, cache, pooled -> 1);
    }

    public Pool(StrategyType strategyType, int maxEntries, boolean cache, ToIntFunction<T> maxMultiplex) {
        this.maxEntries = maxEntries;
        this.strategyType = Objects.requireNonNull(strategyType);
        this.cache = cache ? new ThreadLocal() : null;
        this.nextIndex = strategyType == StrategyType.ROUND_ROBIN ? new AtomicInteger() : null;
        this.maxMultiplex = Objects.requireNonNull(maxMultiplex);
    }

    @ManagedAttribute(value="The number of reserved entries")
    public int getReservedCount() {
        return (int)this.entries.stream().filter(Entry::isReserved).count();
    }

    @ManagedAttribute(value="The number of idle entries")
    public int getIdleCount() {
        return (int)this.entries.stream().filter(Entry::isIdle).count();
    }

    @ManagedAttribute(value="The number of in-use entries")
    public int getInUseCount() {
        return (int)this.entries.stream().filter(Entry::isInUse).count();
    }

    @ManagedAttribute(value="The number of closed entries")
    public int getClosedCount() {
        return (int)this.entries.stream().filter(Entry::isClosed).count();
    }

    @ManagedAttribute(value="The maximum number of entries")
    public int getMaxEntries() {
        return this.maxEntries;
    }

    private int getMaxMultiplex(T pooled) {
        return this.maxMultiplex.applyAsInt(pooled);
    }

    public Entry reserve() {
        try (AutoLock ignored = this.lock.lock();){
            if (this.closed) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("{} is closed, returning null reserved entry", (Object)this);
                }
                Entry entry = null;
                return entry;
            }
            int entriesSize = this.entries.size();
            if (this.maxEntries > 0 && entriesSize >= this.maxEntries) {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("{} has no space: {} >= {}, returning null reserved entry", new Object[]{this, entriesSize, this.maxEntries});
                }
                Entry entry = null;
                return entry;
            }
            Entry entry = new Entry();
            this.entries.add(entry);
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("{} returning new reserved entry {}", (Object)this, (Object)entry);
            }
            Entry entry2 = entry;
            return entry2;
        }
    }

    public Entry acquire() {
        Entry entry;
        if (this.closed) {
            return null;
        }
        int size = this.entries.size();
        if (size == 0) {
            return null;
        }
        if (this.cache != null && (entry = this.cache.get()) != null && entry.tryAcquire()) {
            return entry;
        }
        int index = this.startIndex(size);
        int tries = size;
        while (tries-- > 0) {
            try {
                Entry entry2 = this.entries.get(index);
                if (entry2 != null && entry2.tryAcquire()) {
                    return entry2;
                }
            }
            catch (IndexOutOfBoundsException e) {
                LOGGER.trace("IGNORED", (Throwable)e);
                size = this.entries.size();
                if (size == 0) break;
            }
            index = (index + 1) % size;
        }
        return null;
    }

    private int startIndex(int size) {
        return switch (this.strategyType) {
            default -> throw new IncompatibleClassChangeError();
            case StrategyType.FIRST -> 0;
            case StrategyType.RANDOM -> ThreadLocalRandom.current().nextInt(size);
            case StrategyType.ROUND_ROBIN -> this.nextIndex.getAndUpdate(c -> Math.max(0, c + 1)) % size;
            case StrategyType.THREAD_ID -> (int)(Thread.currentThread().getId() % (long)size);
        };
    }

    public Entry acquire(Function<Entry, T> creator) {
        T value;
        Entry entry = this.acquire();
        if (entry != null) {
            return entry;
        }
        entry = this.reserve();
        if (entry == null) {
            return null;
        }
        try {
            value = creator.apply(entry);
        }
        catch (Throwable th) {
            this.remove(entry);
            throw th;
        }
        if (value == null) {
            this.remove(entry);
            return null;
        }
        return entry.enable(value, true) ? entry : null;
    }

    private boolean release(Entry entry) {
        if (this.closed) {
            return false;
        }
        boolean released = entry.tryRelease();
        if (released && this.cache != null) {
            this.cache.set(entry);
        }
        return released;
    }

    private boolean remove(Entry entry) {
        if (this.closed) {
            return false;
        }
        if (!entry.tryRemove()) {
            if (LOGGER.isDebugEnabled()) {
                LOGGER.debug("Attempt to remove an object from the pool that is still in use: {}", (Object)entry);
            }
            return false;
        }
        boolean removed = this.entries.remove(entry);
        if (!removed && LOGGER.isDebugEnabled()) {
            LOGGER.debug("Attempt to remove an object from the pool that does not exist: {}", (Object)entry);
        }
        return removed;
    }

    public boolean isClosed() {
        return this.closed;
    }

    @Override
    public void close() {
        ArrayList<Entry> copy;
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Closing {}", (Object)this);
        }
        try (AutoLock ignored = this.lock.lock();){
            this.closed = true;
            copy = new ArrayList<Entry>(this.entries);
            this.entries.clear();
        }
        for (Entry entry : copy) {
            boolean removed = entry.tryRemove();
            if (removed) {
                Object t = entry.pooled;
                if (!(t instanceof Closeable)) continue;
                Closeable closeable = (Closeable)t;
                IO.close(closeable);
                continue;
            }
            if (!LOGGER.isDebugEnabled()) continue;
            LOGGER.debug("Pooled object still in use: {}", (Object)entry);
        }
    }

    @ManagedAttribute(value="The number of entries")
    public int size() {
        return this.entries.size();
    }

    public Collection<Entry> values() {
        return Collections.unmodifiableCollection(this.entries);
    }

    @Override
    public void dump(Appendable out, String indent) throws IOException {
        Dumpable.dumpObjects(out, indent, this, new DumpableCollection("entries", this.entries));
    }

    public String toString() {
        return String.format("%s@%x[inUse=%d,size=%d,max=%d,closed=%b]", this.getClass().getSimpleName(), this.hashCode(), this.getInUseCount(), this.size(), this.getMaxEntries(), this.isClosed());
    }

    public static enum StrategyType {
        FIRST,
        RANDOM,
        THREAD_ID,
        ROUND_ROBIN;

    }

    public class Entry {
        private final AtomicInteger state = new AtomicInteger(Integer.MIN_VALUE);
        private T pooled;

        public boolean enable(T pooled, boolean acquire) {
            Objects.requireNonNull(pooled);
            if (!this.isReserved()) {
                if (this.isClosed()) {
                    return false;
                }
                throw new IllegalStateException("Entry already enabled: " + this);
            }
            this.pooled = pooled;
            if (this.tryEnable(acquire)) {
                return true;
            }
            this.pooled = null;
            if (this.isClosed()) {
                return false;
            }
            throw new IllegalStateException("Entry already enabled: " + this);
        }

        public T getPooled() {
            return this.pooled;
        }

        public boolean release() {
            return Pool.this.release(this);
        }

        public boolean remove() {
            return Pool.this.remove(this);
        }

        private boolean tryEnable(boolean acquire) {
            return this.state.compareAndSet(Integer.MIN_VALUE, acquire ? 1 : 0);
        }

        private boolean tryAcquire() {
            int newMultiplexCount;
            int multiplexCount;
            do {
                if ((multiplexCount = this.state.get()) < 0) {
                    return false;
                }
                int maxMultiplexed = Pool.this.getMaxMultiplex(this.pooled);
                if (maxMultiplexed > 0 && multiplexCount >= maxMultiplexed) {
                    return false;
                }
                newMultiplexCount = multiplexCount + 1;
                if (newMultiplexCount >= 0) continue;
                return false;
            } while (!this.state.compareAndSet(multiplexCount, newMultiplexCount));
            return true;
        }

        private boolean tryRelease() {
            int newMultiplexCount;
            int multiplexCount;
            do {
                if ((multiplexCount = this.state.get()) < 0) {
                    return false;
                }
                newMultiplexCount = multiplexCount - 1;
                if (newMultiplexCount >= 0) continue;
                return false;
            } while (!this.state.compareAndSet(multiplexCount, newMultiplexCount));
            return true;
        }

        private boolean tryRemove() {
            int newMultiplexCount;
            int multiplexCount;
            do {
                if ((multiplexCount = this.state.get()) == Integer.MIN_VALUE || multiplexCount == 0) {
                    newMultiplexCount = -1;
                    continue;
                }
                if (multiplexCount > 0) {
                    newMultiplexCount = -multiplexCount;
                    continue;
                }
                if (multiplexCount == -1) {
                    return false;
                }
                newMultiplexCount = multiplexCount + 1;
            } while (!this.state.compareAndSet(multiplexCount, newMultiplexCount));
            return newMultiplexCount == -1;
        }

        public boolean isClosed() {
            int s = this.state.get();
            return s < 0 && s != Integer.MIN_VALUE;
        }

        private boolean isReserved() {
            return this.state.get() == Integer.MIN_VALUE;
        }

        public boolean isIdle() {
            return this.state.get() == 0;
        }

        public boolean isInUse() {
            return this.state.get() > 0;
        }

        public String toString() {
            return String.format("%s@%x{multiplex=%d,pooled=%s}", this.getClass().getSimpleName(), this.hashCode(), this.state.get(), this.getPooled());
        }
    }
}

