/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.websocket.common.test;

import java.io.IOException;
import java.net.SocketException;
import java.nio.ByteBuffer;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.eclipse.jetty.util.BufferUtil;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;
import org.eclipse.jetty.websocket.api.extensions.Frame;
import org.eclipse.jetty.websocket.common.CloseInfo;
import org.eclipse.jetty.websocket.common.Generator;
import org.eclipse.jetty.websocket.common.OpCode;
import org.eclipse.jetty.websocket.common.WebSocketFrame;
import org.eclipse.jetty.websocket.common.test.BlockheadClient;
import org.eclipse.jetty.websocket.common.test.BlockheadClientRequest;
import org.eclipse.jetty.websocket.common.test.BlockheadConnection;
import org.eclipse.jetty.websocket.common.test.ByteBufferAssert;
import org.eclipse.jetty.websocket.common.test.Fuzzed;
import org.eclipse.jetty.websocket.common.test.Timeouts;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;

public class Fuzzer
implements AutoCloseable {
    private static final int KBYTE = 1024;
    private static final int MBYTE = 0x100000;
    private static final Logger LOG = Log.getLogger(Fuzzer.class);
    protected static final byte[] MASK = new byte[]{17, 34, 51, 68};
    private final Fuzzed testcase;
    private final BlockheadClient client;
    private final Generator generator;
    private BlockheadConnection clientConnection;
    private SendMode sendMode = SendMode.BULK;
    private int slowSendSegmentSize = 5;

    public Fuzzer(Fuzzed testcase) throws Exception {
        this.testcase = testcase;
        this.client = new BlockheadClient();
        int bigMessageSize = 0x1400000;
        this.client.getPolicy().setMaxTextMessageSize(bigMessageSize);
        this.client.getPolicy().setMaxBinaryMessageSize(bigMessageSize);
        this.client.getPolicy().setIdleTimeout(5000L);
        this.client.setIdleTimeout(TimeUnit.SECONDS.toMillis(2L));
        this.client.start();
        this.generator = testcase.getLaxGenerator();
    }

    public ByteBuffer asNetworkBuffer(List<WebSocketFrame> send) {
        int buflen = 0;
        for (Frame frame : send) {
            buflen += frame.getPayloadLength() + 28;
        }
        ByteBuffer buf = ByteBuffer.allocate(buflen);
        for (WebSocketFrame f : send) {
            this.setClientMask(f);
            this.generator.generateWholeFrame((Frame)f, buf);
        }
        buf.flip();
        return buf;
    }

    @Override
    public void close() {
        this.clientConnection.close();
        try {
            this.client.stop();
        }
        catch (Exception ignore) {
            LOG.ignore((Throwable)ignore);
        }
    }

    public void disconnect() {
        this.clientConnection.abort();
    }

    public void connect() throws IOException {
        BlockheadClientRequest request = this.client.newWsRequest(this.testcase.getServerURI());
        request.idleTimeout(2L, TimeUnit.SECONDS);
        CompletableFuture<BlockheadConnection> connFut = request.sendAsync();
        try {
            this.clientConnection = (BlockheadConnection)((Object)connFut.get(2L, Timeouts.CONNECT_UNIT));
        }
        catch (InterruptedException e) {
            throw new IOException("Connect interrupted", e);
        }
        catch (ExecutionException e) {
            throw new IOException("Connect execution failed", e);
        }
        catch (TimeoutException e) {
            throw new IOException("Connect timed out", e);
        }
    }

    public void expect(List<WebSocketFrame> expect) throws Exception {
        this.expect(expect, 2L, Timeouts.POLL_EVENT_UNIT);
    }

    public void expect(List<WebSocketFrame> expect, long duration, TimeUnit unit) throws Exception {
        int expectedCount = expect.size();
        LOG.debug("expect() {} frame(s)", (long)expect.size());
        LinkedBlockingQueue<WebSocketFrame> frames = this.clientConnection.getFrameQueue();
        String prefix = "";
        for (int i = 0; i < expectedCount; ++i) {
            WebSocketFrame expected = expect.get(i);
            WebSocketFrame actual = frames.poll(duration, unit);
            prefix = "Frame[" + i + "]";
            LOG.debug("{} {}", new Object[]{prefix, actual});
            MatcherAssert.assertThat((String)prefix, (Object)actual, (Matcher)Matchers.is((Matcher)Matchers.notNullValue()));
            MatcherAssert.assertThat((String)(prefix + ".opcode"), (Object)OpCode.name((byte)actual.getOpCode()), (Matcher)Matchers.is((Object)OpCode.name((byte)expected.getOpCode())));
            prefix = prefix + "/" + actual.getOpCode();
            if (expected.getOpCode() == 8) {
                CloseInfo expectedClose = new CloseInfo((Frame)expected);
                CloseInfo actualClose = new CloseInfo((Frame)actual);
                MatcherAssert.assertThat((String)(prefix + ".statusCode"), (Object)actualClose.getStatusCode(), (Matcher)Matchers.is((Object)expectedClose.getStatusCode()));
                continue;
            }
            MatcherAssert.assertThat((String)(prefix + ".payloadLength"), (Object)actual.getPayloadLength(), (Matcher)Matchers.is((Object)expected.getPayloadLength()));
            ByteBufferAssert.assertEquals(prefix + ".payload", expected.getPayload(), actual.getPayload());
        }
    }

    public void expect(WebSocketFrame expect) throws Exception {
        this.expect(Collections.singletonList(expect));
    }

    public void expectNoMoreFrames() {
    }

    public SendMode getSendMode() {
        return this.sendMode;
    }

    public int getSlowSendSegmentSize() {
        return this.slowSendSegmentSize;
    }

    public void send(ByteBuffer buf) throws IOException {
        MatcherAssert.assertThat((String)"Client connected", (Object)this.clientConnection.isOpen(), (Matcher)Matchers.is((Object)true));
        LOG.debug("Sending bytes {}", new Object[]{BufferUtil.toDetailString((ByteBuffer)buf)});
        if (this.sendMode == SendMode.SLOW) {
            this.clientConnection.writeRawSlowly(buf, this.slowSendSegmentSize);
        } else {
            this.clientConnection.writeRaw(buf);
        }
    }

    public void send(ByteBuffer buf, int numBytes) throws IOException {
        this.clientConnection.writeRaw(buf, numBytes);
    }

    public void send(List<WebSocketFrame> send) throws IOException {
        MatcherAssert.assertThat((String)"Client connected", (Object)this.clientConnection.isOpen(), (Matcher)Matchers.is((Object)true));
        LOG.debug("Sending {} frames (mode {})", new Object[]{send.size(), this.sendMode});
        if (this.sendMode == SendMode.BULK || this.sendMode == SendMode.SLOW) {
            int buflen = 0;
            for (Frame frame : send) {
                buflen += frame.getPayloadLength() + 28;
            }
            ByteBuffer buf = ByteBuffer.allocate(buflen);
            for (WebSocketFrame f : send) {
                this.setClientMask(f);
                buf.put(this.generator.generateHeaderBytes((Frame)f));
                if (!f.hasPayload()) continue;
                buf.put(f.getPayload());
            }
            BufferUtil.flipToFlush((ByteBuffer)buf, (int)0);
            switch (this.sendMode) {
                case BULK: {
                    this.clientConnection.writeRaw(buf);
                    break;
                }
                case SLOW: {
                    this.clientConnection.writeRawSlowly(buf, this.slowSendSegmentSize);
                    break;
                }
                default: {
                    throw new RuntimeException("Whoops, unsupported sendMode: " + (Object)((Object)this.sendMode));
                }
            }
        } else if (this.sendMode == SendMode.PER_FRAME) {
            for (WebSocketFrame f : send) {
                f.setMask(MASK);
                ByteBuffer byteBuffer = ByteBuffer.allocate(f.getPayloadLength() + 28);
                BufferUtil.clearToFill((ByteBuffer)byteBuffer);
                this.generator.generateWholeFrame((Frame)f, byteBuffer);
                BufferUtil.flipToFlush((ByteBuffer)byteBuffer, (int)0);
                this.clientConnection.writeRaw(byteBuffer);
            }
        }
    }

    public void send(WebSocketFrame send) throws IOException {
        this.send(Collections.singletonList(send));
    }

    public void sendAndIgnoreBrokenPipe(List<WebSocketFrame> send) throws IOException {
        try {
            this.send(send);
        }
        catch (SocketException ignore) {
            MatcherAssert.assertThat((String)"Allowed to be a broken pipe", (Object)ignore.getMessage().toLowerCase(Locale.ENGLISH), (Matcher)Matchers.containsString((String)"broken pipe"));
        }
    }

    private void setClientMask(WebSocketFrame f) {
        if (LOG.isDebugEnabled()) {
            f.setMask(new byte[]{0, 0, 0, 0});
        } else {
            f.setMask(MASK);
        }
    }

    public void setSendMode(SendMode sendMode) {
        this.sendMode = sendMode;
    }

    public void setSlowSendSegmentSize(int segmentSize) {
        this.slowSendSegmentSize = segmentSize;
    }

    public static enum SendMode {
        BULK,
        PER_FRAME,
        SLOW;

    }
}

