/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jkube.kit.build.service.docker;

import com.google.common.collect.ImmutableMap;
import com.google.gson.JsonObject;
import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import org.eclipse.jkube.kit.build.api.helper.DockerFileUtil;
import org.eclipse.jkube.kit.build.service.docker.ArchiveService;
import org.eclipse.jkube.kit.build.service.docker.ImagePullManager;
import org.eclipse.jkube.kit.build.service.docker.QueryService;
import org.eclipse.jkube.kit.build.service.docker.RegistryService;
import org.eclipse.jkube.kit.build.service.docker.access.BuildOptions;
import org.eclipse.jkube.kit.build.service.docker.access.DockerAccess;
import org.eclipse.jkube.kit.build.service.docker.access.DockerAccessException;
import org.eclipse.jkube.kit.common.AssemblyConfiguration;
import org.eclipse.jkube.kit.common.KitLogger;
import org.eclipse.jkube.kit.common.util.EnvUtil;
import org.eclipse.jkube.kit.config.image.ImageConfiguration;
import org.eclipse.jkube.kit.config.image.ImageName;
import org.eclipse.jkube.kit.config.image.build.BuildConfiguration;
import org.eclipse.jkube.kit.config.image.build.CleanupMode;
import org.eclipse.jkube.kit.config.image.build.JKubeConfiguration;

public class BuildService {
    private static final String ARG_PREFIX = "docker.buildArg.";
    private final DockerAccess docker;
    private final QueryService queryService;
    private final ArchiveService archiveService;
    private final RegistryService registryService;
    private final KitLogger log;

    BuildService(DockerAccess docker, QueryService queryService, RegistryService registryService, ArchiveService archiveService, KitLogger log) {
        this.docker = docker;
        this.queryService = queryService;
        this.registryService = registryService;
        this.archiveService = archiveService;
        this.log = log;
    }

    public void buildImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, JKubeConfiguration configuration) throws IOException {
        if (imagePullManager != null) {
            this.autoPullBaseImage(imageConfig, imagePullManager, configuration);
        }
        this.buildImage(imageConfig, configuration, this.checkForNocache(imageConfig), this.addBuildArgs(configuration));
    }

    public void tagImage(String imageName, ImageConfiguration imageConfig) throws DockerAccessException {
        List tags = imageConfig.getBuildConfiguration().getTags();
        if (!tags.isEmpty()) {
            this.log.info("%s: Tag with %s", new Object[]{imageConfig.getDescription(), EnvUtil.stringJoin((List)tags, (String)",")});
            for (String tag : tags) {
                if (tag == null) continue;
                this.docker.tag(imageName, new ImageName(imageName, tag).getFullName(), true);
            }
            this.log.debug("Tagging image successful!", new Object[0]);
        }
    }

    protected void buildImage(ImageConfiguration imageConfig, JKubeConfiguration params, boolean noCache, Map<String, String> buildArgs) throws IOException {
        String imageName = imageConfig.getName();
        ImageName.validate((String)imageName);
        BuildConfiguration buildConfig = imageConfig.getBuildConfiguration();
        String oldImageId = null;
        CleanupMode cleanupMode = buildConfig.cleanupMode();
        if (cleanupMode.isRemove()) {
            oldImageId = this.queryService.getImageId(imageName);
        }
        long time = System.currentTimeMillis();
        if (buildConfig.getDockerArchive() != null) {
            this.docker.loadImage(imageName, buildConfig.getAbsoluteDockerTarPath(params.getSourceDirectory(), params.getProject().getBaseDirectory() != null ? params.getProject().getBaseDirectory().toString() : null));
            this.log.info("%s: Loaded tarball in %s", new Object[]{buildConfig.getDockerArchive(), EnvUtil.formatDurationTill((long)time)});
            return;
        }
        File dockerArchive = this.archiveService.createArchive(imageName, buildConfig, params, this.log);
        this.log.info("%s: Created %s in %s", new Object[]{imageConfig.getDescription(), dockerArchive.getName(), EnvUtil.formatDurationTill((long)time)});
        Map<String, String> mergedBuildMap = this.prepareBuildArgs(buildArgs, buildConfig);
        BuildOptions opts = new BuildOptions(buildConfig.getBuildOptions()).dockerfile(this.getDockerfileName(buildConfig)).forceRemove(cleanupMode.isRemove()).noCache(noCache).cacheFrom(buildConfig.getCacheFrom()).buildArgs(mergedBuildMap);
        String newImageId = this.doBuildImage(imageName, dockerArchive, opts);
        this.log.info("%s: Built image %s", new Object[]{imageConfig.getDescription(), newImageId});
        if (oldImageId != null && !oldImageId.equals(newImageId)) {
            try {
                this.docker.removeImage(oldImageId, true);
                this.log.info("%s: Removed old image %s", new Object[]{imageConfig.getDescription(), oldImageId});
            }
            catch (DockerAccessException exp) {
                if (cleanupMode == CleanupMode.TRY_TO_REMOVE) {
                    this.log.warn("%s: %s (old image)%s", new Object[]{imageConfig.getDescription(), exp.getMessage(), exp.getCause() != null ? " [" + exp.getCause().getMessage() + "]" : ""});
                }
                throw exp;
            }
        }
    }

    private Map<String, String> prepareBuildArgs(Map<String, String> buildArgs, BuildConfiguration buildConfig) {
        ImmutableMap.Builder builder = ImmutableMap.builder().putAll(buildArgs);
        if (buildConfig.getArgs() != null) {
            builder.putAll(buildConfig.getArgs());
        }
        return builder.build();
    }

    private String getDockerfileName(BuildConfiguration buildConfig) {
        if (buildConfig.isDockerFileMode()) {
            return buildConfig.getDockerFile().getName();
        }
        return null;
    }

    private String doBuildImage(String imageName, File dockerArchive, BuildOptions options) throws DockerAccessException {
        this.docker.buildImage(imageName, dockerArchive, options);
        return this.queryService.getImageId(imageName);
    }

    private Map<String, String> addBuildArgs(JKubeConfiguration configuration) {
        Map<String, String> buildArgsFromProject = this.addBuildArgsFromProperties(configuration.getProject().getProperties());
        Map<String, String> buildArgsFromSystem = this.addBuildArgsFromProperties(System.getProperties());
        Map<String, String> buildArgsFromDockerConfig = this.addBuildArgsFromDockerConfig();
        return ImmutableMap.builder().putAll(buildArgsFromDockerConfig).putAll(Optional.ofNullable(configuration.getBuildArgs()).orElse(Collections.emptyMap())).putAll(buildArgsFromProject).putAll(buildArgsFromSystem).build();
    }

    private Map<String, String> addBuildArgsFromProperties(Properties properties) {
        HashMap<String, String> buildArgs = new HashMap<String, String>();
        for (Object keyObj : properties.keySet()) {
            String key = (String)keyObj;
            if (!key.startsWith(ARG_PREFIX)) continue;
            String argKey = key.replaceFirst(ARG_PREFIX, "");
            String value = properties.getProperty(key);
            if (this.isEmpty(value)) continue;
            buildArgs.put(argKey, value);
        }
        this.log.debug("Build args set %s", new Object[]{buildArgs});
        return buildArgs;
    }

    private Map<String, String> addBuildArgsFromDockerConfig() {
        JsonObject proxies;
        JsonObject dockerConfig = DockerFileUtil.readDockerConfig();
        if (dockerConfig == null) {
            return Collections.emptyMap();
        }
        HashMap<String, String> buildArgs = new HashMap<String, String>();
        if (dockerConfig.has("proxies") && (proxies = dockerConfig.getAsJsonObject("proxies")).has("default")) {
            JsonObject defaultProxyObj = proxies.getAsJsonObject("default");
            String[] proxyMapping = new String[]{"httpProxy", "http_proxy", "httpsProxy", "https_proxy", "noProxy", "no_proxy", "ftpProxy", "ftp_proxy"};
            for (int index = 0; index < proxyMapping.length; index += 2) {
                if (!defaultProxyObj.has(proxyMapping[index])) continue;
                buildArgs.put(ARG_PREFIX + proxyMapping[index + 1], defaultProxyObj.get(proxyMapping[index]).getAsString());
            }
        }
        this.log.debug("Build args set %s", new Object[]{buildArgs});
        return buildArgs;
    }

    private void autoPullBaseImage(ImageConfiguration imageConfig, ImagePullManager imagePullManager, JKubeConfiguration configuration) throws IOException {
        List<Object> fromImages;
        BuildConfiguration buildConfig = imageConfig.getBuildConfiguration();
        if (buildConfig.getDockerArchive() != null) {
            return;
        }
        if (buildConfig.isDockerFileMode()) {
            fromImages = this.extractBaseFromDockerfile(buildConfig, configuration);
        } else {
            fromImages = new LinkedList();
            String baseImage = this.extractBaseFromConfiguration(buildConfig);
            if (baseImage != null) {
                fromImages.add(this.extractBaseFromConfiguration(buildConfig));
            }
        }
        for (String string : fromImages) {
            if (string == null || "scratch".equals(string)) continue;
            this.registryService.pullImageWithPolicy(string, imagePullManager, configuration.getRegistryConfig(), this.queryService.hasImage(string));
        }
    }

    private String extractBaseFromConfiguration(BuildConfiguration buildConfig) {
        AssemblyConfiguration assemblyConfig;
        String fromImage = buildConfig.getFrom();
        if (fromImage == null && (assemblyConfig = buildConfig.getAssembly()) == null) {
            fromImage = "busybox:latest";
        }
        return fromImage;
    }

    private List<String> extractBaseFromDockerfile(BuildConfiguration buildConfig, JKubeConfiguration configuration) {
        List fromImage;
        try {
            File fullDockerFilePath = buildConfig.getAbsoluteDockerFilePath(configuration.getSourceDirectory(), (String)Optional.ofNullable(configuration.getProject().getBaseDirectory()).map(File::toString).orElse(null));
            fromImage = DockerFileUtil.extractBaseImages((File)fullDockerFilePath, (Properties)configuration.getProperties(), (String)buildConfig.getFilter(), (Map)buildConfig.getArgs());
        }
        catch (IOException e) {
            return Collections.emptyList();
        }
        return fromImage;
    }

    private boolean checkForNocache(ImageConfiguration imageConfig) {
        String nocache = System.getProperty("docker.nocache");
        if (nocache != null) {
            return nocache.length() == 0 || Boolean.parseBoolean(nocache);
        }
        BuildConfiguration buildConfig = imageConfig.getBuildConfiguration();
        return buildConfig.nocache();
    }

    private boolean isEmpty(String str) {
        return str == null || str.isEmpty();
    }
}

