/*
 * Copyright (c) 1998, 2024 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     Oracle - initial API and implementation from Oracle TopLink
package org.eclipse.persistence.annotations;

import jakarta.persistence.Basic;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.ElementType.METHOD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * A Mutable annotation can be used on a {@linkplain Basic} mapping.
 * It can be used on complex field types to indicate that the value itself can be changed or not changed (instead of being replaced).
 * Most basic types are not mutable, i.e. int, long, float, double, String, BigDecimal.
 * Date or Calendar types are assumed not to be mutable by default, if it is desired to call
 * the set methods on the Date or Calendar, then the mapping must be set to be Mutable.
 * <p>
 * For Date and Calendar types the global persistence property "{@systemProperty eclipselink.temporal.mutable}" can also be set to "{@code true}".
 * <p>
 * For serialized types, by default they are assumed to be mutable, if they are not mutable this annotation can be set to "{@code false}".
 * <p>
 * Mutable basic mappings affect the overhead of change tracking, attribute change tracking can only be woven with non-mutable mappings.
 */
@Target({METHOD, FIELD})
@Retention(RUNTIME)
public @interface Mutable {
    /**
     * Set the mapping to be mutable, or not mutable.
     */
    boolean value() default true;
}
