/*
 * Copyright (c) 1998, 2024 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     James Sutherland - initial implementation
package org.eclipse.persistence.annotations;

import jakarta.persistence.Column;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * The PrimaryKey annotation allows advanced configuration of the Id.
 * <p>
 * A validation policy can be given that allows specifying if zero is a valid id value.
 * The set of primary key columns can also be specified precisely.
 *
 * @author James Sutherland
 * @since EclipseLink 1.1
 */
@Target({TYPE})
@Retention(RUNTIME)
public @interface PrimaryKey {
    /**
     * Configures what id validation is done.
     * <p>
     * By default, 0 is not a valid id value, this can be used to allow 0 id values.
     */
    IdValidation validation() default IdValidation.ZERO;

    /**
     * Configures what cache key type is used to store the object in the cache.
     * <p>
     * By default, the type is determined by what type is optimal for the class.
     */
    CacheKeyType cacheKeyType() default CacheKeyType.AUTO;

    /**
     * Used to specify the primary key columns directly.
     * <p>
     * This can be used instead of {@linkplain jakarta.persistence.Id} if the primary key includes a non-basic field,
     * such as a foreign key, or an inheritance discriminator, embedded, or transformation mapped field.
     */
    Column[] columns() default {};

}
