/*
 * Copyright (c) 2012, 2024 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     14/05/2012-2.4 Guy Pelletier
//       - 376603: Provide for table per tenant support for multitenant applications
package org.eclipse.persistence.annotations;

import java.lang.annotation.Retention;
import java.lang.annotation.Target;

import static java.lang.annotation.ElementType.TYPE;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

/**
 * <p>
 * A tenant table discriminator is used with a {@linkplain MultitenantType#TABLE_PER_TENANT} multitenant
 * strategy. The tenant table discriminator describes the type of table
 * discriminator to use. The user may choose their own tenant identifier
 * property or use the default property:
 * <p>
 * {@snippet :
 * org.eclipse.persistence.config.PersistenceUnitProperties.MULTITENANT_PROPERTY_DEFAULT = "eclipselink.tenant-id";
 * }
 * <p>
 * Tenant table discriminator can be specified at the Entity or MappedSuperclass
 * level and must always be accompanied with a {@code @Multitenant(TABLE_PER_TENANT)}
 * specification. It is not sufficient to specify only a tenant table discriminator.
 *
 * @see TenantTableDiscriminator
 * @see Multitenant
 * @see MultitenantType
 * @author Guy Pelletier
 * @since EclipseLink 2.4
 */
@Target({TYPE})
@Retention(RUNTIME)
public @interface TenantTableDiscriminator {
    /**
     * The name of the context property to apply to as tenant table discriminator.
     * <p>
     * Default is "{@code eclipselink-tenant.id}".
     */
    String contextProperty() default "eclipselink.tenant-id";

    /**
     * The type of tenant table discriminator to use with the tables
     * of the persistence unit.
     * <p>
     * Defaults to {@linkplain TenantTableDiscriminatorType#SUFFIX}.
     */
    TenantTableDiscriminatorType type() default TenantTableDiscriminatorType.SUFFIX;
}
