/*
 * Copyright (c) 2011, 2021 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0,
 * or the Eclipse Distribution License v. 1.0 which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: EPL-2.0 OR BSD-3-Clause
 */

// Contributors:
//     Oracle - initial API and implementation
//
package org.eclipse.persistence.jpa.jpql.tools;

import org.eclipse.persistence.jpa.jpql.parser.Expression;
import org.eclipse.persistence.jpa.jpql.parser.JPQLGrammar;
import org.eclipse.persistence.jpa.jpql.tools.resolver.DefaultResolverBuilder;

/**
 * This context is used to store information related to the JPQL query.
 *
 * <pre><code> {@link org.eclipse.persistence.jpa.jpql.tools.spi.IQuery IQuery} externalQuery = ...;
 *
 * JPQLQueryContext context = new JPQLQueryContext(DefaultJPQLGrammar.instance());
 * context.setQuery(query);</code></pre>
 *
 * If the JPQL query is already parsed, then the context can use it and it needs to be set before
 * setting the {@link org.eclipse.persistence.jpa.jpql.tools.spi.IQuery IQuery}:
 * <pre><code> {@link org.eclipse.persistence.jpa.jpql.parser.JPQLExpression JPQLExpression} jpqlExpression = ...;
 *
 * JPQLQueryContext context = new JPQLQueryContext(DefaultJPQLGrammar.instance());
 * context.setJPQLExpression(jpqlExpression);
 * context.setQuery(query);</code></pre>
 *
 * @version 2.4
 * @since 2.4
 * @author Pascal Filion
 */
public class DefaultJPQLQueryContext extends JPQLQueryContext {

    /**
     * Creates a new <code>DefaultJPQLQueryContext</code>.
     */
    public DefaultJPQLQueryContext(JPQLGrammar jpqlGrammar) {
        super(jpqlGrammar);
    }

    /**
     * Creates a new sub-<code>DefaultJPQLQueryContext</code>.
     *
     * @param parent The parent context
     * @param currentQuery The parsed tree representation of the subquery
     */
    protected DefaultJPQLQueryContext(JPQLQueryContext parent, Expression currentQuery) {
        super(parent, currentQuery);
    }

    @Override
    protected JPQLQueryContext buildJPQLQueryContext(JPQLQueryContext currentContext,
                                                     Expression currentQuery) {

        return new DefaultJPQLQueryContext(currentContext, currentQuery);
    }

    @Override
    protected DefaultLiteralVisitor buildLiteralVisitor() {
        return new DefaultLiteralVisitor();
    }

    @Override
    protected DefaultParameterTypeVisitor buildParameterTypeVisitor() {
        return new DefaultParameterTypeVisitor(this);
    }

    @Override
    protected DefaultResolverBuilder buildResolverBuilder() {
        return new DefaultResolverBuilder(this);
    }

    @Override
    public DefaultJPQLQueryContext getParent() {
        return (DefaultJPQLQueryContext) super.getParent();
    }
}
