/*******************************************************************************
 * Copyright (c) 2012, 2022 Sonatype Inc. and others.
 * This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *    Sonatype Inc. - initial API and implementation
 *    Christoph Läubrich - Add default excludes to JGitTimestampProvider #613 
 *******************************************************************************/
package org.eclipse.tycho.extras.buildtimestamp.jgit;

import java.io.IOException;
import java.util.List;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import org.eclipse.jgit.ignore.FastIgnoreRule;
import org.eclipse.jgit.lib.Constants;
import org.eclipse.jgit.treewalk.TreeWalk;
import org.eclipse.jgit.treewalk.filter.TreeFilter;

public class PathFilter extends TreeFilter {

	private static final Pattern FILTER_DELIMITER = Pattern.compile("[\n\r\f]");
	private final byte[] basedir;

	private final List<FastIgnoreRule> rules;

	public PathFilter(String basedir, String filters) {
		this.basedir = Constants.encode(basedir);

		Stream<FastIgnoreRule> standardRules = Stream.of(
				// the consumer pom generated by the UpdateConsumerPomMojo
				new FastIgnoreRule(".tycho-consumer-pom.xml"),
				// polyglot files generated during pomless builds
				new FastIgnoreRule(".polyglot.*"),
				// connector poms generated during pomless builds
				new FastIgnoreRule("pom.tycho"));

		Stream<FastIgnoreRule> filterRules = Stream.empty();
		if (filters != null) {
			filterRules = FILTER_DELIMITER.splitAsStream(filters) //
					.filter(f -> !f.isBlank()).map(String::strip).map(FastIgnoreRule::new);
		}
		this.rules = Stream.concat(standardRules, filterRules).toList();

	}

	@Override
	public boolean include(TreeWalk tw) throws IOException {
		if (tw.isPathPrefix(basedir, basedir.length) != 0) {
			// not under basedir, not interesting
			return false;
		}

		if (!tw.isSubtree()) {
			String path = tw.getPathString();
			for (FastIgnoreRule rule : rules) {
				if (rule.isMatch(path, tw.isSubtree())) {
					return !rule.getResult();
				}
			}
		}

		return true;
	}

	@Override
	public boolean shouldBeRecursive() {
		return true;
	}

	@Override
	public TreeFilter clone() {
		return this;
	}

}
