/**
 * Copyright (c) 2013-2017 Lorenzo Bettini.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *   Lorenzo Bettini - Initial contribution and API
 */
package org.eclipse.xsemantics.runtime;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import java.util.LinkedList;
import java.util.List;
import org.eclipse.xtext.nodemodel.ICompositeNode;
import org.eclipse.xtext.nodemodel.util.NodeModelUtils;
import org.eclipse.xtext.util.Strings;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Several utility methods that act on rule traces, rule failures, etc.
 * 
 * @author Lorenzo Bettini - Initial contribution and API
 * @since 1.6
 */
@SuppressWarnings("all")
public class TraceUtils {
  public LinkedList<RuleFailedException> failureAsList(final RuleFailedException e) {
    LinkedList<RuleFailedException> _xblockexpression = null;
    {
      final LinkedList<RuleFailedException> list = CollectionLiterals.<RuleFailedException>newLinkedList(e);
      RuleFailedException ex = e.previous;
      while ((ex != null)) {
        {
          String _message = ex.getMessage();
          boolean _tripleNotEquals = (_message != null);
          if (_tripleNotEquals) {
            list.add(ex);
          }
          ex = ex.previous;
        }
      }
      _xblockexpression = list;
    }
    return _xblockexpression;
  }

  public List<String> failureTraceAsStrings(final RuleFailedException e) {
    List<String> _xblockexpression = null;
    {
      final StringBuffer indent = new StringBuffer("");
      final Function1<RuleFailedException, String> _function = (RuleFailedException it) -> {
        String _xblockexpression_1 = null;
        {
          String _string = indent.toString();
          String _removeIndentation = this.removeIndentation(it.getMessage());
          final String listElem = (_string + _removeIndentation);
          indent.append(" ");
          _xblockexpression_1 = listElem;
        }
        return _xblockexpression_1;
      };
      _xblockexpression = ListExtensions.<RuleFailedException, String>map(this.failureAsList(e), _function);
    }
    return _xblockexpression;
  }

  public String failureTraceAsString(final RuleFailedException e) {
    return IterableExtensions.join(this.failureTraceAsStrings(e), "\n");
  }

  public LinkedList<String> traceAsStrings(final RuleApplicationTrace ruleTrace) {
    LinkedList<String> _xblockexpression = null;
    {
      final LinkedList<String> result = new LinkedList<String>();
      for (final Object e : ruleTrace.trace) {
        this.buildTrace(result, e, 0);
      }
      _xblockexpression = result;
    }
    return _xblockexpression;
  }

  public void buildTrace(final List<String> trace, final Object element, final int inc) {
    if ((element instanceof RuleApplicationTrace)) {
      for (final Object e : ((RuleApplicationTrace)element).trace) {
        this.buildTrace(trace, e, (inc + 1));
      }
    } else {
      String _increment = this.increment(inc);
      String _removeIndentation = this.removeIndentation(element.toString());
      String _plus = (_increment + _removeIndentation);
      trace.add(_plus);
    }
  }

  public String traceAsString(final RuleApplicationTrace ruleTrace) {
    return IterableExtensions.join(this.traceAsStrings(ruleTrace), "\n");
  }

  public String increment(final int inc) {
    String _xblockexpression = null;
    {
      StringBuffer buffer = new StringBuffer();
      int i = 0;
      while ((i < inc)) {
        {
          buffer.append(" ");
          i = (i + 1);
        }
      }
      _xblockexpression = buffer.toString();
    }
    return _xblockexpression;
  }

  public LinkedList<ErrorInformation> allErrorInformation(final RuleFailedException e) {
    LinkedList<ErrorInformation> _xblockexpression = null;
    {
      final LinkedList<ErrorInformation> list = Lists.<ErrorInformation>newLinkedList(e.errorInformations);
      RuleFailedException ex = e.previous;
      while ((ex != null)) {
        {
          list.addAll(ex.errorInformations);
          ex = ex.previous;
        }
      }
      _xblockexpression = list;
    }
    return _xblockexpression;
  }

  public List<ErrorInformation> removeDuplicateErrorInformation(final Iterable<ErrorInformation> errorInformations) {
    List<ErrorInformation> _xblockexpression = null;
    {
      final List<ErrorInformation> noDuplicates = new LinkedList<ErrorInformation>();
      for (final ErrorInformation errorInformation : errorInformations) {
        final Function1<ErrorInformation, Boolean> _function = (ErrorInformation it) -> {
          return Boolean.valueOf(((Objects.equal(it.getSource(), errorInformation.getSource()) && 
            Objects.equal(it.getFeature(), errorInformation.getFeature())) && 
            Objects.equal(it.getData(), errorInformation.getData())));
        };
        boolean _exists = IterableExtensions.<ErrorInformation>exists(noDuplicates, _function);
        boolean _not = (!_exists);
        if (_not) {
          noDuplicates.add(errorInformation);
        }
      }
      _xblockexpression = noDuplicates;
    }
    return _xblockexpression;
  }

  public LinkedList<ErrorInformation> removeNonNodeModelSources(final Iterable<ErrorInformation> errorInformations) {
    final Function1<ErrorInformation, Boolean> _function = (ErrorInformation it) -> {
      ICompositeNode _node = NodeModelUtils.getNode(it.getSource());
      return Boolean.valueOf((_node != null));
    };
    return Lists.<ErrorInformation>newLinkedList(IterableExtensions.<ErrorInformation>filter(errorInformations, _function));
  }

  public LinkedList<ErrorInformation> filteredErrorInformation(final RuleFailedException e) {
    return this.removeNonNodeModelSources(this.removeDuplicateErrorInformation(this.allErrorInformation(e)));
  }

  public RuleFailedException innermostRuleFailedExceptionWithNodeModelSources(final RuleFailedException e) {
    final Function1<RuleFailedException, Boolean> _function = (RuleFailedException it) -> {
      boolean _isEmpty = this.filteredErrorInformation(it).isEmpty();
      return Boolean.valueOf((!_isEmpty));
    };
    return IterableExtensions.<RuleFailedException>findLast(this.failureAsList(e), _function);
  }

  public String removeIndentation(final String s) {
    return Strings.removeLeadingWhitespace(s);
  }

  /**
   * Returns the last element in the trace that is not a RuleApplicationTrace
   */
  public Object lastElementNotTrace(final RuleApplicationTrace trace) {
    final Function1<Object, Boolean> _function = (Object it) -> {
      return Boolean.valueOf((!(it instanceof RuleApplicationTrace)));
    };
    return IterableExtensions.<Object>findLast(trace.trace, _function);
  }
}
