/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.client;

import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.entity.ByteArrayEntity;
import org.apache.http.entity.ContentType;
import org.elasticsearch.client.watcher.AckWatchRequest;
import org.elasticsearch.client.watcher.ActivateWatchRequest;
import org.elasticsearch.client.watcher.DeactivateWatchRequest;
import org.elasticsearch.client.watcher.DeleteWatchRequest;
import org.elasticsearch.client.watcher.ExecuteWatchRequest;
import org.elasticsearch.client.watcher.GetWatchRequest;
import org.elasticsearch.client.watcher.PutWatchRequest;
import org.elasticsearch.client.watcher.StartWatchServiceRequest;
import org.elasticsearch.client.watcher.StopWatchServiceRequest;
import org.elasticsearch.client.watcher.WatcherStatsRequest;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.xcontent.XContentType;

import java.io.IOException;

final class WatcherRequestConverters {

    private WatcherRequestConverters() {}

    static Request startWatchService(StartWatchServiceRequest startWatchServiceRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
                .addPathPartAsIs("_xpack")
                .addPathPartAsIs("watcher")
                .addPathPartAsIs("_start")
                .build();

        return new Request(HttpPost.METHOD_NAME, endpoint);
    }

    static Request stopWatchService(StopWatchServiceRequest stopWatchServiceRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
                .addPathPartAsIs("_xpack")
                .addPathPartAsIs("watcher")
                .addPathPartAsIs("_stop")
                .build();

        return new Request(HttpPost.METHOD_NAME, endpoint);
    }

    static Request putWatch(PutWatchRequest putWatchRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack", "watcher", "watch")
            .addPathPart(putWatchRequest.getId())
            .build();

        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params(request).withVersion(putWatchRequest.getVersion());
        if (putWatchRequest.isActive() == false) {
            params.putParam("active", "false");
        }
        ContentType contentType = RequestConverters.createContentType(putWatchRequest.xContentType());
        BytesReference source = putWatchRequest.getSource();
        request.setEntity(new ByteArrayEntity(source.toBytesRef().bytes, 0, source.length(), contentType));
        return request;
    }


    static Request getWatch(GetWatchRequest getWatchRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack", "watcher", "watch")
            .addPathPart(getWatchRequest.getId())
            .build();

        return new Request(HttpGet.METHOD_NAME, endpoint);
    }

    static Request deactivateWatch(DeactivateWatchRequest deactivateWatchRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack")
            .addPathPartAsIs("watcher")
            .addPathPartAsIs("watch")
            .addPathPart(deactivateWatchRequest.getWatchId())
            .addPathPartAsIs("_deactivate")
            .build();
        return new Request(HttpPut.METHOD_NAME, endpoint);
    }

    static Request deleteWatch(DeleteWatchRequest deleteWatchRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack", "watcher", "watch")
            .addPathPart(deleteWatchRequest.getId())
            .build();

        Request request = new Request(HttpDelete.METHOD_NAME, endpoint);
        return request;
    }

    static Request executeWatch(ExecuteWatchRequest executeWatchRequest) throws IOException {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack", "watcher", "watch")
            .addPathPart(executeWatchRequest.getId())       // will ignore if ID is null
            .addPathPartAsIs("_execute").build();

        Request request = new Request(HttpPost.METHOD_NAME, endpoint);
        RequestConverters.Params params = new RequestConverters.Params(request);
        if (executeWatchRequest.isDebug()) {
            params.putParam("debug", "true");
        }
        if (executeWatchRequest.ignoreCondition()) {
            params.putParam("ignore_condition", "true");
        }
        if (executeWatchRequest.recordExecution()) {
            params.putParam("record_execution", "true");
        }

        request.setEntity(RequestConverters.createEntity(executeWatchRequest, XContentType.JSON));
        return request;
    }

    public static Request ackWatch(AckWatchRequest ackWatchRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack", "watcher", "watch")
            .addPathPart(ackWatchRequest.getWatchId())
            .addPathPartAsIs("_ack")
            .addCommaSeparatedPathParts(ackWatchRequest.getActionIds())
            .build();
        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        return request;
    }

    static Request activateWatch(ActivateWatchRequest activateWatchRequest) {
        String endpoint = new RequestConverters.EndpointBuilder()
            .addPathPartAsIs("_xpack", "watcher", "watch")
            .addPathPart(activateWatchRequest.getWatchId())
            .addPathPartAsIs("_activate")
            .build();
        Request request = new Request(HttpPut.METHOD_NAME, endpoint);
        return request;
    }

    static Request watcherStats(WatcherStatsRequest watcherStatsRequest) {
        RequestConverters.EndpointBuilder builder = new RequestConverters.EndpointBuilder().addPathPartAsIs("_xpack", "watcher", "stats");
        String endpoint = builder.build();
        Request request = new Request(HttpGet.METHOD_NAME, endpoint);
        RequestConverters.Params parameters = new RequestConverters.Params(request);
        StringBuilder metric = new StringBuilder();
        if (watcherStatsRequest.includeCurrentWatches()) {
            metric.append("current_watches");
        }
        if (watcherStatsRequest.includeQueuedWatches()) {
            if (metric.length() > 0) {
                metric.append(",");
            }
            metric.append("queued_watches");
        }
        if (metric.length() > 0) {
            parameters.putParam("metric", metric.toString());
        }
        return request;
    }
}
