/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.geometry.simplify;

import java.util.Locale;
import org.elasticsearch.geometry.simplify.SloppyMath;

public interface SimplificationErrorCalculator {
    public static final SimplificationErrorCalculator CARTESIAN_TRIANGLE_AREA = new CartesianTriangleAreaCalculator();
    public static final SimplificationErrorCalculator TRIANGLE_AREA = new TriangleAreaCalculator();
    public static final SimplificationErrorCalculator TRIANGLE_HEIGHT = new TriangleHeightCalculator();
    public static final SimplificationErrorCalculator HEIGHT_AND_BACKPATH_DISTANCE = new CartesianHeightAndBackpathDistanceCalculator();
    public static final SimplificationErrorCalculator SPHERICAL_HEIGHT_AND_BACKPATH_DISTANCE = new SphericalHeightAndBackpathDistanceCalculator();

    public double calculateError(PointLike var1, PointLike var2, PointLike var3);

    public static SimplificationErrorCalculator byName(String calculatorName) {
        return switch (calculatorName.toLowerCase(Locale.ROOT)) {
            case "cartesiantrianglearea" -> CARTESIAN_TRIANGLE_AREA;
            case "trianglearea" -> TRIANGLE_AREA;
            case "triangleheight" -> TRIANGLE_HEIGHT;
            case "heightandbackpathdistance" -> HEIGHT_AND_BACKPATH_DISTANCE;
            case "sphericalheightandbackpathdistance" -> SPHERICAL_HEIGHT_AND_BACKPATH_DISTANCE;
            default -> throw new IllegalArgumentException("Unknown simplification error calculator: " + calculatorName);
        };
    }

    public static class CartesianTriangleAreaCalculator
    implements SimplificationErrorCalculator {
        @Override
        public double calculateError(PointLike left, PointLike middle, PointLike right) {
            double xb = middle.x() - left.x();
            double yb = middle.y() - left.y();
            double xc = right.x() - left.x();
            double yc = right.y() - left.y();
            return 0.5 * Math.abs(xb * yc - xc * yb);
        }
    }

    public static class TriangleAreaCalculator
    implements SimplificationErrorCalculator {
        @Override
        public double calculateError(PointLike left, PointLike middle, PointLike right) {
            double a = this.distance(left, right);
            double b = this.distance(right, middle);
            double c = this.distance(middle, left);
            double s = 0.5 * (a + b + c);
            double da = s - a;
            double db = s - b;
            double dc = s - c;
            if (da >= 0.0 && db >= 0.0 && dc >= 0.0) {
                return Math.sqrt(s * da * db * dc);
            }
            return 0.0;
        }

        private double distance(PointLike a, PointLike b) {
            return SloppyMath.haversinMeters(a.y(), a.x(), b.y(), b.x());
        }
    }

    public static class TriangleHeightCalculator
    implements SimplificationErrorCalculator {
        @Override
        public double calculateError(PointLike left, PointLike middle, PointLike right) {
            double a = this.distance(left, right);
            double b = this.distance(right, middle);
            double c = this.distance(middle, left);
            double s = 0.5 * (a + b + c);
            double da = s - a;
            double db = s - b;
            double dc = s - c;
            if (da >= 0.0 && db >= 0.0 && dc >= 0.0) {
                return 2.0 * Math.sqrt(s * da * db * dc) / a;
            }
            return 0.0;
        }

        private double distance(PointLike a, PointLike b) {
            return SloppyMath.haversinMeters(a.y(), a.x(), b.y(), b.x());
        }
    }

    public static class CartesianHeightAndBackpathDistanceCalculator
    implements SimplificationErrorCalculator {
        @Override
        public double calculateError(PointLike left, PointLike middle, PointLike right) {
            double rightX = right.x() - left.x();
            double rightY = right.y() - left.y();
            if (Math.abs(rightX) > 1.0E-10 || Math.abs(rightY) > 1.0E-10) {
                double len = Math.sqrt(rightX * rightX + rightY * rightY);
                double cos = rightX / len;
                double sin = rightY / len;
                double middleX = middle.x() - left.x();
                double middleY = middle.y() - left.y();
                double middleXrotated = middleX * cos + middleY * sin;
                double middleYrotated = middleY * cos - middleX * sin;
                double rightXrotated = rightX * cos + rightY * sin;
                double rightYrotated = rightY * cos - rightX * sin;
                assert (Math.abs(rightYrotated) < 1.0E-10);
                assert (Math.abs(rightXrotated - len) < len / 1.0E10);
                double height = Math.abs(middleYrotated);
                double deltaL = -Math.min(0.0, middleXrotated);
                double deltaR = Math.max(0.0, middleXrotated - rightXrotated);
                double backDistance = Math.max(deltaR, deltaL);
                return Math.max(height, backDistance);
            }
            return 0.0;
        }
    }

    public static class SphericalHeightAndBackpathDistanceCalculator
    implements SimplificationErrorCalculator {
        @Override
        public double calculateError(PointLike leftLL, PointLike middleLL, PointLike rightLL) {
            Point3D left = Point3D.from(leftLL);
            Point3D middle = Point3D.from(middleLL);
            Point3D right = Point3D.from(rightLL);
            Point3D rotationAxis = right.cross(left);
            Point3D xAxis = new Point3D(1.0, 0.0, 0.0);
            double rotationAngle = xAxis.angleTo(left);
            left = left.rotate(rotationAxis, rotationAngle);
            middle = middle.rotate(rotationAxis, rotationAngle);
            double len = left.angleTo(right = right.rotate(rotationAxis, rotationAngle));
            if (len > 1.0E-10) {
                double height = Math.abs(middle.z);
                double deltaL = -Math.min(0.0, middle.y);
                double deltaR = Math.max(0.0, middle.y - right.y);
                double backDistance = Math.max(deltaR, deltaL);
                return Math.max(height, backDistance);
            }
            return 0.0;
        }
    }

    public record RotationMatrix(double[][] matrix) {
        public RotationMatrix {
            assert (matrix.length == 3);
            for (double[] doubles : matrix) {
                assert (doubles.length == 3);
            }
        }

        public RotationMatrix(Point3D axis, double angleInRadian) {
            this(RotationMatrix.matrixFrom(axis, angleInRadian));
        }

        public Point3D multiply(Point3D incoming) {
            double x = RotationMatrix.dot(incoming, this.matrix[0]);
            double y = RotationMatrix.dot(incoming, this.matrix[1]);
            double z = RotationMatrix.dot(incoming, this.matrix[2]);
            return new Point3D(x, y, z);
        }

        private static double dot(Point3D incoming, double[] row) {
            return incoming.x * row[0] + incoming.y * row[1] + incoming.z * row[2];
        }

        private static double[][] matrixFrom(Point3D axis, double angle) {
            double[][] matrix = new double[3][];
            double x = axis.x;
            double y = axis.y;
            double z = axis.z;
            matrix[0] = new double[]{RotationMatrix.diag(angle, x), RotationMatrix.cell(angle, x, y, -z), RotationMatrix.cell(angle, x, z, y)};
            matrix[1] = new double[]{RotationMatrix.cell(angle, y, x, z), RotationMatrix.diag(angle, y), RotationMatrix.cell(angle, y, z, -x)};
            matrix[2] = new double[]{RotationMatrix.cell(angle, z, x, -y), RotationMatrix.cell(angle, z, y, x), RotationMatrix.diag(angle, z)};
            return matrix;
        }

        private static double diag(double angle, double coord) {
            double cosTheta = SloppyMath.cos(angle);
            return cosTheta + coord * coord * (1.0 - cosTheta);
        }

        private static double cell(double angle, double a, double b, double c) {
            return a * b * (1.0 - SloppyMath.cos(angle)) + c * Point3D.sin(angle);
        }
    }

    public record Point3D(double x, double y, double z) {
        public static Point3D from(double latDegrees, double lonDegrees) {
            double lat = Math.toRadians(latDegrees);
            double lon = Math.toRadians(lonDegrees);
            double x = SloppyMath.cos(lat) * SloppyMath.cos(lon);
            double y = SloppyMath.cos(lat) * Point3D.sin(lon);
            double z = Point3D.sin(lat);
            return new Point3D(x, y, z);
        }

        public static Point3D from(PointLike point) {
            return Point3D.from(point.y(), point.x());
        }

        public Point3D cross(Point3D o) {
            return new Point3D(this.y * o.z - this.z * o.y, this.z * o.x - this.x * o.z, this.x * o.y - this.y * o.x);
        }

        static double sin(double radians) {
            return SloppyMath.cos(radians - 1.5707963267948966);
        }

        public double length() {
            return Math.sqrt(this.x * this.x + this.y * this.y + this.z * this.z);
        }

        public Point3D inverse() {
            return new Point3D(-this.x, -this.y, -this.z);
        }

        public double angleTo(Point3D other) {
            double a = this.dot(other) / (this.length() * other.length());
            return Math.acos(Math.min(1.0, a));
        }

        public double dot(Point3D o) {
            return this.x * o.x + this.y * o.y + this.z * o.z;
        }

        public Point3D rotate(Point3D rotationAxis, double rotationAngle) {
            RotationMatrix rotation = new RotationMatrix(rotationAxis, rotationAngle);
            return rotation.multiply(this);
        }
    }

    public static interface PointLike {
        public double x();

        public double y();
    }
}

